<?php
/** cms-core/backend/analytics/AnalyticsTracker.php */

/**
 * DSGVO-Compliant Analytics Tracker
 * No cookies, IP anonymization, minimal data retention
 */

class AnalyticsTracker {
    
    /**
     * Track pageview (call this on every page load)
     */
    public static function trackPageview() {
        // Only track if internal analytics is enabled
        if (!get_setting('internal_analytics_enabled', true)) {
            return;
        }
        
        // Don't track admin pages
        if (strpos($_SERVER['REQUEST_URI'], '/admin/') !== false) {
            return;
        }
        
        $pageUrl = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        $pageTitle = $_SERVER['REQUEST_URI']; // Will be set by page if available
        
        // Generate DSGVO-compliant hashes (no personal data)
        $visitorHash = self::generateVisitorHash();
        $sessionHash = self::generateSessionHash();
        
        // Detect device type
        $deviceType = self::detectDeviceType();
        
        // Get location (DSGVO-compliant - only country/city, no precise location)
        $location = self::getLocation();
        
        // Get referrer (anonymized)
        $referrer = isset($_SERVER['HTTP_REFERER']) ? parse_url($_SERVER['HTTP_REFERER'], PHP_URL_HOST) : null;
        
        // Check if visitor is new or returning
        $isNewVisitor = self::isNewVisitor($visitorHash);
        $isReturning = !$isNewVisitor;
        
        // Insert pageview
        db()->insert(
            "INSERT INTO analytics_pageviews 
            (page_url, page_title, visitor_hash, session_hash, is_new_visitor, is_returning, device_type, country, country_name, city, referrer, viewed_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())",
            [$pageUrl, $pageTitle, $visitorHash, $sessionHash, $isNewVisitor, $isReturning, $deviceType, $location['country'], $location['country_name'], $location['city'], $referrer]
        );
        
        // Update or create session
        self::updateSession($sessionHash, $visitorHash, $pageUrl);
    }
    
    /**
     * Generate anonymized visitor hash (DSGVO-compliant)
     * Uses IP + User-Agent but hashed with daily salt
     */
    private static function generateVisitorHash() {
        $ip = self::anonymizeIP($_SERVER['REMOTE_ADDR']);
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $dailySalt = date('Y-m-d'); // Changes daily for privacy
        
        return hash('sha256', $ip . $userAgent . $dailySalt);
    }
    
    /**
     * Generate session hash (valid for 30 minutes)
     */
    private static function generateSessionHash() {
        $visitorHash = self::generateVisitorHash();
        $timeSlot = floor(time() / 1800); // 30-minute slots
        
        return hash('sha256', $visitorHash . $timeSlot);
    }
    
    /**
     * Anonymize IP address (DSGVO requirement)
     * Removes last octet for IPv4, last 80 bits for IPv6
     */
    private static function anonymizeIP($ip) {
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            // IPv4: Remove last octet
            return preg_replace('/\.\d+$/', '.0', $ip);
        } elseif (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            // IPv6: Keep only first 48 bits
            $parts = explode(':', $ip);
            return implode(':', array_slice($parts, 0, 3)) . '::';
        }
        
        return '0.0.0.0'; // Fallback
    }
    
    /**
     * Detect device type from user agent
     */
    private static function detectDeviceType() {
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        
        if (preg_match('/mobile|android|iphone|ipod|blackberry|iemobile|opera mini/i', $userAgent)) {
            return 'mobile';
        } elseif (preg_match('/tablet|ipad/i', $userAgent)) {
            return 'tablet';
        }
        
        return 'desktop';
    }
    
    /**
     * Get visitor location (country & city only, DSGVO-compliant)
     * Uses ip-api.com free service (no key required)
     */
    private static function getLocation() {
        $location = [
            'country' => null,
            'country_name' => null,
            'city' => null
        ];
        
        // Get real IP (handle proxies/CloudFlare)
        $ip = self::getRealIP();
        
        // Don't track localhost/private IPs
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === false) {
            return $location;
        }
        
        // Try CloudFlare headers first (if behind CloudFlare)
        if (isset($_SERVER['HTTP_CF_IPCOUNTRY']) && $_SERVER['HTTP_CF_IPCOUNTRY'] !== 'XX') {
            $location['country'] = $_SERVER['HTTP_CF_IPCOUNTRY'];
            $location['country_name'] = self::getCountryName($_SERVER['HTTP_CF_IPCOUNTRY']);
        }
        
        // If no CloudFlare, use ip-api.com (free, 45 req/min limit)
        if (!$location['country']) {
            try {
                $apiUrl = "http://ip-api.com/json/{$ip}?fields=status,country,countryCode,city";
                $context = stream_context_create([
                    'http' => [
                        'timeout' => 2,
                        'ignore_errors' => true
                    ]
                ]);
                
                $response = @file_get_contents($apiUrl, false, $context);
                
                if ($response) {
                    $data = json_decode($response, true);
                    
                    if ($data && $data['status'] === 'success') {
                        $location['country'] = $data['countryCode'];
                        $location['country_name'] = $data['country'];
                        $location['city'] = $data['city'];
                    }
                }
            } catch (Exception $e) {
                // Silently fail, location is optional
                error_log('Analytics location API error: ' . $e->getMessage());
            }
        }
        
        return $location;
    }
    
    /**
     * Get real IP address (handles proxies/CloudFlare)
     */
    private static function getRealIP() {
        // Check for CloudFlare
        if (isset($_SERVER['HTTP_CF_CONNECTING_IP'])) {
            return $_SERVER['HTTP_CF_CONNECTING_IP'];
        }
        
        // Check for proxies
        $headers = [
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_REAL_IP',
            'HTTP_CLIENT_IP',
            'REMOTE_ADDR'
        ];
        
        foreach ($headers as $header) {
            if (isset($_SERVER[$header])) {
                $ip = trim(explode(',', $_SERVER[$header])[0]);
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
    
    /**
     * Get country name from ISO code
     */
    private static function getCountryName($code) {
        $countries = [
            'DE' => 'Deutschland',
            'AT' => 'Österreich',
            'CH' => 'Schweiz',
            'US' => 'United States',
            'GB' => 'United Kingdom',
            'FR' => 'France',
            'ES' => 'Spain',
            'IT' => 'Italy',
            'NL' => 'Netherlands',
            'BE' => 'Belgium',
            'PL' => 'Poland',
            'RU' => 'Russia',
            'CN' => 'China',
            'JP' => 'Japan',
            'BR' => 'Brazil',
            'IN' => 'India',
            'CA' => 'Canada',
            'AU' => 'Australia',
            'MX' => 'Mexico',
            'SE' => 'Sweden',
            'NO' => 'Norway',
            'DK' => 'Denmark',
            'FI' => 'Finland',
        ];
        
        return $countries[$code] ?? $code;
    }
    
    /**
     * Check if visitor is new (first time in last 30 days)
     */
    private static function isNewVisitor($visitorHash) {
        $exists = db()->fetchOne(
            "SELECT id FROM analytics_pageviews 
             WHERE visitor_hash = ? 
             AND viewed_at > DATE_SUB(NOW(), INTERVAL 30 DAY) 
             LIMIT 1",
            [$visitorHash]
        );
        
        return empty($exists);
    }
    
    /**
     * Update or create session
     */
    private static function updateSession($sessionHash, $visitorHash, $pageUrl) {
        $session = db()->fetchOne(
            "SELECT * FROM analytics_sessions WHERE session_hash = ?",
            [$sessionHash]
        );
        
        if ($session) {
            // Update existing session
            $pageviews = $session['pageviews'] + 1;
            $duration = strtotime('now') - strtotime($session['started_at']);
            $bounced = ($pageviews > 1) ? 0 : 1;
            
            db()->execute(
                "UPDATE analytics_sessions 
                 SET last_activity = NOW(), 
                     pageviews = ?, 
                     duration = ?,
                     bounced = ?,
                     exit_page = ?
                 WHERE session_hash = ?",
                [$pageviews, $duration, $bounced, $pageUrl, $sessionHash]
            );
        } else {
            // Create new session
            db()->insert(
                "INSERT INTO analytics_sessions 
                (session_hash, visitor_hash, started_at, last_activity, pageviews, entry_page, exit_page) 
                VALUES (?, ?, NOW(), NOW(), 1, ?, ?)",
                [$sessionHash, $visitorHash, $pageUrl, $pageUrl]
            );
        }
    }
    
    /**
     * Get analytics stats for dashboard
     */
    public static function getStats($period = '24h') {
        // Determine date range
        switch ($period) {
            case '24h':
                $dateCondition = "viewed_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)";
                break;
            case '7d':
                $dateCondition = "viewed_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)";
                break;
            case '30d':
                $dateCondition = "viewed_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)";
                break;
            default:
                $dateCondition = "viewed_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)";
        }
        
        // Total visitors
        $totalVisitors = db()->fetchOne(
            "SELECT COUNT(DISTINCT visitor_hash) as count 
             FROM analytics_pageviews 
             WHERE $dateCondition"
        )['count'] ?? 0;
        
        // Total sessions
        $totalSessions = db()->fetchOne(
            "SELECT COUNT(DISTINCT session_hash) as count 
             FROM analytics_pageviews 
             WHERE $dateCondition"
        )['count'] ?? 0;
        
        // New vs Returning visitors
        $newVisitors = db()->fetchOne(
            "SELECT COUNT(DISTINCT visitor_hash) as count 
             FROM analytics_pageviews 
             WHERE $dateCondition AND is_new_visitor = 1"
        )['count'] ?? 0;
        
        $returningVisitors = $totalVisitors - $newVisitors;
        
        // Sessions per visitor
        $sessionsPerVisitor = $totalVisitors > 0 ? round($totalSessions / $totalVisitors, 2) : 0;
        
        // Average session duration
        $avgDuration = db()->fetchOne(
            "SELECT AVG(duration) as avg 
             FROM analytics_sessions 
             WHERE started_at >= DATE_SUB(NOW(), INTERVAL " . 
             (strpos($period, 'h') !== false ? '24 HOUR' : 
             (strpos($period, 'd') !== false ? str_replace('d', ' DAY', $period) : '24 HOUR')) . ")"
        )['avg'] ?? 0;
        
        // Bounce rate (% of single-page sessions)
        $bouncedSessions = db()->fetchOne(
            "SELECT COUNT(*) as count 
             FROM analytics_sessions 
             WHERE bounced = 1 
             AND started_at >= DATE_SUB(NOW(), INTERVAL " . 
             (strpos($period, 'h') !== false ? '24 HOUR' : 
             (strpos($period, 'd') !== false ? str_replace('d', ' DAY', $period) : '24 HOUR')) . ")"
        )['count'] ?? 0;
        
        $bounceRate = $totalSessions > 0 ? round(($bouncedSessions / $totalSessions) * 100, 1) : 0;
        
        // Top 5 pages with unique visitors
        $topPages = db()->fetchAll(
            "SELECT page_url, 
                    COUNT(*) as views,
                    COUNT(DISTINCT visitor_hash) as visitors
             FROM analytics_pageviews 
             WHERE $dateCondition 
             GROUP BY page_url 
             ORDER BY views DESC 
             LIMIT 5"
        );
        
        // User flow (entry -> exit)
        $userFlow = db()->fetchAll(
            "SELECT entry_page, exit_page, COUNT(*) as sessions 
             FROM analytics_sessions 
             WHERE started_at >= DATE_SUB(NOW(), INTERVAL " . 
             (strpos($period, 'h') !== false ? '24 HOUR' : 
             (strpos($period, 'd') !== false ? str_replace('d', ' DAY', $period) : '24 HOUR')) . ") 
             GROUP BY entry_page, exit_page 
             ORDER BY sessions DESC 
             LIMIT 5"
        );
        
        // Top countries with cities (show country name only, no code)
        $topCountries = db()->fetchAll(
            "SELECT country_name, city, COUNT(DISTINCT visitor_hash) as visitors 
             FROM analytics_pageviews 
             WHERE $dateCondition AND country IS NOT NULL 
             GROUP BY country_name, city 
             ORDER BY visitors DESC 
             LIMIT 10"
        );
        
        return [
            'total_visitors' => $totalVisitors,
            'total_sessions' => $totalSessions,
            'new_visitors' => $newVisitors,
            'returning_visitors' => $returningVisitors,
            'sessions_per_visitor' => $sessionsPerVisitor,
            'avg_session_duration' => round($avgDuration),
            'bounce_rate' => $bounceRate,
            'top_pages' => $topPages,
            'user_flow' => $userFlow,
            'top_countries' => $topCountries
        ];
    }
    
    /**
     * Clean old data (DSGVO: max 90 days retention)
     */
    public static function cleanOldData() {
        db()->execute("DELETE FROM analytics_pageviews WHERE viewed_at < DATE_SUB(NOW(), INTERVAL 90 DAY)");
        db()->execute("DELETE FROM analytics_sessions WHERE started_at < DATE_SUB(NOW(), INTERVAL 90 DAY)");
    }
}