<?php
/** cms-core/backend/auth/UserRoles.php - Multi-User System */

// Load dependencies
if (!class_exists('Validator')) {
    require_once dirname(__DIR__) . '/security/validation.php';
}

/**
 * User Role Management
 * Simple but powerful role-based access control
 */

class UserRoles {
    
    /**
     * Available roles with permissions
     */
    const ROLES = [
        'super_admin' => [
            'name' => 'Super Admin',
            'description' => 'Full system access',
            'icon' => '👑',
            'permissions' => ['*'] // All permissions
        ],
        'admin' => [
            'name' => 'Administrator',
            'description' => 'Manage content and settings',
            'icon' => '🔑',
            'permissions' => [
                'pages.view', 'pages.create', 'pages.edit', 'pages.delete',
                'blog.view', 'blog.create', 'blog.edit', 'blog.delete',
                'media.view', 'media.upload', 'media.delete',
                'menu.edit',
                'settings.view', 'settings.edit',
                'users.view', 'users.manage', // Can manage users
                'analytics.view'
            ]
        ],
        'editor' => [
            'name' => 'Editor',
            'description' => 'Create and edit content',
            'icon' => '✍️',
            'permissions' => [
                'pages.view', 'pages.create', 'pages.edit', // No delete
                'blog.view', 'blog.create', 'blog.edit', 'blog.delete',
                'media.view', 'media.upload',
                'analytics.view'
            ]
        ],
        'author' => [
            'name' => 'Author',
            'description' => 'Create blog posts',
            'icon' => '📝',
            'permissions' => [
                'pages.view', // Read-only for pages
                'blog.view', 'blog.create', 'blog.edit_own', // Only own posts
                'media.view', 'media.upload'
            ]
        ],
        'analytics' => [
            'name' => 'Analytics',
            'description' => 'View analytics and reports',
            'icon' => '📊',
            'permissions' => [
                'pages.view',
                'blog.view',
                'analytics.view',
                'analytics.export'
            ]
        ]
    ];
    
    /**
     * Check if user has permission
     */
    public static function can($permission, $userId = null) {
        if (!$userId) {
            $userId = $_SESSION['user_id'] ?? null;
        }
        
        if (!$userId) {
            return false;
        }
        
        $user = self::getUser($userId);
        if (!$user) {
            return false;
        }
        
        $role = $user['role'];
        
        // Super admin can do everything
        if ($role === 'super_admin') {
            return true;
        }
        
        // Check if role has permission
        $rolePerms = self::ROLES[$role]['permissions'] ?? [];
        
        // Wildcard check
        if (in_array('*', $rolePerms)) {
            return true;
        }
        
        // Exact permission
        if (in_array($permission, $rolePerms)) {
            return true;
        }
        
        // Check wildcard patterns (e.g., pages.* includes pages.view, pages.edit)
        $permParts = explode('.', $permission);
        $wildcard = $permParts[0] . '.*';
        if (in_array($wildcard, $rolePerms)) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Check if user owns resource (for _own permissions)
     */
    public static function owns($resourceType, $resourceId, $userId = null) {
        if (!$userId) {
            $userId = $_SESSION['user_id'] ?? null;
        }
        
        if (!$userId) {
            return false;
        }
        
        $db = Database::getInstance();
        
        switch ($resourceType) {
            case 'blog_post':
                $post = $db->fetchOne("SELECT author_id FROM blog_posts WHERE id = ?", [$resourceId]);
                return $post && $post['author_id'] == $userId;
                
            case 'page':
                $page = $db->fetchOne("SELECT created_by FROM pages WHERE id = ?", [$resourceId]);
                return $page && $page['created_by'] == $userId;
                
            default:
                return false;
        }
    }
    
    /**
     * Require permission or die
     */
    public static function require($permission, $message = null) {
        if (!self::can($permission)) {
            http_response_code(403);
            if ($message) {
                die($message);
            }
            include APP_ROOT . '/cms-core/public/webview/403.php';
            exit;
        }
    }
    
    /**
     * Get user by ID
     */
    public static function getUser($userId) {
        $db = Database::getInstance();
        return $db->fetchOne(
            "SELECT id, username, email, role, created_at, last_login FROM users WHERE id = ?",
            [$userId]
        );
    }
    
    /**
     * Get all users
     */
    public static function getAllUsers() {
        $db = Database::getInstance();
        return $db->fetchAll(
            "SELECT id, username, email, role, created_at, last_login FROM users ORDER BY created_at DESC"
        );
    }
    
    /**
     * Create user
     */
    public static function createUser($username, $email, $password, $role = 'editor') {
        // Validation
        $validator = new Validator([
            'username' => $username,
            'email' => $email,
            'password' => $password,
            'role' => $role
        ]);
        
        $validator
            ->required('username')
            ->username('username')
            ->required('email')
            ->email('email')
            ->required('password')
            ->password('password')
            ->in('role', array_keys(self::ROLES));
        
        if (!$validator->isValid()) {
            return ['success' => false, 'errors' => $validator->getErrors()];
        }
        
        // Check if username/email exists
        $db = Database::getInstance();
        
        $existing = $db->fetchOne("SELECT id FROM users WHERE username = ? OR email = ?", [$username, $email]);
        if ($existing) {
            return ['success' => false, 'error' => 'Username oder E-Mail bereits vergeben.'];
        }
        
        // Hash password
        $passwordHash = password_hash($password, PASSWORD_BCRYPT);
        
        // Insert user
        $userId = $db->insert(
            "INSERT INTO users (username, email, password_hash, role, created_at) VALUES (?, ?, ?, ?, NOW())",
            [$username, $email, $passwordHash, $role]
        );
        
        return ['success' => true, 'user_id' => $userId];
    }
    
    /**
     * Update user
     */
    public static function updateUser($userId, $data) {
        $db = Database::getInstance();
        $updates = [];
        $params = [];
        
        if (isset($data['email'])) {
            $validator = new Validator(['email' => $data['email']]);
            $validator->email('email');
            if (!$validator->isValid()) {
                return ['success' => false, 'error' => 'Ungültige E-Mail.'];
            }
            $updates[] = "email = ?";
            $params[] = $data['email'];
        }
        
        if (isset($data['role'])) {
            if (!array_key_exists($data['role'], self::ROLES)) {
                return ['success' => false, 'error' => 'Ungültige Rolle.'];
            }
            $updates[] = "role = ?";
            $params[] = $data['role'];
        }
        
        if (isset($data['password']) && !empty($data['password'])) {
            $validator = new Validator(['password' => $data['password']]);
            $validator->password('password');
            if (!$validator->isValid()) {
                return ['success' => false, 'error' => 'Passwort zu schwach.'];
            }
            $updates[] = "password_hash = ?";
            $params[] = password_hash($data['password'], PASSWORD_BCRYPT);
        }
        
        if (empty($updates)) {
            return ['success' => false, 'error' => 'Keine Änderungen.'];
        }
        
        $params[] = $userId;
        $sql = "UPDATE users SET " . implode(', ', $updates) . " WHERE id = ?";
        
        $db->execute($sql, $params);
        
        return ['success' => true];
    }
    
    /**
     * Delete user
     */
    public static function deleteUser($userId) {
        // Prevent deleting yourself
        if ($userId == ($_SESSION['user_id'] ?? 0)) {
            return ['success' => false, 'error' => 'Sie können sich nicht selbst löschen.'];
        }
        
        // Prevent deleting last super_admin
        $user = self::getUser($userId);
        if ($user['role'] === 'super_admin') {
            $db = Database::getInstance();
            $count = $db->fetchOne("SELECT COUNT(*) as count FROM users WHERE role = 'super_admin'");
            if ($count['count'] <= 1) {
                return ['success' => false, 'error' => 'Der letzte Super Admin kann nicht gelöscht werden.'];
            }
        }
        
        $db = Database::getInstance();
        $db->execute("DELETE FROM users WHERE id = ?", [$userId]);
        
        return ['success' => true];
    }
    
    /**
     * Get role info
     */
    public static function getRoleInfo($role) {
        return self::ROLES[$role] ?? null;
    }
    
    /**
     * Get all roles
     */
    public static function getAllRoles() {
        return self::ROLES;
    }
    
    /**
     * Check if user is admin or higher
     */
    public static function isAdmin($userId = null) {
        if (!$userId) {
            $userId = $_SESSION['user_id'] ?? null;
        }
        
        $user = self::getUser($userId);
        return $user && in_array($user['role'], ['super_admin', 'admin']);
    }
    
    /**
     * Log user activity
     */
    public static function logActivity($userId, $action, $details = null) {
        $db = Database::getInstance();
        $db->insert(
            "INSERT INTO user_activity_log (user_id, action, details, ip_address, created_at) 
             VALUES (?, ?, ?, ?, NOW())",
            [$userId, $action, $details, $_SERVER['REMOTE_ADDR'] ?? '']
        );
    }
}