<?php
/** cms-core/backend/auth/login.php - ULTRA SIMPLE */

/**
 * Login Handler
 * Verarbeitet Benutzer-Login
 */

require_once dirname(__DIR__) . '/core/init.php';

class LoginHandler {
    
    /**
     * Verarbeitet Login-Anfrage
     */
    public static function process($username, $password) {
        try {
            // CSRF-Validierung
            if (!CSRF::validate()) {
                return [
                    'success' => false,
                    'error' => 'Ungültiges Sicherheits-Token. Bitte versuchen Sie es erneut.'
                ];
            }
            
            // Eingabe validieren
            if (empty($username) || empty($password)) {
                return [
                    'success' => false,
                    'error' => 'Benutzername und Passwort sind erforderlich.'
                ];
            }
            
            // Benutzer aus DB holen
            $db = Database::getInstance();
            $user = $db->fetchOne(
                "SELECT id, username, password_hash, role FROM users WHERE username = ?",
                [$username]
            );
            
            // Prüfe ob User existiert und Passwort korrekt ist
            if (!$user || !password_verify($password, $user['password_hash'])) {
                return [
                    'success' => false,
                    'error' => 'Ungültiger Benutzername oder Passwort.'
                ];
            }
            
            // Login erfolgreich
            self::createSession($user);
            
            // Aktualisiere letzten Login
            $db->execute(
                "UPDATE users SET last_login = NOW() WHERE id = ?",
                [$user['id']]
            );
            
            return [
                'success' => true,
                'user' => $user
            ];
            
        } catch (Exception $e) {
            // Log the error but return user-friendly message
            if (class_exists('Logger')) {
                Logger::error('Login error: ' . $e->getMessage(), [
                    'username' => $username,
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ]);
            }
            
            error_log('Login error: ' . $e->getMessage());
            
            return [
                'success' => false,
                'error' => 'Ein Fehler ist aufgetreten. Bitte versuchen Sie es später erneut.'
            ];
        }
    }
    
    /**
     * Erstellt Session für eingeloggten User
     */
    private static function createSession($user) {
        // Session regenerieren für Sicherheit
        if (class_exists('SessionManager')) {
            SessionManager::regenerate();
        } else {
            session_regenerate_id(true);
        }
        
        // User-Daten in Session speichern
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['username'] = $user['username'];
        $_SESSION['user_role'] = $user['role'];
        $_SESSION['logged_in'] = true;
        $_SESSION['login_time'] = time();
    }
    
    /**
     * Prüft ob User eingeloggt ist
     */
    public static function isLoggedIn() {
        if (class_exists('SessionManager')) {
            return SessionManager::isLoggedIn();
        }
        return isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
    }
    
    /**
     * Prüft ob User Admin ist
     */
    public static function isAdmin() {
        if (class_exists('SessionManager')) {
            return SessionManager::hasRole('admin');
        }
        return isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'admin';
    }
}