<?php
/** cms-core/backend/auth/session.php - IONOS COMPATIBLE + ULTRA SECURE */

/**
 * Session Manager - OWASP Compliant
 * IONOS Compatible: Handles auto-started sessions
 */

class SessionManager {
    
    /**
     * Startet Session mit sicheren Einstellungen
     */
    public static function start() {
        // Check if session is already active (auto_start on IONOS)
        if (session_status() === PHP_SESSION_ACTIVE) {
            // Session already running - just load security
            self::loadSessionSecurity();
            return;
        }
        
        // Check if session is disabled on server
        if (session_status() === PHP_SESSION_DISABLED) {
            error_log('CRITICAL: Sessions are disabled on this server!');
            return;
        }
        
        // Only set ini settings if session NOT started yet
        if (session_status() === PHP_SESSION_NONE && !headers_sent()) {
            ini_set('session.cookie_httponly', 1);
            ini_set('session.cookie_samesite', 'Strict');
            ini_set('session.use_strict_mode', 1);
            ini_set('session.use_only_cookies', 1);
            ini_set('session.cookie_lifetime', 0);
            ini_set('session.gc_maxlifetime', defined('SESSION_LIFETIME') ? SESSION_LIFETIME : 7200);
            
            // CRITICAL: Dynamic secure cookie based on HTTPS
            $isSecure = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') || 
                        (!empty($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https') ||
                        (defined('FORCE_SSL') && FORCE_SSL);
            
            ini_set('session.cookie_secure', $isSecure ? 1 : 0);
            
            session_name(defined('SESSION_NAME') ? SESSION_NAME : 'PHPSESSID');
        }
        
        // Start session if not already started
        if (session_status() === PHP_SESSION_NONE) {
            if (headers_sent($file, $line)) {
                error_log("Cannot start session - headers already sent in $file on line $line");
                return;
            }
            
            session_start();
        }
        
        // Initialize session metadata
        self::initSessionMetadata();
        
        // Load session security
        self::loadSessionSecurity();
    }
    
    /**
     * Initialize session metadata for tracking
     */
    private static function initSessionMetadata() {
        if (!isset($_SESSION['_created'])) {
            $_SESSION['_created'] = time();
        }
        if (!isset($_SESSION['_last_activity'])) {
            $_SESSION['_last_activity'] = time();
        }
        if (!isset($_SESSION['_last_regeneration'])) {
            $_SESSION['_last_regeneration'] = time();
        }
        
        // Update last activity
        $_SESSION['_last_activity'] = time();
        
        // Auto-regenerate every 30 minutes
        if (time() - $_SESSION['_last_regeneration'] > 1800) {
            self::regenerate();
            $_SESSION['_last_regeneration'] = time();
        }
    }
    
    /**
     * Lädt und validiert Session Security
     */
    private static function loadSessionSecurity() {
        $sessionHijackFile = dirname(__DIR__) . '/security/session_hijack.php';
        
        if (file_exists($sessionHijackFile)) {
            require_once $sessionHijackFile;
            
            if (class_exists('SessionSecurity')) {
                if (!SessionSecurity::validate()) {
                    // Session ungültig - erstelle neue
                    self::destroy();
                    if (session_status() === PHP_SESSION_NONE) {
                        session_start();
                        self::initSessionMetadata();
                    }
                }
            }
        }
    }
    
    /**
     * Setzt Session-Wert
     */
    public static function set($key, $value) {
        $_SESSION[$key] = $value;
    }
    
    /**
     * Holt Session-Wert
     */
    public static function get($key, $default = null) {
        return $_SESSION[$key] ?? $default;
    }
    
    /**
     * Prüft ob Session-Key existiert
     */
    public static function has($key) {
        return isset($_SESSION[$key]);
    }
    
    /**
     * Löscht Session-Wert
     */
    public static function delete($key) {
        unset($_SESSION[$key]);
    }
    
    /**
     * Flash Message System
     */
    public static function flash($key, $value = null) {
        if ($value === null) {
            $flash = $_SESSION['_flash'][$key] ?? null;
            unset($_SESSION['_flash'][$key]);
            return $flash;
        }
        $_SESSION['_flash'][$key] = $value;
    }
    
    /**
     * Zerstört Session
     */
    public static function destroy() {
        if (class_exists('SessionSecurity')) {
            SessionSecurity::destroy();
        } else {
            // Fallback
            $_SESSION = [];
            if (isset($_COOKIE[session_name()])) {
                setcookie(session_name(), '', time() - 3600, '/');
            }
            if (session_status() === PHP_SESSION_ACTIVE) {
                session_destroy();
            }
        }
    }
    
    /**
     * Regeneriert Session-ID
     */
    public static function regenerate() {
        if (class_exists('SessionSecurity')) {
            SessionSecurity::regenerate();
        } else {
            // Fallback
            if (session_status() === PHP_SESSION_ACTIVE) {
                session_regenerate_id(true);
            }
        }
    }
    
    /**
     * Prüft ob User eingeloggt ist
     */
    public static function isLoggedIn() {
        return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
    }
    
    /**
     * Prüft User-Rolle
     */
    public static function hasRole($role) {
        if (!self::isLoggedIn()) {
            return false;
        }
        return ($_SESSION['user_role'] ?? '') === $role;
    }
    
    /**
     * Validiert Session-Integrität
     */
    public static function validate() {
        // Check for session hijack protection class
        if (class_exists('SessionSecurity')) {
            return SessionSecurity::validate();
        }
        
        // Basic validation fallback
        // Check session timeout (2 hours)
        if (isset($_SESSION['_last_activity']) && 
            (time() - $_SESSION['_last_activity'] > 7200)) {
            self::destroy();
            return false;
        }
        
        return true;
    }
    
    /**
     * Get session age in seconds
     */
    public static function getAge() {
        if (!isset($_SESSION['_created'])) {
            return 0;
        }
        return time() - $_SESSION['_created'];
    }
    
    /**
     * Get time since last activity (idle time)
     */
    public static function getIdleTime() {
        if (!isset($_SESSION['_last_activity'])) {
            return 0;
        }
        return time() - $_SESSION['_last_activity'];
    }
    
    /**
     * Session-Info für Debugging
     */
    public static function getInfo() {
        if (class_exists('SessionSecurity') && method_exists('SessionSecurity', 'getInfo')) {
            return SessionSecurity::getInfo();
        }
        
        return [
            'status' => session_status(),
            'session_id' => session_id(),
            'age' => self::getAge(),
            'idle_time' => self::getIdleTime(),
            'secure_cookie' => ini_get('session.cookie_secure'),
            'httponly' => ini_get('session.cookie_httponly'),
            'samesite' => ini_get('session.cookie_samesite'),
            'data' => $_SESSION
        ];
    }
}