<?php
/** cms-core/backend/authorization/check_admin.php - COMPLETE WORKING VERSION */

/**
 * Admin Access Check
 * Verifies user is logged in and has admin access
 * Works with or without role system (backwards compatible)
 */

// Load init if not already loaded
if (!defined('APP_ROOT')) {
    require_once dirname(__DIR__) . '/core/init.php';
}

// Check if UserRoles exists
$hasRoleSystem = class_exists('UserRoles');

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: /admin/login.php');
    exit;
}

// If role system exists, verify user has admin access
if ($hasRoleSystem) {
    try {
        $user = UserRoles::getUser($_SESSION['user_id']);
        
        if (!$user) {
            // User not found - destroy session
            session_destroy();
            header('Location: /admin/login.php');
            exit;
        }
        
        // Check if user is active
        if (isset($user['is_active']) && !$user['is_active']) {
            http_response_code(403);
            die('Ihr Konto wurde deaktiviert. Bitte kontaktieren Sie einen Administrator.');
        }
        
        // Store user info in session for easy access
        $_SESSION['user_role'] = $user['role'];
        if (isset($user['email'])) {
            $_SESSION['user_email'] = $user['email'];
        }
        
    } catch (Exception $e) {
        // Log error but don't break
        if (function_exists('Logger')) {
            Logger::error('check_admin.php error: ' . $e->getMessage());
        }
        // Fallback to basic check
        if (!isset($_SESSION['user_role'])) {
            $_SESSION['user_role'] = 'admin';
        }
    }
} else {
    // Legacy mode - no role system
    if (!isset($_SESSION['user_role'])) {
        $_SESSION['user_role'] = 'admin';
    }
}

/**
 * Helper function to check if logged-in user is admin
 */
if (!function_exists('is_admin')) {
    function is_admin() {
        global $hasRoleSystem;
        
        if ($hasRoleSystem && class_exists('UserRoles')) {
            return UserRoles::isAdmin();
        }
        
        return isset($_SESSION['user_role']) && 
               in_array($_SESSION['user_role'], ['admin', 'super_admin']);
    }
}

/**
 * Helper function to get current user
 */
if (!function_exists('current_user')) {
    function current_user() {
        global $hasRoleSystem;
        
        if ($hasRoleSystem && class_exists('UserRoles')) {
            return UserRoles::getUser($_SESSION['user_id']);
        }
        
        return [
            'id' => $_SESSION['user_id'] ?? 0,
            'username' => $_SESSION['username'] ?? 'admin',
            'role' => $_SESSION['user_role'] ?? 'admin'
        ];
    }
}

/**
 * Helper function to check permission
 */
if (!function_exists('can')) {
    function can($permission) {
        global $hasRoleSystem;
        
        if ($hasRoleSystem && class_exists('UserRoles')) {
            try {
                return UserRoles::can($permission);
            } catch (Exception $e) {
                // Fallback to admin check
                return is_admin();
            }
        }
        
        // Legacy fallback - admins can do everything
        return is_admin();
    }
}

/**
 * Require specific permission
 */
if (!function_exists('require_permission')) {
    function require_permission($permission, $message = null) {
        if (!can($permission)) {
            http_response_code(403);
            die($message ?? 'Sie haben keine Berechtigung für diese Aktion.');
        }
    }
}