<?php
/**
 * cms-core/backend/core/FlexibleBlocks.php
 * Supports both built-in blocks and external block files
 * ALL inline color styles removed, uses render_richtext() for RichText fields
 */
class FlexibleBlocks {
/**
 * Get blocks for a page
 */
public static function getPageBlocks($pageSlug) {
    try {
        $blocks = db()->fetchAll(
            "SELECT * FROM page_blocks 
             WHERE page_slug = ? 
             ORDER BY sort_order ASC, id ASC",
            [$pageSlug]
        );
        
        return $blocks;
    } catch (Exception $e) {
        error_log("FlexibleBlocks::getPageBlogs Error: " . $e->getMessage());
        return [];
    }
}

public static function renderBlock($block) {
    if (empty($block)) {
        return '';
    }
    
    $blockType = $block['block_type_template'] ?? $block['block_type'] ?? 'text';
    
    // Decode block_data JSON
    $data = [];
    if (!empty($block['block_data'])) {
        $data = json_decode($block['block_data'], true) ?? [];
    }
    
    // Also support legacy block_value field
    if (empty($data) && !empty($block['block_value'])) {
        $data = json_decode($block['block_value'], true) ?? [];
    }
    
    // Check multiple possible template locations (in order of priority):
    $possiblePaths = [
        // 1. Direct block file in blocks/
        APP_ROOT . '/cms-core/public/webview/blocks/' . $blockType . '.php',
        
        // 3. Custom blocks (if CUSTOM_ROOT is defined)
        defined('CUSTOM_ROOT') ? CUSTOM_ROOT . '/flexible-blocks/' . $blockType . '.php' : null,
        
        // 4. Custom templates subfolder
        defined('CUSTOM_ROOT') ? CUSTOM_ROOT . '/flexible-blocks/templates/' . $blockType . '.php' : null,
    ];
    
    // Find the first template that exists
    $templatePath = null;
    foreach ($possiblePaths as $path) {
        if ($path && file_exists($path)) {
            $templatePath = $path;
            break;
        }
    }
    
    // If template file exists, use it
    if ($templatePath) {
        // CRITICAL: Start fresh output buffer
        ob_start();
        try {
            include $templatePath;
            return ob_get_clean();
        } catch (Exception $e) {
            ob_end_clean();
            error_log("Error rendering block template {$blockType}: " . $e->getMessage());
            return '<!-- Error rendering block: ' . htmlspecialchars($e->getMessage()) . ' -->';
        }
    }
    
    // Otherwise, try built-in render methods (backwards compatibility)
    $renderMethod = 'render' . str_replace('_', '', ucwords($blockType, '_'));
    
    if (method_exists(self::class, $renderMethod)) {
        return self::$renderMethod($data);
    }
    
    // Nothing found - show helpful debug message
    $checkedPaths = array_filter($possiblePaths);
    error_log("Block template not found for '{$blockType}'. Checked: " . implode(', ', $checkedPaths));
    
    return '<!-- Block template not found: ' . htmlspecialchars($blockType) . ' -->';
}
/**
 * Render Hero Banner - FIXED (No inline styles, proper alignment)
 */
private static function renderHeroBanner($data) {
    $title = $data['title'] ?? ['text' => 'Willkommen'];
    $subtitle = $data['subtitle'] ?? '';
    $image = $data['image'] ?? $data['background_image'] ?? '';
    $buttonText = $data['button_text'] ?? 'Mehr erfahren';
    $buttonLink = $data['button_link'] ?? '#';
    $showButton = isset($data['button_text']) && !empty($data['button_text']);

    $isFullframe = $data['fullframe'] ?? false;

    if ($image && !str_starts_with($image, 'http')) {
        $image = '/media/pages/' . $image;
    }

    ob_start();

    if ($isFullframe) {
        $bgStyle = $image 
            ? "background-image: url('" . e($image) . "'); background-size: cover; background-position: center;" 
            : "background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);";
        ?>
        <section class="hero-banner fullframe" style="<?= $bgStyle ?> padding: 120px 20px; position: relative; min-height: 500px;">
            <div style="position: absolute; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.4);"></div>
            <div class="container" style="max-width: 1200px; margin: 0 auto; position: relative; z-index: 1; text-align: center;">
                <?= render_richtext($title, 'h1') ?>
                <?php if (!empty($subtitle)): ?>
                    <?= render_richtext($subtitle, 'p') ?>
                <?php endif; ?>
                <?php if ($showButton): ?>
                    <a href="<?= e($buttonLink) ?>" class="btn btn-white">
                        <?= e($buttonText) ?>
                    </a>
                <?php endif; ?>
            </div>
        </section>
        <?php
    } else {
        ?>
        <section class="hero-banner two-column">
            <div class="container">
                <div>
                    <div>
                        <?= render_richtext($title, 'h1') ?>
                        <?php if (!empty($subtitle)): ?>
                            <?= render_richtext($subtitle, 'p') ?>
                        <?php endif; ?>
                        <?php if ($showButton): ?>
                            <a href="<?= e($buttonLink) ?>" class="btn btn-primary">
                                <?= e($buttonText) ?>
                            </a>
                        <?php endif; ?>
                    </div>
                    
                    <div>
                        <?php if ($image): ?>
                            <img src="<?= e($image) ?>" alt="Hero">
                        <?php else: ?>
                            <div style="width: 100%; height: 400px; background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%); border-radius: 20px; display: flex; align-items: center; justify-content: center; font-size: 24px; color: white;">
                                Bild hier
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </section>
        <?php
    }

    return ob_get_clean();
}
/**
 * Render Leistungen Block - FIXED button colors
 * Replace the renderLeistungen method in FlexibleBlocks.php
 */
private static function renderLeistungen($data) {
    $title = $data['title'] ?? ['text' => 'Unsere Leistungen'];
    $services = $data['services'] ?? [];
    
    ob_start();
    ?>
    <section class="leistungen">
        <div class="container">
            <?= render_richtext($title, 'h2') ?>
            <div class="services-grid">
                <?php foreach ($services as $service): ?>
                    <div class="service-card">
                        <?php 
                        // Handle Icon OR Image with size
                        $iconType = $service['icon_type'] ?? 'emoji';
                        $iconValue = $service['icon'] ?? '';
                        $iconSize = $service['icon_size'] ?? 'medium-square';
                        
                        // Size mapping
                        $sizeMap = [
                            'small-square' => ['width' => '60px', 'height' => '60px'],
                            'medium-square' => ['width' => '80px', 'height' => '80px'],
                            'large-square' => ['width' => '120px', 'height' => '120px'],
                            'small-rect' => ['width' => '100px', 'height' => '60px'],
                            'medium-rect' => ['width' => '150px', 'height' => '90px'],
                            'large-rect' => ['width' => '200px', 'height' => '120px']
                        ];
                        
                        $dimensions = $sizeMap[$iconSize] ?? $sizeMap['medium-square'];
                        
                        if (!empty($iconValue)): ?>
                            <div class="service-icon">
                                <?php if ($iconType === 'image'): ?>
                                    <img src="/media/pages/<?= e($iconValue) ?>" 
                                         alt="Service Icon" 
                                         style="width: <?= $dimensions['width'] ?>; height: <?= $dimensions['height'] ?>; object-fit: contain; border-radius: 8px;">
                                <?php else: ?>
                                    <span style="font-size: 3rem;"><?= e($iconValue) ?></span>
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>
                        
                        <?php
                        $serviceTitle = $service['title'] ?? '';
                        if (is_string($serviceTitle)) {
                            $decoded = json_decode($serviceTitle, true);
                            if (is_array($decoded)) {
                                echo render_richtext($decoded, 'h3');
                            } else {
                                echo '<h3>' . e($serviceTitle) . '</h3>';
                            }
                        } elseif (is_array($serviceTitle)) {
                            echo render_richtext($serviceTitle, 'h3');
                        }
                        ?>
                        
                        <?php
                        $serviceDesc = $service['description'] ?? '';
                        if (is_string($serviceDesc)) {
                            $decoded = json_decode($serviceDesc, true);
                            if (is_array($decoded)) {
                                echo render_richtext($decoded, 'p');
                            } else {
                                echo '<p>' . e($serviceDesc) . '</p>';
                            }
                        } elseif (is_array($serviceDesc)) {
                            echo render_richtext($serviceDesc, 'p');
                        }
                        ?>
                        
                        <?php if (!empty($service['button_link'])): ?>
                            <a href="/<?= e(str_replace('.php', '', $service['button_link'])) ?>" class="btn btn-primary" style="margin-top: 15px; display: inline-block;">
                                <?= e($service['button_text'] ?? 'Mehr erfahren') ?>
                            </a>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    </section>
    <?php
    return ob_get_clean();
}
/**
 * Render Text Block - FIXED (No padding)
 */
private static function renderTextBlock($data) {
    $title = $data['title'] ?? '';
    $text = $data['text'] ?? '';
    
    if (empty($title) && empty($text)) {
        return '';
    }
    
    ob_start();
    ?>
    <section class="text-block">
        <div class="container" style="max-width: 900px; margin: 0 auto; text-align: center;">
            <?php if (!empty($title)): ?>
                <?php 
                if (is_array($title)) {
                    echo render_richtext($title, 'h3');
                } else {
                    echo '<h2>' . e($title) . '</h3>';
                }
                ?>
            <?php endif; ?>
            
            <?php if (!empty($text)): ?>
                <div style="line-height: 1.8; font-size: 18px; margin-top: 20px;">
                    <?php 
                    if (is_array($text)) {
                        echo render_richtext($text, 'div');
                    } else {
                        echo '<div>' . nl2br(e($text)) . '</div>';
                    }
                    ?>
                </div>
            <?php endif; ?>
        </div>
    </section>
    <?php
    return ob_get_clean();
}

/**
 * Render Contact Form - FIXED
 */
private static function renderKontaktForm($data) {
    $title = $data['title'] ?? ['text' => 'Kontaktieren Sie uns'];
    $subtitle = $data['subtitle'] ?? '';
    $showPhone = $data['show_phone'] ?? true;
    $showEmail = $data['show_email'] ?? true;
    
    ob_start();
    ?>
    <section class="kontakt-form" style="padding: 80px 20px;">
        <div class="container" style="max-width: 800px; margin: 0 auto;">
            <?= render_richtext($title, 'h2') ?>
            <?php if (!empty($subtitle)): ?>
                <?= render_richtext($subtitle, 'p') ?>
            <?php endif; ?>
            
            <form method="POST" action="/kontakt-submit.php" style="background: var(--white); padding: 40px; border-radius: 12px; box-shadow: 0 4px 6px rgba(0,0,0,0.07); margin-top: 40px;">
                <div style="margin-bottom: 20px;">
                    <label style="display: block; margin-bottom: 8px; font-weight: 600;">Name *</label>
                    <input type="text" name="name" required style="width: 100%; padding: 12px; border: 1px solid var(--border-color); border-radius: 8px; font-size: 16px; box-sizing: border-box;">
                </div>
                
                <div style="margin-bottom: 20px;">
                    <label style="display: block; margin-bottom: 8px; font-weight: 600;">E-Mail *</label>
                    <input type="email" name="email" required style="width: 100%; padding: 12px; border: 1px solid var(--border-color); border-radius: 8px; font-size: 16px; box-sizing: border-box;">
                </div>
                
                <?php if ($showPhone): ?>
                <div style="margin-bottom: 20px;">
                    <label style="display: block; margin-bottom: 8px; font-weight: 600;">Telefon</label>
                    <input type="tel" name="phone" style="width: 100%; padding: 12px; border: 1px solid var(--border-color); border-radius: 8px; font-size: 16px; box-sizing: border-box;">
                </div>
                <?php endif; ?>
                
                <div style="margin-bottom: 20px;">
                    <label style="display: block; margin-bottom: 8px; font-weight: 600;">Nachricht *</label>
                    <textarea name="message" required rows="6" style="width: 100%; padding: 12px; border: 1px solid var(--border-color); border-radius: 8px; font-size: 16px; resize: vertical; box-sizing: border-box;"></textarea>
                </div>
                
                <button type="submit" class="btn btn-primary" style="padding: 15px 40px; border: none; border-radius: 8px; font-size: 16px; font-weight: 600; cursor: pointer; width: 100%; transition: background 0.3s;">
                    Nachricht senden
                </button>
            </form>
        </div>
    </section>
    <?php
    return ob_get_clean();
}

/**
 * Render Two Column Image Right - FIXED
 */
private static function renderTwoColumnImageRight($data) {
    $title = $data['title'] ?? '';
    $text = $data['text'] ?? '';
    $image = $data['image'] ?? '';
    
    if ($image && !str_starts_with($image, 'http')) {
        $image = '/media/pages/' . $image;
    }
    
    ob_start();
    ?>
    <section class="two-column-image-right" style="padding: 80px 20px;">
        <div class="container" style="max-width: 1200px; margin: 0 auto;">
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 60px; align-items: center;">
                <div>
                    <?= render_richtext($title, 'h2') ?>
                    <div style="line-height: 1.8; font-size: 16px; margin-top: 20px;">
                        <?= render_richtext($text, 'div') ?>
                    </div>
                </div>
                <div>
                    <?php if ($image): ?>
                        <img src="<?= e($image) ?>" alt="<?= get_richtext_plain($title) ?>" style="width: 100%; border-radius: 12px; box-shadow: 0 10px 20px rgba(0,0,0,0.1);">
                    <?php else: ?>
                        <div style="width: 100%; height: 400px; border-radius: 12px; background: var(--bg-light); display: flex; align-items: center; justify-content: center; color: var(--text-light);">
                            Bild hier
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </section>
    <?php
    return ob_get_clean();
}

/**
 * Render Two Column Image Left - FIXED
 */
private static function renderTwoColumnImageLeft($data) {
    $title = $data['title'] ?? '';
    $text = $data['text'] ?? '';
    $image = $data['image'] ?? '';
    
    if ($image && !str_starts_with($image, 'http')) {
        $image = '/media/pages/' . $image;
    }
    
    ob_start();
    ?>
    <section class="two-column-image-left" style="padding: 80px 20px;">
        <div class="container" style="max-width: 1200px; margin: 0 auto;">
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 60px; align-items: center;">
                <div>
                    <?php if ($image): ?>
                        <img src="<?= e($image) ?>" alt="<?= get_richtext_plain($title) ?>" style="width: 100%; border-radius: 12px; box-shadow: 0 10px 20px rgba(0,0,0,0.1);">
                    <?php else: ?>
                        <div style="width: 100%; height: 400px; border-radius: 12px; background: var(--bg-light); display: flex; align-items: center; justify-content: center; color: var(--text-light);">
                            Bild hier
                        </div>
                    <?php endif; ?>
                </div>
                <div>
                    <?= render_richtext($title, 'h2') ?>
                    <div style="line-height: 1.8; font-size: 16px; margin-top: 20px;">
                        <?= render_richtext($text, 'div') ?>
                    </div>
                </div>
            </div>
        </div>
    </section>
    <?php
    return ob_get_clean();
}

/**
 * Render Image Block - FIXED
 */
private static function renderImageBlock($data) {
    $image = $data['image'] ?? '';
    $caption = $data['caption'] ?? '';
    $size = $data['size'] ?? 'large';
    
    if ($image && !str_starts_with($image, 'http')) {
        $image = '/media/pages/' . $image;
    }
    
    if (empty($image)) {
        return '';
    }
    
    $widthMap = [
        'small' => '400px',
        'medium' => '800px',
        'large' => '1000px',
        'full' => '100%'
    ];
    $maxWidth = $widthMap[$size] ?? '1000px';
    
    ob_start();
    ?>
    <section class="image-block" style="padding: 40px 20px;">
        <div class="container" style="max-width: <?= $maxWidth ?>; margin: 0 auto; text-align: center;">
            <img src="<?= e($image) ?>" alt="<?= get_richtext_plain($caption) ?>" style="width: 100%; border-radius: 12px; box-shadow: 0 10px 20px rgba(0,0,0,0.1);">
            <?php if (!empty($caption)): ?>
                <?= render_richtext($caption, 'p') ?>
            <?php endif; ?>
        </div>
    </section>
    <?php
    return ob_get_clean();
}


/**
 * Render Blog Posts - FIXED
 */
private static function renderBlogPosts($data) {
    $title = $data['title'] ?? ['text' => 'Neueste Beiträge'];
    $limit = $data['limit'] ?? 3;
    
    $posts = db()->fetchAll(
        "SELECT * FROM blog_posts WHERE status = 'published' ORDER BY published_at DESC LIMIT ?",
        [(int)$limit]
    );
    
    if (empty($posts)) {
        return '';
    }
    
    ob_start();
    ?>
    <section class="blog-posts" style="padding: 80px 20px;">
        <div class="container" style="max-width: 1200px; margin: 0 auto;">
            <?= render_richtext($title, 'h2') ?>
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 30px; margin-top: 60px;">
                <?php foreach ($posts as $post): ?>
                    <?php
                    $featuredImage = $post['featured_image'] ?? '';
                    if ($featuredImage && !str_starts_with($featuredImage, 'http')) {
                        $featuredImage = '/media/blog/' . $featuredImage;
                    }
                    ?>
                    <article style="background: var(--white); border-radius: 12px; overflow: hidden; box-shadow: 0 4px 6px rgba(0,0,0,0.07); transition: transform 0.3s;" onmouseover="this.style.transform='translateY(-5px)'; this.style.boxShadow='0 8px 12px rgba(0,0,0,0.1)'" onmouseout="this.style.transform=''; this.style.boxShadow='0 4px 6px rgba(0,0,0,0.07)'">
                        <?php if ($featuredImage): ?>
                            <img src="<?= e($featuredImage) ?>" alt="<?= e($post['title']) ?>" style="width: 100%; height: 200px; object-fit: cover;">
                        <?php endif; ?>
                        <div style="padding: 25px;">
                            <h3 style="font-size: 22px; margin-bottom: 15px;">
                                <a href="/blog/<?= e($post['slug']) ?>" class="blog-post-link" style="text-decoration: none;">
                                    <?= e($post['title']) ?>
                                </a>
                            </h3>
                            <p style="line-height: 1.6; margin-bottom: 20px;">
                                <?= e(substr($post['excerpt'] ?? '', 0, 150)) ?>...
                            </p>
                            <a href="/blog/<?= e($post['slug']) ?>" class="blog-readmore-link" style="text-decoration: none; font-weight: 600;">
                                Weiterlesen →
                            </a>
                        </div>
                    </article>
                <?php endforeach; ?>
            </div>
        </div>
    </section>
    <?php
    return ob_get_clean();
}

/**
 * Render Kontakt Banner - FIXED (No inline button styles)
 */
private static function renderKontaktBanner($data) {
    $title = $data['title'] ?? ['text' => 'Bereit für den nächsten Schritt?'];
    $subtitle = $data['subtitle'] ?? '';
    $buttonText = $data['button_text'] ?? 'Kontakt aufnehmen';
    $buttonLink = $data['button_link'] ?? '/kontakt';
    
    ob_start();
    ?>
    <section class="kontakt-banner">
        <div class="container">
            <?= render_richtext($title, 'h2') ?>
            <?php if (!empty($subtitle)): ?>
                <?= render_richtext($subtitle, 'p') ?>
            <?php endif; ?>
            <a href="<?= e($buttonLink) ?>" class="btn btn-white">
                <?= e($buttonText) ?>
            </a>
        </div>
    </section>
    <?php
    return ob_get_clean();
}

/**
 * Render Pricing Block - FIXED
 */
private static function renderPricing($data) {
    $title = $data['title'] ?? ['text' => 'Unsere Preise'];
    $subtitle = $data['subtitle'] ?? '';
    $plans = $data['plans'] ?? [];
    
    if (empty($plans)) {
        return '';
    }
    
    ob_start();
    ?>
    <section class="pricing-section" style="padding: 80px 20px;">
        <div class="container" style="max-width: 1200px; margin: 0 auto;">
            <div style="text-align: center; margin-bottom: 60px;">
                <?= render_richtext($title, 'h2') ?>
                <?php if (!empty($subtitle)): ?>
                    <?= render_richtext($subtitle, 'p') ?>
                <?php endif; ?>
            </div>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(280px, 1fr)); gap: 30px;">
                <?php foreach ($plans as $plan): ?>
                    <?php 
                    $isHighlighted = $plan['highlighted'] ?? false;
                    $cardStyle = $isHighlighted 
                        ? "border-color: var(--primary-color); box-shadow: 0 10px 30px rgba(59, 130, 246, 0.2); transform: scale(1.05);" 
                        : "";
                    ?>
                    <div class="pricing-card" style="background: var(--white); border: 2px solid var(--border-color); border-radius: 16px; padding: 40px 30px; text-align: center; position: relative; transition: all 0.3s; <?= $cardStyle ?>" onmouseover="this.style.transform='translateY(-5px) <?= $isHighlighted ? 'scale(1.05)' : '' ?>'; this.style.boxShadow='0 20px 40px rgba(0,0,0,0.1)'" onmouseout="this.style.transform='<?= $isHighlighted ? 'scale(1.05)' : '' ?>'; this.style.boxShadow='<?= $isHighlighted ? '0 10px 30px rgba(59, 130, 246, 0.2)' : '0 4px 6px rgba(0,0,0,0.07)' ?>'">
                        <?php if ($isHighlighted && !empty($plan['badge'])): ?>
                            <div style="position: absolute; top: -15px; left: 50%; transform: translateX(-50%); background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%); color: white; padding: 6px 20px; border-radius: 20px; font-size: 13px; font-weight: 600;">
                                <?= render_richtext($plan['badge'], 'span') ?>
                            </div>
                        <?php endif; ?>
                        
                        <?= render_richtext($plan['name'] ?? '', 'h3') ?>
                        
                        <div style="margin-bottom: 30px;">
                            <?= render_richtext($plan['price'] ?? '', 'div') ?>
                            <?= render_richtext($plan['period'] ?? '', 'div') ?>
                        </div>
                        
                        <?php if (!empty($plan['features'])): ?>
                            <ul style="list-style: none; padding: 0; margin: 0 0 30px 0; text-align: left;">
                                <?php foreach ($plan['features'] as $feature): ?>
                                    <li style="padding: 12px 0; border-bottom: 1px solid var(--border-color); font-size: 15px;">
                                        ✓ <?= e($feature) ?>
                                    </li>
                                <?php endforeach; ?>
                            </ul>
                        <?php endif; ?>
                        
                        <?php if (!empty($plan['button_text'])): ?>
                            <a href="<?= e($plan['button_link'] ?? '#') ?>" class="btn pricing-btn" style="display: block; background: <?= $isHighlighted ? 'var(--primary-color)' : 'var(--border-color)' ?>; color: <?= $isHighlighted ? 'white' : 'var(--text-dark)' ?> !important; padding: 12px 30px; border-radius: 8px; text-decoration: none; font-weight: 600; transition: all 0.3s;">
                                <?= e($plan['button_text']) ?>
                            </a>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    </section>
    <?php
    return ob_get_clean();
}

/**
 * Render FAQ Block - FIXED
 */
private static function renderFaq($data) {
    $title = $data['title'] ?? ['text' => 'Häufig gestellte Fragen'];
    $faqs = $data['faqs'] ?? [];
    
    if (empty($faqs)) {
        return '';
    }

    ob_start();
    ?>
    <section class="faq-section" style="padding: 80px 20px;">
        <div class="container" style="max-width: 900px; margin: 0 auto;">
            <?= render_richtext($title, 'h2') ?>
            
            <div style="margin-top: 40px;">
                <?php foreach ($faqs as $index => $faq): ?>
                    <div class="faq-item" style="margin-bottom: 20px; border: 1px solid var(--border-color); border-radius: 12px; overflow: hidden; background: var(--white);">
                        <button 
                            type="button" 
                            onclick="this.parentElement.classList.toggle('active'); this.querySelector('.faq-icon').style.transform = this.parentElement.classList.contains('active') ? 'rotate(180deg)' : 'rotate(0deg)';"
                            style="width: 100%; padding: 20px 24px; text-align: left; background: var(--white); border: none; cursor: pointer; display: flex; justify-content: space-between; align-items: center; font-size: 18px; font-weight: 600; transition: background 0.3s;"
                            onmouseover="this.style.background='var(--bg-light)'"
                            onmouseout="this.style.background='var(--white)'">
                            <span><?= e($faq['question'] ?? '') ?></span>
                            <span class="faq-icon" style="font-size: 24px; transition: transform 0.3s; color: var(--primary-color);">▼</span>
                        </button>
                        <div class="faq-answer" style="max-height: 0; overflow: hidden; transition: max-height 0.3s ease;">
                            <div style="padding: 0 24px 20px 24px; line-height: 1.8;">
                                <?= nl2br(e($faq['answer'] ?? '')) ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        
        <style>
            .faq-item.active .faq-answer {
                max-height: 500px;
            }
        </style>
    </section>
    <?php
    return ob_get_clean();
}

/**
 * Render Mailchimp Signup Block
 */
private static function renderMailchimpSignup($data) {
    $title = $data['title'] ?? 'Newsletter abonnieren';
    $subtitle = $data['subtitle'] ?? '';
    $buttonText = $data['button_text'] ?? 'Jetzt anmelden';
    $successMessage = $data['success_message'] ?? 'Vielen Dank!';
    $showPrivacy = $data['show_privacy_checkbox'] ?? true;
    $privacyText = $data['privacy_text'] ?? 'Ich habe die Datenschutzerklärung gelesen.';
    $bgColor = $data['background_color'] ?? '#f9fafb';
    $blockId = 'mailchimp-' . uniqid();
    
    ob_start();
    ?>
    <section class="mailchimp-signup-block" style="padding: 80px 20px; background: <?= e($bgColor) ?>;">
        <div class="container" style="max-width: 700px; margin: 0 auto; text-align: center;">
            
            <!-- Title -->
            <h2 style="font-size: 2.5rem; margin-bottom: 20px; color: #1f2937; font-weight: 700;">
                <?= e($title) ?>
            </h2>
            
            <!-- Subtitle -->
            <?php if ($subtitle): ?>
                <p style="color: #6b7280; margin-bottom: 40px; font-size: 1.125rem; line-height: 1.6;">
                    <?= nl2br(e($subtitle)) ?>
                </p>
            <?php endif; ?>
            
            <!-- Newsletter Form -->
            <div id="<?= $blockId ?>-wrapper">
                <form 
                    id="<?= $blockId ?>" 
                    class="mailchimp-form" 
                    method="POST" 
                    action="/api/mailchimp-subscribe.php"
                    style="background: white; padding: 40px; border-radius: 16px; box-shadow: 0 10px 25px rgba(0,0,0,0.08);"
                >
                    <!-- Success Message (hidden by default) -->
                    <div id="<?= $blockId ?>-success" style="display: none; background: #d1fae5; color: #065f46; padding: 20px; border-radius: 12px; margin-bottom: 20px; font-weight: 600;">
                        ✓ <?= e($successMessage) ?>
                    </div>
                    
                    <!-- Error Message (hidden by default) -->
                    <div id="<?= $blockId ?>-error" style="display: none; background: #fee2e2; color: #991b1b; padding: 20px; border-radius: 12px; margin-bottom: 20px; font-weight: 600;">
                        ⚠ Ein Fehler ist aufgetreten. Bitte versuchen Sie es erneut.
                    </div>
                    
                    <!-- Email Input -->
                    <div style="display: flex; gap: 12px; margin-bottom: <?= $showPrivacy ? '20px' : '0' ?>; flex-wrap: wrap;">
                        <input 
                            type="email" 
                            name="email" 
                            placeholder="Ihre E-Mail-Adresse" 
                            required
                            style="flex: 1; min-width: 250px; padding: 16px 20px; border: 2px solid #e5e7eb; border-radius: 12px; font-size: 16px; transition: border-color 0.3s;"
                            onfocus="this.style.borderColor='#667eea'"
                            onblur="this.style.borderColor='#e5e7eb'"
                        >
                        <button 
                            type="submit" 
                            style="padding: 16px 32px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none; border-radius: 12px; font-weight: 600; font-size: 16px; cursor: pointer; transition: transform 0.2s, box-shadow 0.2s; box-shadow: 0 4px 12px rgba(102, 126, 234, 0.3);"
                            onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 6px 20px rgba(102, 126, 234, 0.4)'"
                            onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='0 4px 12px rgba(102, 126, 234, 0.3)'"
                        >
                            <?= e($buttonText) ?>
                        </button>
                    </div>
                    
                    <!-- Privacy Checkbox -->
                    <?php if ($showPrivacy): ?>
                        <div style="text-align: left;">
                            <label style="display: flex; align-items: flex-start; gap: 10px; cursor: pointer;">
                                <input type="checkbox" name="privacy_consent" required style="margin-top: 4px; cursor: pointer;">
                                <span style="font-size: 14px; color: #6b7280; line-height: 1.5;">
                                    <?= e($privacyText) ?>
                                </span>
                            </label>
                        </div>
                    <?php endif; ?>
                </form>
            </div>
        </div>
    </section>
    
    <script>
    document.getElementById('<?= $blockId ?>').addEventListener('submit', function(e) {
        e.preventDefault();
        
        const form = this;
        const formData = new FormData(form);
        const successDiv = document.getElementById('<?= $blockId ?>-success');
        const errorDiv = document.getElementById('<?= $blockId ?>-error');
        
        // Hide previous messages
        successDiv.style.display = 'none';
        errorDiv.style.display = 'none';
        
        fetch(form.action, {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                successDiv.style.display = 'block';
                form.reset();
                
                // Hide success message after 5 seconds
                setTimeout(() => {
                    successDiv.style.display = 'none';
                }, 5000);
            } else {
                errorDiv.textContent = '⚠ ' + (data.message || 'Ein Fehler ist aufgetreten.');
                errorDiv.style.display = 'block';
            }
        })
        .catch(error => {
            errorDiv.style.display = 'block';
        });
    });
    </script>
    <?php
    return ob_get_clean();
}
/**
 * Render Testimonials Block - FIXED
 */
private static function renderTestimonials($data) {
    $title = $data['title'] ?? ['text' => 'Was unsere Kunden sagen'];
    $testimonials = $data['testimonials'] ?? [];
    
    if (empty($testimonials)) {
        return '';
    }
    
    ob_start();
    ?>
    <section class="testimonials-section" style="padding: 80px 20px;">
        <div class="container" style="max-width: 1200px; margin: 0 auto;">
            <?= render_richtext($title, 'h2') ?>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 30px; margin-top: 60px;">
                <?php foreach ($testimonials as $testimonial): ?>
                    <div class="testimonial-card" style="background: var(--white); padding: 30px; border-radius: 16px; box-shadow: 0 4px 6px rgba(0,0,0,0.07); position: relative; transition: transform 0.3s;" onmouseover="this.style.transform='translateY(-5px)'; this.style.boxShadow='0 8px 12px rgba(0,0,0,0.1)'" onmouseout="this.style.transform=''; this.style.boxShadow='0 4px 6px rgba(0,0,0,0.07)'">
                        <!-- Rating Stars -->
                        <div style="margin-bottom: 20px; font-size: 24px;">
                            <?php 
                            $rating = (int)($testimonial['rating'] ?? 5);
                            for ($i = 0; $i < 5; $i++) {
                                echo $i < $rating ? '⭐' : '☆';
                            }
                            ?>
                        </div>
                        
                        <!-- Testimonial Text -->
                        <p style="line-height: 1.8; margin-bottom: 24px; font-style: italic;">
                            "<?= e($testimonial['text'] ?? '') ?>"
                        </p>
                        
                        <!-- Author -->
                        <div style="display: flex; align-items: center; gap: 16px;">
                            <?php if (!empty($testimonial['image'])): ?>
                                <?php 
                                $imagePath = $testimonial['image'];
                                if (!str_starts_with($imagePath, 'http')) {
                                    $imagePath = '/media/pages/' . $imagePath;
                                }
                                ?>
                                <img src="<?= e($imagePath) ?>" alt="<?= e($testimonial['name'] ?? '') ?>" style="width: 50px; height: 50px; border-radius: 50%; object-fit: cover;">
                            <?php else: ?>
                                <div style="width: 50px; height: 50px; border-radius: 50%; background: var(--primary-color); display: flex; align-items: center; justify-content: center; font-weight: 600; font-size: 20px; color: white;">
                                    <?= strtoupper(substr($testimonial['name'] ?? 'A', 0, 1)) ?>
                                </div>
                            <?php endif; ?>
                            <div>
                                <div style="font-weight: 600; font-size: 16px;">
                                    <?= e($testimonial['name'] ?? '') ?>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    </section>
    <?php
    return ob_get_clean();
}

/**
 * Render Business Hours Block - FIXED
 */
private static function renderBusinessHours($data) {
    $title = $data['title'] ?? ['text' => 'Öffnungszeiten'];
    $hours = $data['hours'] ?? [];
    
    if (empty($hours)) {
        return '';
    }
    
    ob_start();
    ?>
    <section class="business-hours-section" style="padding: 80px 20px;">
        <div class="container" style="max-width: 600px; margin: 0 auto;">
            <?= render_richtext($title, 'h2') ?>
            
            <div style="background: var(--white); padding: 40px; border-radius: 16px; box-shadow: 0 4px 6px rgba(0,0,0,0.07); margin-top: 40px;">
                <?php foreach ($hours as $index => $hour): ?>
                    <?php 
                    $isClosed = !empty($hour['closed']);
                    $isLastItem = $index === count($hours) - 1;
                    ?>
                    <div style="display: flex; justify-content: space-between; align-items: center; padding: 16px 0; <?= !$isLastItem ? 'border-bottom: 1px solid var(--border-color);' : '' ?>">
                        <div style="font-weight: 600; font-size: 16px;">
                            <?= e($hour['day'] ?? '') ?>
                        </div>
                        <div style="font-size: 16px; <?= $isClosed ? 'font-style: italic;' : '' ?>">
                            <?php if ($isClosed): ?>
                                <span style="opacity: 0.6;">Geschlossen</span>
                            <?php else: ?>
                                <?= e($hour['time'] ?? '') ?>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    </section>
    <?php
    return ob_get_clean();
}

/**
 * Get available block types with field definitions
 */
public static function getAvailableBlockTypes() {
    $blocks = [
        'hero_banner' => [
            'name' => 'Hero Banner',
            'icon' => '🎯',
            'description' => 'Großer Hero-Bereich mit Bild und Text',
            'fields' => [
                'title' => ['type' => 'richtext', 'label' => 'Titel', 'required' => true],
                'subtitle' => ['type' => 'richtext', 'label' => 'Untertitel'],
                'image' => ['type' => 'image', 'label' => 'Bild'],
                'fullframe' => ['type' => 'checkbox', 'label' => 'Vollbild-Modus', 'help' => '✓ Aktiviert: Bild als Vollbild-Hintergrund | ✗ Deaktiviert: Zweispalten-Layout'],
                'button_text' => ['type' => 'text', 'label' => 'Button Text'],
                'button_link' => ['type' => 'text', 'label' => 'Button Link']
            ]
        ],
        // ONLY THE LEISTUNGEN BLOCK DEFINITION - UPDATE THIS IN YOUR FlexibleBlocks.php

'leistungen' => [
    'name' => 'Leistungen / Services',
    'icon' => '📋',
    'description' => 'Grid mit mehreren Service-Karten',
    'fields' => [
        'title' => ['type' => 'richtext', 'label' => 'Titel'],
        'services' => [
            'type' => 'repeater',
            'label' => 'Services',
            'item_label' => 'Leistung',
            'add_button' => 'Leistung hinzufügen',
            'fields' => [
                'icon' => [
                    'type' => 'icon_or_image',
                    'label' => 'Icon (Emoji oder Bild)'
                ],
                'title' => [
                    'type' => 'richtext',
                    'label' => 'Titel',
                    'required' => true
                ],
                'description' => [
                    'type' => 'richtext',
                    'label' => 'Beschreibung'
                ],
                'button_text' => [
                    'type' => 'text',
                    'label' => 'Button Text'
                ],
                'button_link' => [
                    'type' => 'text',
                    'label' => 'Button Link'
                ]
            ]
        ]
    ]
],
        'text_block' => [
            'name' => 'Text Block',
            'icon' => '📝',
            'description' => 'Einfacher Textbereich',
            'fields' => [
                'title' => ['type' => 'richtext', 'label' => 'Titel (optional)'],
                'text' => ['type' => 'richtext', 'label' => 'Text']
            ]
        ],
        'kontakt_form' => [
            'name' => 'Kontakt Formular',
            'icon' => '📧',
            'description' => 'Kontaktformular mit Feldern',
            'fields' => [
                'title' => ['type' => 'richtext', 'label' => 'Titel'],
                'subtitle' => ['type' => 'richtext', 'label' => 'Untertitel'],
                'show_phone' => ['type' => 'checkbox', 'label' => 'Telefon-Feld anzeigen'],
                'show_email' => ['type' => 'checkbox', 'label' => 'E-Mail-Feld anzeigen']
            ]
        ],
        'two_column_image_right' => [
            'name' => 'Zwei Spalten (Bild rechts)',
            'icon' => '📐',
            'description' => 'Text links, Bild rechts',
            'fields' => [
                'title' => ['type' => 'richtext', 'label' => 'Titel'],
                'text' => ['type' => 'richtext', 'label' => 'Text'],
                'image' => ['type' => 'image', 'label' => 'Bild']
            ]
        ],
        'two_column_image_left' => [
            'name' => 'Zwei Spalten (Bild links)',
            'icon' => '📐',
            'description' => 'Bild links, Text rechts',
            'fields' => [
                'title' => ['type' => 'richtext', 'label' => 'Titel'],
                'text' => ['type' => 'richtext', 'label' => 'Text'],
                'image' => ['type' => 'image', 'label' => 'Bild']
            ]
        ],
        'image_block' => [
            'name' => 'Bild Block',
            'icon' => '🖼️',
            'description' => 'Einzelnes Bild mit Beschriftung',
            'fields' => [
                'image' => ['type' => 'image', 'label' => 'Bild'],
                'caption' => ['type' => 'richtext', 'label' => 'Bildunterschrift'],
                'size' => [
                    'type' => 'select',
                    'label' => 'Größe',
                    'options' => [
                        'small' => 'Klein',
                        'medium' => 'Mittel',
                        'large' => 'Groß',
                        'full' => 'Vollbreite'
                    ]
                ]
            ]
        ],
        'bekannt_aus' => [
    'name' => 'Bekannt aus',
    'icon' => '⭐',
    'description' => 'Logo-Leiste für Partner/Medien mit Emoji oder Bild',
    'fields' => [
        'title' => ['type' => 'richtext', 'label' => 'Titel', 'default' => ['text' => 'Bekannt aus']],
        'subtitle' => ['type' => 'richtext', 'label' => 'Untertitel'],
        'logos' => [
            'type' => 'repeater',
            'label' => 'Partner/Logos',
            'item_label' => 'Partner',
            'add_button' => 'Partner hinzufügen',
            'fields' => [
                'name' => [
                    'type' => 'text', 
                    'label' => 'Name', 
                    'required' => true,
                    'placeholder' => 'z.B. TechCrunch, Forbes, etc.'
                ],
                'icon' => [
                    'type' => 'icon_or_image',
                    'label' => 'Logo/Icon',
                    'help' => 'Emoji oder Bild hochladen. Kann auch leer bleiben für nur Text.'
                ],
                'link' => [
                    'type' => 'url',
                    'label' => 'Link (optional)',
                    'placeholder' => 'https://...'
                ]
            ]
        ]
    ]
],
        'blog_posts' => [
            'name' => 'Blog Beiträge',
            'icon' => '📰',
            'description' => 'Zeigt neueste Blog-Posts',
            'fields' => [
                'title' => ['type' => 'richtext', 'label' => 'Titel'],
                'limit' => ['type' => 'number', 'label' => 'Anzahl Beiträge', 'default' => 3]
            ]
        ],
        'kontakt_banner' => [
            'name' => 'Kontakt Banner',
            'icon' => '💬',
            'description' => 'Call-to-Action Banner',
            'fields' => [
                'title' => ['type' => 'richtext', 'label' => 'Titel'],
                'subtitle' => ['type' => 'richtext', 'label' => 'Untertitel'],
                'button_text' => ['type' => 'text', 'label' => 'Button Text'],
                'button_link' => ['type' => 'text', 'label' => 'Button Link']
            ]
        ],
        'pricing' => [
            'name' => 'Preise',
            'icon' => '💰',
            'description' => 'Preistabelle mit Paketen',
            'fields' => [
                'title' => ['type' => 'richtext', 'label' => 'Titel'],
                'subtitle' => ['type' => 'richtext', 'label' => 'Untertitel'],
                'plans' => [
                    'type' => 'repeater',
                    'label' => 'Preispläne',
                    'fields' => [
                        'name' => ['type' => 'richtext', 'label' => 'Paketname'],
                        'price' => ['type' => 'richtext', 'label' => 'Preis'],
                        'period' => ['type' => 'richtext', 'label' => 'Zeitraum'],
                        'highlighted' => ['type' => 'checkbox', 'label' => 'Hervorgehoben'],
                        'badge' => ['type' => 'richtext', 'label' => 'Badge Text'],
                        'features' => ['type' => 'repeater', 'label' => 'Leistungen', 'simple' => true],
                        'button_text' => ['type' => 'text', 'label' => 'Button Text'],
                        'button_link' => ['type' => 'text', 'label' => 'Button Link']
                    ]
                ]
            ]
        ],
        'faq' => [
            'name' => 'FAQ',
            'icon' => '❓',
            'description' => 'Häufig gestellte Fragen',
            'fields' => [
                'title' => ['type' => 'richtext', 'label' => 'Titel'],
                'faqs' => [
                    'type' => 'repeater',
                    'label' => 'Fragen',
                    'fields' => [
                        'question' => ['type' => 'text', 'label' => 'Frage'],
                        'answer' => ['type' => 'textarea', 'label' => 'Antwort']
                    ]
                ]
            ]
        ],
        'testimonials' => [
            'name' => 'Testimonials',
            'icon' => '💬',
            'description' => 'Kundenbewertungen',
            'fields' => [
                'title' => ['type' => 'richtext', 'label' => 'Titel'],
                'testimonials' => [
                    'type' => 'repeater',
                    'label' => 'Bewertungen',
                    'fields' => [
                        'name' => ['type' => 'text', 'label' => 'Name'],
                        'text' => ['type' => 'textarea', 'label' => 'Text'],
                        'rating' => ['type' => 'number', 'label' => 'Bewertung (1-5)', 'default' => 5],
                        'image' => ['type' => 'image', 'label' => 'Bild (optional)']
                    ]
                ]
            ]
        ],
'spacer' => [
    'name' => 'Spacer / Abstand',
    'icon' => '📏',
    'description' => 'Vertikaler Abstand zwischen Blöcken',
    'fields' => [
        'height' => [
            'type' => 'select',
            'label' => 'Höhe',
            'required' => true,
            'default' => 'medium',
            'options' => [
                'small' => 'Klein (20px)',
                'medium' => 'Mittel (40px)',
                'large' => 'Groß (60px)',
                'xlarge' => 'Extra Groß (80px)',
                'xxlarge' => 'XXL (120px)'
            ]
        ],
        'show_line' => [
            'type' => 'checkbox',
            'label' => 'Trennlinie anzeigen',
            'default' => false,
            'help' => 'Zeigt eine dünne Linie in der Mitte des Abstands'
        ]
    ]
],

'mailchimp_signup' => [
    'name' => 'Newsletter Anmeldung',
    'icon' => '📧',
    'description' => 'Mailchimp Newsletter Signup Formular',
    'fields' => [
        'title' => [
            'type' => 'text',
            'label' => 'Titel',
            'default' => 'Newsletter abonnieren'
        ],
        'subtitle' => [
            'type' => 'textarea',
            'label' => 'Untertitel',
            'rows' => 2
        ],
        'button_text' => [
            'type' => 'text',
            'label' => 'Button Text',
            'default' => 'Jetzt anmelden'
        ],
        'success_message' => [
            'type' => 'text',
            'label' => 'Erfolgsmeldung',
            'default' => 'Vielen Dank für Ihre Anmeldung!'
        ],
        'show_privacy_checkbox' => [
            'type' => 'checkbox',
            'label' => 'Datenschutz-Checkbox anzeigen',
            'default' => true
        ],
        'privacy_text' => [
            'type' => 'text',
            'label' => 'Datenschutz-Text',
            'default' => 'Ich habe die Datenschutzerklärung gelesen.'
        ],
        'background_color' => [
            'type' => 'color',
            'label' => 'Hintergrundfarbe',
            'default' => '#f9fafb'
        ]
    ]
],
        'business_hours' => [
            'name' => 'Öffnungszeiten',
            'icon' => '🕐',
            'description' => 'Geschäftszeiten',
            'fields' => [
                'title' => ['type' => 'richtext', 'label' => 'Titel'],
                'hours' => [
                    'type' => 'repeater',
                    'label' => 'Zeiten',
                    'fields' => [
                        'day' => ['type' => 'text', 'label' => 'Tag'],
                        'time' => ['type' => 'text', 'label' => 'Zeit'],
                        'closed' => ['type' => 'checkbox', 'label' => 'Geschlossen']
                    ]
                ]
            ]
        ]
    ];
    
    // Load external blocks from cms-core/public/webview/blocks/
    $coreBlocksPath = APP_ROOT . '/cms-core/public/webview/blocks/';
    if (is_dir($coreBlocksPath)) {
        $coreBlockFiles = glob($coreBlocksPath . '*_block.php');
        if ($coreBlockFiles) {
            foreach ($coreBlockFiles as $file) {
                try {
                    $blockDef = require $file;
                    if (is_array($blockDef)) {
                        $blocks = array_merge($blocks, $blockDef);
                    }
                } catch (Exception $e) {
                    error_log('Error loading core block ' . basename($file) . ': ' . $e->getMessage());
                }
            }
        }
    }
    
    // Load custom blocks from customizations/flexible-blocks/
    if (defined('CUSTOM_ROOT')) {
        $customBlocksPath = CUSTOM_ROOT . '/flexible-blocks/';
        if (is_dir($customBlocksPath)) {
            $customBlockFiles = glob($customBlocksPath . '*_block.php');
            if ($customBlockFiles) {
                foreach ($customBlockFiles as $file) {
                    try {
                        $blockDef = require $file;
                        if (is_array($blockDef)) {
                            $blocks = array_merge($blocks, $blockDef);
                        }
                    } catch (Exception $e) {
                        error_log('Error loading custom block ' . basename($file) . ': ' . $e->getMessage());
                    }
                }
            }
        }
    }
    
    return $blocks;
}

/**
 * Add block
 */
public static function addBlock($pageSlug, $blockType, $data = []) {
    $maxOrder = db()->fetchOne(
        "SELECT MAX(sort_order) as max_order FROM page_blocks WHERE page_slug = ?",
        [$pageSlug]
    );
    $sortOrder = ($maxOrder['max_order'] ?? -1) + 1;
    
    $blockKey = 'block_' . uniqid();
    
    return db()->insert(
        "INSERT INTO page_blocks (page_slug, block_key, block_type, block_type_template, block_data, sort_order, created_at, updated_at) 
         VALUES (?, ?, 'text', ?, ?, ?, NOW(), NOW())",
        [$pageSlug, $blockKey, $blockType, json_encode($data), $sortOrder]
    );
}

/**
 * Update block
 */
public static function updateBlock($blockId, $data) {
    return db()->execute(
        "UPDATE page_blocks SET block_data = ?, updated_at = NOW() WHERE id = ?",
        [json_encode($data), $blockId]
    );
}

/**
 * Delete block
 */
public static function deleteBlock($blockId) {
    return db()->execute("DELETE FROM page_blocks WHERE id = ?", [$blockId]);
}

/**
 * Reorder blocks
 */
public static function reorderBlocks($pageSlug, $blockIds) {
    foreach ($blockIds as $order => $blockId) {
        db()->execute(
            "UPDATE page_blocks SET sort_order = ? WHERE id = ? AND page_slug = ?",
            [$order, $blockId, $pageSlug]
        );
    }
    return true;
}

/**
 * Check if page uses flexible blocks
 */
public static function usesBlocks($pageSlug) {
    $count = db()->fetchOne(
        "SELECT COUNT(*) as count FROM page_blocks WHERE page_slug = ?",
        [$pageSlug]
    );
    return $count['count'] > 0;
}

/**
 * Enable flexible blocks for page
 */
public static function enableFlexibleBlocks($pageSlug) {
    db()->execute(
        "UPDATE pages SET uses_flexible_blocks = 1 WHERE slug = ?",
        [$pageSlug]
    );
    return true;
}
}