<?php
/** cms-core/backend/core/LegalPagesManager.php - FULLY FIXED */

/**
 * Legal Pages Manager
 * Manages mandatory legal pages (Impressum, Datenschutz, etc.)
 */

class LegalPagesManager {
    
    const LEGAL_PAGE_TYPES = [
        'impressum' => [
            'title' => 'Impressum',
            'icon' => '📋',
            'required' => true,
            'description' => 'Anbieterkennzeichnung gemäß TMG'
        ],
        'datenschutz' => [
            'title' => 'Datenschutzerklärung',
            'icon' => '🔒',
            'required' => true,
            'description' => 'Informationen zur Datenverarbeitung gemäß DSGVO'
        ],
        'agb' => [
            'title' => 'AGB',
            'icon' => '📜',
            'required' => false,
            'description' => 'Allgemeine Geschäftsbedingungen'
        ],
        'cookies' => [
            'title' => 'Cookie-Richtlinie',
            'icon' => '🍪',
            'required' => false,
            'description' => 'Informationen zur Verwendung von Cookies'
        ]
    ];
    
    /**
     * Get all legal pages
     * FIXED: Always use COALESCE for page_icon
     */
    public static function getAllLegalPages() {
        return db()->fetchAll(
            "SELECT id, slug, title, COALESCE(page_icon, '📄') AS page_icon, meta_description, status, 
                    show_in_menu, show_in_footer, is_legal_page, legal_page_type, is_deletable, 
                    menu_order, page_type, uses_flexible_blocks, parent_id 
             FROM pages 
             WHERE is_legal_page = 1 
             ORDER BY 
                CASE legal_page_type
                    WHEN 'impressum' THEN 1
                    WHEN 'datenschutz' THEN 2
                    WHEN 'agb' THEN 3
                    WHEN 'cookies' THEN 4
                    ELSE 5
                END"
        );
    }
    
    /**
     * Get legal page by type
     */
    public static function getByType($type) {
        return db()->fetchOne(
            "SELECT id, slug, title, COALESCE(page_icon, '📄') AS page_icon, meta_description, status, 
                    show_in_menu, show_in_footer, is_legal_page, legal_page_type, is_deletable, 
                    menu_order, page_type, uses_flexible_blocks, parent_id 
             FROM pages 
             WHERE legal_page_type = ? 
             LIMIT 1",
            [$type]
        );
    }
    
    /**
     * Ensure legal page exists
     */
    public static function ensureLegalPageExists($type) {
        if (!isset(self::LEGAL_PAGE_TYPES[$type])) {
            return false;
        }
        
        $existing = self::getByType($type);
        if ($existing) {
            return $existing['id'];
        }
        
        $config = self::LEGAL_PAGE_TYPES[$type];
        
        $pageId = db()->insert(
            "INSERT INTO pages (
                slug, title, page_icon, meta_description, status,
                show_in_menu, menu_order, is_deletable, is_system_page,
                is_legal_page, legal_page_type, show_in_footer,
                page_type, uses_flexible_blocks
            ) VALUES (?, ?, ?, ?, 'draft', 0, 999, 0, 1, 1, ?, 1, 'flexible', 1)",
            [
                $type,
                $config['title'],
                $config['icon'],
                $config['description'],
                $type
            ]
        );
        
        return $pageId;
    }
    
    /**
     * Update status
     */
    public static function updateStatus($pageId, $status) {
        if (!in_array($status, ['draft', 'published'])) {
            return false;
        }
        
        return db()->execute(
            "UPDATE pages SET status = ? WHERE id = ? AND is_legal_page = 1",
            [$status, $pageId]
        );
    }
    
    /**
     * Update footer visibility
     */
    public static function updateFooterVisibility($pageId, $showInFooter) {
        return db()->execute(
            "UPDATE pages SET show_in_footer = ? WHERE id = ? AND is_legal_page = 1",
            [$showInFooter ? 1 : 0, $pageId]
        );
    }
    
    /**
     * Check if all required pages are published
     */
    public static function checkRequiredPages() {
        $missing = [];
        
        foreach (self::LEGAL_PAGE_TYPES as $type => $config) {
            if (!$config['required']) {
                continue;
            }
            
            $page = self::getByType($type);
            
            if (!$page) {
                $missing[] = $config['title'] . ' (nicht erstellt)';
            } elseif ($page['status'] !== 'published') {
                $missing[] = $config['title'] . ' (nicht veröffentlicht)';
            }
        }
        
        return [
            'complete' => empty($missing),
            'missing' => $missing
        ];
    }
    
    /**
     * Check if page can be deleted
     */
    public static function canDelete($pageId) {
        $page = db()->fetchOne(
            "SELECT is_legal_page, is_system_page, slug FROM pages WHERE id = ?",
            [$pageId]
        );
        
        if (!$page) {
            return true;
        }
        
        // Protect index page
        if ($page['slug'] === 'index') {
            return false;
        }
        
        return !($page['is_legal_page'] == 1 || $page['is_system_page'] == 1);
    }
    
    /**
     * Get footer pages (published only)
     */
    public static function getFooterPages() {
        return db()->fetchAll(
            "SELECT slug, title, COALESCE(page_icon, '📄') AS page_icon, legal_page_type 
             FROM pages 
             WHERE is_legal_page = 1 
             AND show_in_footer = 1 
             AND status = 'published'
             ORDER BY 
                CASE legal_page_type
                    WHEN 'impressum' THEN 1
                    WHEN 'datenschutz' THEN 2
                    WHEN 'agb' THEN 3
                    WHEN 'cookies' THEN 4
                    ELSE 5
                END"
        );
    }
    
    /**
     * Initialize all legal pages
     */
    public static function initializeAll() {
        $created = [];
        
        foreach (self::LEGAL_PAGE_TYPES as $type => $config) {
            $pageId = self::ensureLegalPageExists($type);
            if ($pageId) {
                $created[$type] = $pageId;
            }
        }
        
        return $created;
    }
}