<?php
/**
 * cms-core/backend/core/Mailer.php
 * Professional Email System with SMTP Support
 * Uses .env configuration for email settings
 */

class Mailer {
    
    private $to = [];
    private $from = '';
    private $fromName = '';
    private $subject = '';
    private $body = '';
    private $headers = [];
    private $attachments = [];
    private $isHTML = true;
    
    /**
     * Initialize with default from settings
     */
    public function __construct() {
        $this->from = MAIL_FROM;
        $this->fromName = MAIL_FROM_NAME;
    }
    
    /**
     * Set recipient(s)
     */
    public function to($email, $name = null) {
        if (is_array($email)) {
            $this->to = array_merge($this->to, $email);
        } else {
            $this->to[] = ['email' => $email, 'name' => $name];
        }
        return $this;
    }
    
    /**
     * Set sender
     */
    public function from($email, $name = null) {
        $this->from = $email;
        if ($name) {
            $this->fromName = $name;
        }
        return $this;
    }
    
    /**
     * Set subject
     */
    public function subject($subject) {
        $this->subject = $subject;
        return $this;
    }
    
    /**
     * Set body content
     */
    public function body($body) {
        $this->body = $body;
        return $this;
    }
    
    /**
     * Set if email is HTML
     */
    public function html($isHTML = true) {
        $this->isHTML = $isHTML;
        return $this;
    }
    
    /**
     * Add attachment
     */
    public function attach($filepath, $filename = null) {
        if (file_exists($filepath)) {
            $this->attachments[] = [
                'path' => $filepath,
                'name' => $filename ?? basename($filepath)
            ];
        }
        return $this;
    }
    
    /**
     * Send email using SMTP or PHP mail()
     */
    public function send() {
        try {
            // Validate recipients
            if (empty($this->to)) {
                throw new Exception('Keine Empfänger angegeben');
            }
            
            // Validate subject and body
            if (empty($this->subject)) {
                throw new Exception('Kein Betreff angegeben');
            }
            
            if (empty($this->body)) {
                throw new Exception('Kein Inhalt angegeben');
            }
            
            // Use SMTP if configured, otherwise fallback to PHP mail()
            if (!empty(MAIL_SMTP_HOST) && !empty(MAIL_SMTP_USER)) {
                return $this->sendViaSMTP();
            } else {
                return $this->sendViaPHPMail();
            }
            
        } catch (Exception $e) {
            Logger::error('Email send failed: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Send via SMTP
     */
    private function sendViaSMTP() {
        $smtp = new SMTPClient(
            MAIL_SMTP_HOST,
            MAIL_SMTP_PORT,
            MAIL_SMTP_USER,
            MAIL_SMTP_PASS
        );
        
        $smtp->setFrom($this->from, $this->fromName);
        
        foreach ($this->to as $recipient) {
            $smtp->addRecipient($recipient['email'], $recipient['name']);
        }
        
        $smtp->setSubject($this->subject);
        $smtp->setBody($this->body, $this->isHTML);
        
        foreach ($this->attachments as $attachment) {
            $smtp->addAttachment($attachment['path'], $attachment['name']);
        }
        
        return $smtp->send();
    }
    
    /**
     * Send via PHP mail() function (fallback)
     */
    private function sendViaPHPMail() {
        $boundary = md5(time());
        
        // Build headers
        $headers = [];
        
        if ($this->fromName) {
            $headers[] = "From: {$this->fromName} <{$this->from}>";
        } else {
            $headers[] = "From: {$this->from}";
        }
        
        $headers[] = "Reply-To: {$this->from}";
        $headers[] = "MIME-Version: 1.0";
        $headers[] = "X-Mailer: PHP/" . phpversion();
        
        // Handle attachments
        if (!empty($this->attachments)) {
            $headers[] = "Content-Type: multipart/mixed; boundary=\"{$boundary}\"";
        } else if ($this->isHTML) {
            $headers[] = "Content-Type: text/html; charset=UTF-8";
        } else {
            $headers[] = "Content-Type: text/plain; charset=UTF-8";
        }
        
        // Build message body
        $message = $this->buildMessage($boundary);
        
        // Send to each recipient
        $success = true;
        foreach ($this->to as $recipient) {
            $toAddress = $recipient['name'] 
                ? "{$recipient['name']} <{$recipient['email']}>" 
                : $recipient['email'];
            
            if (!mail($toAddress, $this->subject, $message, implode("\r\n", $headers))) {
                $success = false;
                Logger::error("Failed to send email to: {$recipient['email']}");
            }
        }
        
        return $success;
    }
    
    /**
     * Build email message with optional attachments
     */
    private function buildMessage($boundary) {
        if (empty($this->attachments)) {
            return $this->body;
        }
        
        $message = "--{$boundary}\r\n";
        
        if ($this->isHTML) {
            $message .= "Content-Type: text/html; charset=UTF-8\r\n";
        } else {
            $message .= "Content-Type: text/plain; charset=UTF-8\r\n";
        }
        
        $message .= "Content-Transfer-Encoding: 8bit\r\n\r\n";
        $message .= $this->body . "\r\n\r\n";
        
        // Add attachments
        foreach ($this->attachments as $attachment) {
            $content = chunk_split(base64_encode(file_get_contents($attachment['path'])));
            $message .= "--{$boundary}\r\n";
            $message .= "Content-Type: application/octet-stream; name=\"{$attachment['name']}\"\r\n";
            $message .= "Content-Transfer-Encoding: base64\r\n";
            $message .= "Content-Disposition: attachment; filename=\"{$attachment['name']}\"\r\n\r\n";
            $message .= $content . "\r\n";
        }
        
        $message .= "--{$boundary}--";
        
        return $message;
    }
    
    /**
     * Static helper: Send simple email quickly
     */
    public static function quick($to, $subject, $body, $isHTML = true) {
        $mailer = new self();
        return $mailer->to($to)
                     ->subject($subject)
                     ->body($body)
                     ->html($isHTML)
                     ->send();
    }
    
    /**
     * Static helper: Send contact form email
     */
    public static function sendContactForm($data) {
        $mailer = new self();
        
        // Email to site owner
        $mailer->to(get_setting('contact_email', MAIL_FROM));
        $mailer->subject('Neue Kontaktanfrage von ' . $data['name']);
        
        $body = self::getContactFormTemplate($data);
        
        $mailer->body($body);
        $mailer->html(true);
        
        return $mailer->send();
    }
    
    /**
     * Get contact form email template
     */
    private static function getContactFormTemplate($data) {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #667eea; color: white; padding: 20px; border-radius: 8px 8px 0 0; }
                .content { background: #f9fafb; padding: 20px; border: 1px solid #e5e7eb; }
                .field { margin-bottom: 15px; }
                .label { font-weight: bold; color: #4b5563; }
                .value { margin-top: 5px; padding: 10px; background: white; border-radius: 4px; }
                .footer { margin-top: 20px; padding: 15px; background: #f3f4f6; border-radius: 0 0 8px 8px; font-size: 12px; color: #6b7280; text-align: center; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h2 style='margin: 0;'>📧 Neue Kontaktanfrage</h2>
                </div>
                <div class='content'>
                    <div class='field'>
                        <div class='label'>Name:</div>
                        <div class='value'>" . htmlspecialchars($data['name']) . "</div>
                    </div>
                    <div class='field'>
                        <div class='label'>E-Mail:</div>
                        <div class='value'><a href='mailto:" . htmlspecialchars($data['email']) . "'>" . htmlspecialchars($data['email']) . "</a></div>
                    </div>
                    " . (!empty($data['phone']) ? "
                    <div class='field'>
                        <div class='label'>Telefon:</div>
                        <div class='value'>" . htmlspecialchars($data['phone']) . "</div>
                    </div>
                    " : "") . "
                    <div class='field'>
                        <div class='label'>Nachricht:</div>
                        <div class='value'>" . nl2br(htmlspecialchars($data['message'])) . "</div>
                    </div>
                </div>
                <div class='footer'>
                    Gesendet am " . date('d.m.Y H:i') . " Uhr über das Kontaktformular
                </div>
            </div>
        </body>
        </html>
        ";
    }
}

/**
 * Simple SMTP Client
 */
class SMTPClient {
    
    private $host;
    private $port;
    private $username;
    private $password;
    private $socket;
    private $from = '';
    private $fromName = '';
    private $recipients = [];
    private $subject = '';
    private $body = '';
    private $isHTML = true;
    private $attachments = [];
    
    public function __construct($host, $port, $username, $password) {
        $this->host = $host;
        $this->port = $port;
        $this->username = $username;
        $this->password = $password;
    }
    
    public function setFrom($email, $name = '') {
        $this->from = $email;
        $this->fromName = $name;
    }
    
    public function addRecipient($email, $name = '') {
        $this->recipients[] = ['email' => $email, 'name' => $name];
    }
    
    public function setSubject($subject) {
        $this->subject = $subject;
    }
    
    public function setBody($body, $isHTML = true) {
        $this->body = $body;
        $this->isHTML = $isHTML;
    }
    
    public function addAttachment($filepath, $filename) {
        $this->attachments[] = ['path' => $filepath, 'name' => $filename];
    }
    
    public function send() {
        try {
            $this->connect();
            $this->authenticate();
            $this->sendMessage();
            $this->disconnect();
            return true;
        } catch (Exception $e) {
            Logger::error('SMTP Error: ' . $e->getMessage());
            return false;
        }
    }
    
    private function connect() {
        $this->socket = fsockopen($this->host, $this->port, $errno, $errstr, 30);
        
        if (!$this->socket) {
            throw new Exception("SMTP Connection failed: {$errstr} ({$errno})");
        }
        
        $this->getResponse();
        $this->sendCommand("EHLO {$this->host}");
    }
    
    private function authenticate() {
        $this->sendCommand("AUTH LOGIN");
        $this->sendCommand(base64_encode($this->username));
        $this->sendCommand(base64_encode($this->password));
    }
    
    private function sendMessage() {
        // MAIL FROM
        $this->sendCommand("MAIL FROM:<{$this->from}>");
        
        // RCPT TO
        foreach ($this->recipients as $recipient) {
            $this->sendCommand("RCPT TO:<{$recipient['email']}>");
        }
        
        // DATA
        $this->sendCommand("DATA");
        
        // Headers and body
        $message = $this->buildEmailMessage();
        $this->sendCommand($message . "\r\n.");
    }
    
    private function buildEmailMessage() {
        $boundary = md5(time());
        $message = [];
        
        // Headers
        $message[] = "From: {$this->fromName} <{$this->from}>";
        $message[] = "Subject: {$this->subject}";
        $message[] = "MIME-Version: 1.0";
        
        if (!empty($this->attachments)) {
            $message[] = "Content-Type: multipart/mixed; boundary=\"{$boundary}\"";
        } else if ($this->isHTML) {
            $message[] = "Content-Type: text/html; charset=UTF-8";
        } else {
            $message[] = "Content-Type: text/plain; charset=UTF-8";
        }
        
        $message[] = "";
        
        // Body
        if (!empty($this->attachments)) {
            $message[] = "--{$boundary}";
            $message[] = $this->isHTML ? "Content-Type: text/html; charset=UTF-8" : "Content-Type: text/plain; charset=UTF-8";
            $message[] = "";
        }
        
        $message[] = $this->body;
        
        // Attachments
        foreach ($this->attachments as $attachment) {
            $content = chunk_split(base64_encode(file_get_contents($attachment['path'])));
            $message[] = "";
            $message[] = "--{$boundary}";
            $message[] = "Content-Type: application/octet-stream; name=\"{$attachment['name']}\"";
            $message[] = "Content-Transfer-Encoding: base64";
            $message[] = "Content-Disposition: attachment; filename=\"{$attachment['name']}\"";
            $message[] = "";
            $message[] = $content;
        }
        
        if (!empty($this->attachments)) {
            $message[] = "--{$boundary}--";
        }
        
        return implode("\r\n", $message);
    }
    
    private function disconnect() {
        $this->sendCommand("QUIT");
        fclose($this->socket);
    }
    
    private function sendCommand($command) {
        fwrite($this->socket, $command . "\r\n");
        return $this->getResponse();
    }
    
    private function getResponse() {
        $response = '';
        while ($line = fgets($this->socket, 515)) {
            $response .= $line;
            if (substr($line, 3, 1) == ' ') {
                break;
            }
        }
        return $response;
    }
}