<?php
/**
 * cms-core/backend/core/MenuManager.php - WITH PERSISTENTCACHE
 * Manages navigation menu and page hierarchy
 * FIXED: Adds page URLs to menu items + Persistent Caching
 */

class MenuManager {
    
    /**
     * Get menu pages with hierarchy (for frontend navigation) + PERSISTENT CACHE
     * Returns published pages with show_in_menu = 1
     * Includes proper URLs for each page
     * Used by: Header, Sidebar, any hierarchical menu
     */
    public static function getMenuPagesWithHierarchy() {
        // Use PersistentCache if available, fallback to QueryCache
        if (class_exists('PersistentCache')) {
            return PersistentCache::remember('menu_pages_hierarchy', 3600, function() {
                try {
                    $sql = "SELECT 
                                id, title, slug, parent_id, menu_order, status,
                                show_in_menu, show_in_footer, menu_label,
                                COALESCE(page_icon, '📄') AS page_icon,
                                is_legal_page, is_system_page
                            FROM pages 
                            WHERE show_in_menu = 1 
                            AND status = 'published'
                            ORDER BY parent_id ASC, menu_order ASC, title ASC";
                    
                    $pages = db()->fetchAll($sql);
                    
                    // Add URLs to each page
                    foreach ($pages as &$page) {
                        $page['url'] = self::getPageUrl($page['slug']);
                    }
                    
                    // Build hierarchy
                    return self::buildHierarchy($pages);
                    
                } catch (Exception $e) {
                    error_log("MenuManager Error: " . $e->getMessage());
                    return [];
                }
            });
        }
        
        // Fallback to QueryCache
        $cacheKey = 'menu_pages_hierarchy';
        $cached = QueryCache::get($cacheKey);
        
        if ($cached !== null) {
            return $cached;
        }
        
        try {
            $sql = "SELECT 
                        id, title, slug, parent_id, menu_order, status,
                        show_in_menu, show_in_footer, menu_label,
                        COALESCE(page_icon, '📄') AS page_icon,
                        is_legal_page, is_system_page
                    FROM pages 
                    WHERE show_in_menu = 1 
                    AND status = 'published'
                    ORDER BY parent_id ASC, menu_order ASC, title ASC";
            
            $pages = db()->fetchAll($sql);
            
            // Add URLs to each page
            foreach ($pages as &$page) {
                $page['url'] = self::getPageUrl($page['slug']);
            }
            
            // Build hierarchy
            $result = self::buildHierarchy($pages);
            
            QueryCache::set($cacheKey, $result);
            return $result;
            
        } catch (Exception $e) {
            error_log("MenuManager Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get header pages (uses show_in_menu, respects hierarchy)
     * For header navigation
     * IMPORTANT: Header = Menu! No separate show_in_header column needed!
     */
    public static function getHeaderPages() {
        // Header uses the same as menu - they are the same thing!
        return self::getMenuPagesWithHierarchy();
    }
    
    /**
     * Get footer pages (uses show_in_footer, flat list) + PERSISTENT CACHE
     * For footer navigation
     */
    public static function getFooterPages() {
        // Use PersistentCache if available, fallback to QueryCache
        if (class_exists('PersistentCache')) {
            return PersistentCache::remember('footer_pages', 3600, function() {
                try {
                    $sql = "SELECT 
                                id, title, slug, menu_label,
                                COALESCE(page_icon, '📄') AS page_icon
                            FROM pages 
                            WHERE show_in_footer = 1 
                            AND status = 'published'
                            ORDER BY title ASC";
                    
                    $pages = db()->fetchAll($sql);
                    
                    // Add URLs to each page
                    foreach ($pages as &$page) {
                        $page['url'] = self::getPageUrl($page['slug']);
                    }
                    
                    return $pages;
                    
                } catch (Exception $e) {
                    error_log("MenuManager Error: " . $e->getMessage());
                    return [];
                }
            });
        }
        
        // Fallback to QueryCache
        $cacheKey = 'footer_pages';
        $cached = QueryCache::get($cacheKey);
        
        if ($cached !== null) {
            return $cached;
        }
        
        try {
            $sql = "SELECT 
                        id, title, slug, menu_label,
                        COALESCE(page_icon, '📄') AS page_icon
                    FROM pages 
                    WHERE show_in_footer = 1 
                    AND status = 'published'
                    ORDER BY title ASC";
            
            $pages = db()->fetchAll($sql);
            
            // Add URLs to each page
            foreach ($pages as &$page) {
                $page['url'] = self::getPageUrl($page['slug']);
            }
            
            QueryCache::set($cacheKey, $pages);
            return $pages;
            
        } catch (Exception $e) {
            error_log("MenuManager Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Generate page URL based on slug
     * Handles special cases (homepage, index)
     */
    private static function getPageUrl($slug) {
        // Homepage: / instead of /index
        if ($slug === 'index' || $slug === 'home' || empty($slug)) {
            return base_url() . '/';
        }
        
        // Regular pages: /slug
        return base_url() . '/' . $slug;
    }
    
    /**
     * Build hierarchical array from flat page list
     */
    private static function buildHierarchy($pages, $parentId = null) {
        $branch = [];
        
        foreach ($pages as $page) {
            $pageParentId = $page['parent_id'] ?? null;
            
            // Match parent (null matches null, int matches int)
            if ($pageParentId === $parentId || 
                (is_null($parentId) && is_null($pageParentId))) {
                
                // Find children
                $children = self::buildHierarchy($pages, $page['id']);
                if (!empty($children)) {
                    $page['children'] = $children;
                }
                
                $branch[] = $page;
            }
        }
        
        // Sort by menu_order
        usort($branch, function($a, $b) {
            return ($a['menu_order'] ?? 0) <=> ($b['menu_order'] ?? 0);
        });
        
        return $branch;
    }
    
    /**
     * Get menu pages as flat list (no hierarchy)
     */
    public static function getMenuPages() {
        try {
            $sql = "SELECT 
                        id, title, slug, parent_id, menu_order, status,
                        show_in_menu, show_in_header, menu_label,
                        COALESCE(page_icon, '📄') AS page_icon
                    FROM pages 
                    WHERE show_in_menu = 1 
                    AND status = 'published'
                    ORDER BY menu_order ASC, title ASC";
            
            $pages = db()->fetchAll($sql);
            
            // Add URLs
            foreach ($pages as &$page) {
                $page['url'] = self::getPageUrl($page['slug']);
            }
            
            return $pages;
            
        } catch (Exception $e) {
            error_log("MenuManager Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Toggle menu visibility
     */
    public static function toggleMenuVisibility($pageId, $showInMenu) {
        try {
            db()->execute(
                "UPDATE pages SET show_in_menu = ? WHERE id = ?",
                [(int)$showInMenu, (int)$pageId]
            );
            
            // If disabling parent, disable children too
            if ($showInMenu == 0) {
                db()->execute(
                    "UPDATE pages SET show_in_menu = 0 WHERE parent_id = ?",
                    [(int)$pageId]
                );
            }
            
            // Clear cache
            self::clearCache();
            
            return true;
        } catch (Exception $e) {
            error_log("MenuManager Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Toggle header visibility
     */
    public static function toggleHeaderVisibility($pageId, $showInHeader) {
        try {
            db()->execute(
                "UPDATE pages SET show_in_header = ? WHERE id = ?",
                [(int)$showInHeader, (int)$pageId]
            );
            
            // Clear cache
            self::clearCache();
            
            return true;
        } catch (Exception $e) {
            error_log("MenuManager Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get breadcrumbs for a page
     */
    public static function getBreadcrumbs($pageId) {
        $breadcrumbs = [];
        $currentId = $pageId;
        
        try {
            while ($currentId !== null) {
                $page = db()->fetchOne(
                    "SELECT id, title, slug, parent_id, COALESCE(page_icon, '📄') AS page_icon 
                     FROM pages 
                     WHERE id = ?",
                    [$currentId]
                );
                
                if (!$page) break;
                
                $page['url'] = self::getPageUrl($page['slug']);
                array_unshift($breadcrumbs, $page);
                $currentId = $page['parent_id'];
            }
            
            return $breadcrumbs;
            
        } catch (Exception $e) {
            error_log("MenuManager Breadcrumbs Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get child pages of a parent
     */
    public static function getChildPages($parentId = null, $menuOnly = true) {
        try {
            $sql = "SELECT 
                        id, title, slug, parent_id, menu_order, status,
                        show_in_menu, show_in_header, menu_label,
                        COALESCE(page_icon, '📄') AS page_icon
                    FROM pages 
                    WHERE status = 'published'";
            
            $params = [];
            
            if ($parentId === null) {
                $sql .= " AND parent_id IS NULL";
            } else {
                $sql .= " AND parent_id = ?";
                $params[] = $parentId;
            }
            
            if ($menuOnly) {
                $sql .= " AND show_in_menu = 1";
            }
            
            $sql .= " ORDER BY menu_order ASC, title ASC";
            
            $pages = db()->fetchAll($sql, $params);
            
            // Add URLs
            foreach ($pages as &$page) {
                $page['url'] = self::getPageUrl($page['slug']);
            }
            
            return $pages;
            
        } catch (Exception $e) {
            error_log("MenuManager Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Check if page has children
     */
    public static function hasChildren($pageId) {
        try {
            $result = db()->fetchOne(
                "SELECT COUNT(*) as count FROM pages WHERE parent_id = ? AND status = 'published'",
                [$pageId]
            );
            
            return $result['count'] > 0;
            
        } catch (Exception $e) {
            error_log("MenuManager Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get active page based on current URL
     */
    public static function getActivePage($slug) {
        try {
            $page = db()->fetchOne(
                "SELECT *, COALESCE(page_icon, '📄') AS page_icon 
                 FROM pages 
                 WHERE slug = ? AND status = 'published'",
                [$slug]
            );
            
            if ($page) {
                $page['url'] = self::getPageUrl($page['slug']);
            }
            
            return $page;
        } catch (Exception $e) {
            error_log("MenuManager Error: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Reorder menu items
     */
    public static function reorderMenu($pageIds, $parentId = null) {
        try {
            foreach ($pageIds as $order => $pageId) {
                db()->execute(
                    "UPDATE pages SET menu_order = ?, parent_id = ? WHERE id = ?",
                    [$order, $parentId, $pageId]
                );
            }
            
            // Clear cache
            self::clearCache();
            
            return true;
        } catch (Exception $e) {
            error_log("MenuManager Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Set page parent (for dropdown menus)
     */
    public static function setPageParent($pageId, $parentId) {
        try {
            db()->execute(
                "UPDATE pages SET parent_id = ? WHERE id = ?",
                [$parentId, (int)$pageId]
            );
            
            // Clear cache
            self::clearCache();
            
            return true;
        } catch (Exception $e) {
            error_log("MenuManager Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Clear menu cache
     */
    public static function clearCache() {
        // Clear PersistentCache if available
        if (class_exists('PersistentCache')) {
            PersistentCache::delete('menu_pages_hierarchy');
            PersistentCache::delete('footer_pages');
        }
        
        // Also clear QueryCache
        QueryCache::delete('menu_pages_hierarchy');
        QueryCache::delete('footer_pages');
    }
}