<?php
/** cms-core/backend/core/PageManager.php - WITH PERSISTENTCACHE + TRY-CATCH */

/**
 * Page Manager
 * Manages all page CRUD operations
 * ENHANCED: Added try-catch error handling to all methods
 */

class PageManager {
    
    /**
     * Get all pages (ADMIN ONLY - includes drafts) + PERSISTENT CACHE
     */
    public static function getAllPages() {
        try {
            // Use PersistentCache if available, fallback to QueryCache
            if (class_exists('PersistentCache')) {
                return PersistentCache::remember('all_pages', 3600, function() {
                    return db()->fetchAll(
                        "SELECT id, slug, title, COALESCE(page_icon, '📄') AS page_icon, meta_description, status, 
                                show_in_menu, show_in_footer, is_legal_page, is_system_page, 
                                is_deletable, menu_label, menu_order, page_type, uses_flexible_blocks, parent_id 
                         FROM pages 
                         ORDER BY menu_order ASC, id ASC"
                    );
                });
            }
            
            // Fallback to QueryCache
            $cacheKey = 'all_pages';
            $cached = QueryCache::get($cacheKey);
            
            if ($cached !== null) {
                return $cached;
            }
            
            $result = db()->fetchAll(
                "SELECT id, slug, title, COALESCE(page_icon, '📄') AS page_icon, meta_description, status, 
                        show_in_menu, show_in_footer, is_legal_page, is_system_page, 
                        is_deletable, menu_label, menu_order, page_type, uses_flexible_blocks, parent_id 
                 FROM pages 
                 ORDER BY menu_order ASC, id ASC"
            );
            
            QueryCache::set($cacheKey, $result);
            return $result;
        } catch (Exception $e) {
            error_log("PageManager::getAllPages Error: " . $e->getMessage());
            if (class_exists('Logger')) {
                Logger::error('Failed to fetch all pages', ['error' => $e->getMessage()]);
            }
            return [];
        }
    }
    
    /**
     * Get only PUBLISHED pages (FRONTEND) + PERSISTENT CACHE
     */
    public static function getPublishedPages() {
        try {
            // Use PersistentCache if available, fallback to QueryCache
            if (class_exists('PersistentCache')) {
                return PersistentCache::remember('published_pages', 3600, function() {
                    return db()->fetchAll(
                        "SELECT id, slug, title, COALESCE(page_icon, '📄') AS page_icon, meta_description, status, 
                                show_in_menu, show_in_footer, is_legal_page, is_system_page, 
                                is_deletable, menu_label, menu_order, page_type, uses_flexible_blocks, parent_id 
                         FROM pages 
                         WHERE status = 'published' 
                         ORDER BY menu_order ASC, id ASC"
                    );
                });
            }
            
            // Fallback to QueryCache
            $cacheKey = 'published_pages';
            $cached = QueryCache::get($cacheKey);
            
            if ($cached !== null) {
                return $cached;
            }
            
            $result = db()->fetchAll(
                "SELECT id, slug, title, COALESCE(page_icon, '📄') AS page_icon, meta_description, status, 
                        show_in_menu, show_in_footer, is_legal_page, is_system_page, 
                        is_deletable, menu_label, menu_order, page_type, uses_flexible_blocks, parent_id 
                 FROM pages 
                 WHERE status = 'published' 
                 ORDER BY menu_order ASC, id ASC"
            );
            
            QueryCache::set($cacheKey, $result);
            return $result;
        } catch (Exception $e) {
            error_log("PageManager::getPublishedPages Error: " . $e->getMessage());
            if (class_exists('Logger')) {
                Logger::error('Failed to fetch published pages', ['error' => $e->getMessage()]);
            }
            return [];
        }
    }
    
    /**
     * Get page by slug + PERSISTENT CACHE
     */
    public static function getBySlug($slug) {
        try {
            // Use PersistentCache if available, fallback to QueryCache
            if (class_exists('PersistentCache')) {
                return PersistentCache::remember('page_by_slug_' . $slug, 3600, function() use ($slug) {
                    return db()->fetchOne(
                        "SELECT id, slug, title, COALESCE(page_icon, '📄') AS page_icon, meta_description, status, 
                                show_in_menu, show_in_footer, is_legal_page, is_system_page, 
                                is_deletable, menu_label, menu_order, page_type, uses_flexible_blocks, parent_id, content 
                         FROM pages 
                         WHERE slug = ?", 
                        [$slug]
                    );
                });
            }
            
            // Fallback to QueryCache
            $cacheKey = 'page_by_slug_' . $slug;
            $cached = QueryCache::get($cacheKey);
            
            if ($cached !== null) {
                return $cached;
            }
            
            $result = db()->fetchOne(
                "SELECT id, slug, title, COALESCE(page_icon, '📄') AS page_icon, meta_description, status, 
                        show_in_menu, show_in_footer, is_legal_page, is_system_page, 
                        is_deletable, menu_label, menu_order, page_type, uses_flexible_blocks, parent_id, content 
                 FROM pages 
                 WHERE slug = ?", 
                [$slug]
            );
            
            QueryCache::set($cacheKey, $result);
            return $result;
        } catch (Exception $e) {
            error_log("PageManager::getBySlug Error: " . $e->getMessage());
            if (class_exists('Logger')) {
                Logger::error('Failed to fetch page by slug', [
                    'slug' => $slug,
                    'error' => $e->getMessage()
                ]);
            }
            return null;
        }
    }
    
    /**
     * Get pages shown in menu (PUBLISHED ONLY)
     */
    public static function getMenuPages() {
        try {
            return db()->fetchAll(
                "SELECT id, slug, title, COALESCE(page_icon, '📄') AS page_icon, meta_description, status, 
                        show_in_menu, show_in_footer, is_legal_page, is_system_page, 
                        is_deletable, menu_label, menu_order, page_type, uses_flexible_blocks, parent_id 
                 FROM pages 
                 WHERE show_in_menu = 1 
                 AND status = 'published' 
                 ORDER BY menu_order ASC"
            );
        } catch (Exception $e) {
            error_log("PageManager::getMenuPages Error: " . $e->getMessage());
            if (class_exists('Logger')) {
                Logger::error('Failed to fetch menu pages', ['error' => $e->getMessage()]);
            }
            return [];
        }
    }
    
    /**
     * Get all pages with hierarchy (for dropdown menus)
     * FRONTEND: Only published pages
     */
    public static function getAllPagesWithHierarchy() {
        try {
            $pages = self::getPublishedPages();
            return self::buildHierarchy($pages);
        } catch (Exception $e) {
            error_log("PageManager::getAllPagesWithHierarchy Error: " . $e->getMessage());
            if (class_exists('Logger')) {
                Logger::error('Failed to build page hierarchy', ['error' => $e->getMessage()]);
            }
            return [];
        }
    }
    
    /**
     * Build hierarchical tree from flat array
     */
    private static function buildHierarchy($pages, $parentId = null) {
        try {
            $branch = [];
            
            foreach ($pages as $page) {
                if (!isset($page['show_in_menu']) || $page['show_in_menu'] != 1) {
                    continue; // Skip pages not in menu
                }
                
                // Match parent ID
                $pageParentId = $page['parent_id'] ?? null;
                if ($pageParentId === $parentId || (is_null($parentId) && is_null($pageParentId))) {
                    // Find children
                    $children = self::buildHierarchy($pages, $page['id']);
                    if (!empty($children)) {
                        $page['children'] = $children;
                    }
                    $branch[] = $page;
                }
            }
            
            // Sort by menu_order
            usort($branch, function($a, $b) {
                return ($a['menu_order'] ?? 0) <=> ($b['menu_order'] ?? 0);
            });
            
            return $branch;
        } catch (Exception $e) {
            error_log("PageManager::buildHierarchy Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Create new page
     * NO FILE CREATION - everything handled by page.php router
     */
    public static function createPage($data) {
        try {
            // Get highest menu order
            $maxOrder = db()->fetchOne("SELECT MAX(menu_order) as max_order FROM pages");
            $menuOrder = ($maxOrder['max_order'] ?? 0) + 1;
            
            // Prepare data
            $slug = $data['slug'];
            $title = $data['title'];
            $pageIcon = $data['page_icon'] ?? '📄';
            $metaDescription = $data['meta_description'] ?? '';
            $pageType = $data['page_type'] ?? 'flexible';
            $showInMenu = isset($data['show_in_menu']) ? 1 : 0;
            $menuLabel = $data['menu_label'] ?? null;
            $parentId = $data['parent_id'] ?? null;
            $usesFlexibleBlocks = 1; // Always use flexible blocks
            
            // Insert page
            $pageId = db()->insert(
                "INSERT INTO pages (
                    slug, title, page_icon, meta_description, status, 
                    show_in_menu, menu_label, menu_order, parent_id, is_deletable, 
                    page_type, uses_flexible_blocks
                ) VALUES (?, ?, ?, ?, 'draft', ?, ?, ?, ?, 1, ?, ?)",
                [
                    $slug, $title, $pageIcon, $metaDescription,
                    $showInMenu, $menuLabel, $menuOrder, $parentId, $pageType, $usesFlexibleBlocks
                ]
            );
            
            if ($pageId) {
                // Clear cache
                self::clearCache();
                
                if (class_exists('Logger')) {
                    Logger::info("Page created: {$slug} (ID: {$pageId})");
                }
            }
            
            return $pageId;
        } catch (Exception $e) {
            error_log("PageManager::createPage Error: " . $e->getMessage());
            if (class_exists('Logger')) {
                Logger::error('Failed to create page', [
                    'data' => $data,
                    'error' => $e->getMessage()
                ]);
            }
            return false;
        }
    }
    
    /**
     * Update page settings
     */
    public static function updatePageSettings($slug, $data) {
        try {
            $updates = [];
            $params = [];
            
            if (isset($data['title'])) {
                $updates[] = "title = ?";
                $params[] = $data['title'];
            }
            
            if (isset($data['page_icon'])) {
                $updates[] = "page_icon = ?";
                $params[] = $data['page_icon'];
            }
            
            if (isset($data['meta_description'])) {
                $updates[] = "meta_description = ?";
                $params[] = $data['meta_description'];
            }
            
            if (isset($data['show_in_menu'])) {
                $updates[] = "show_in_menu = ?";
                $params[] = $data['show_in_menu'] ? 1 : 0;
            }
            
            if (isset($data['menu_label'])) {
                $updates[] = "menu_label = ?";
                $params[] = $data['menu_label'] ?: null;
            }
            
            if (isset($data['status'])) {
                $updates[] = "status = ?";
                $params[] = $data['status'];
            }
            
            if (isset($data['parent_id'])) {
                $updates[] = "parent_id = ?";
                $params[] = $data['parent_id'] ?: null;
            }
            
            if (empty($updates)) {
                return false;
            }
            
            $params[] = $slug;
            $sql = "UPDATE pages SET " . implode(', ', $updates) . " WHERE slug = ?";
            
            $result = db()->execute($sql, $params);
            
            if ($result) {
                // Clear cache
                self::clearCache();
                
                if (class_exists('Logger')) {
                    Logger::info("Page updated: {$slug}");
                }
            }
            
            return $result;
        } catch (Exception $e) {
            error_log("PageManager::updatePageSettings Error: " . $e->getMessage());
            if (class_exists('Logger')) {
                Logger::error('Failed to update page settings', [
                    'slug' => $slug,
                    'error' => $e->getMessage()
                ]);
            }
            return false;
        }
    }
    
    /**
     * Delete page
     * PROTECTED: Cannot delete system pages, legal pages, or index
     */
    public static function deletePage($pageId) {
        try {
            $page = db()->fetchOne("SELECT * FROM pages WHERE id = ?", [$pageId]);
            
            if (!$page) {
                return ['success' => false, 'error' => 'Seite nicht gefunden.'];
            }
            
            // SECURITY: Protect system pages
            if ($page['slug'] === 'index') {
                return ['success' => false, 'error' => 'Die Startseite kann nicht gelöscht werden.'];
            }
            
            if (isset($page['is_system_page']) && $page['is_system_page'] == 1) {
                return ['success' => false, 'error' => 'Diese System-Seite kann nicht gelöscht werden.'];
            }
            
            if (isset($page['is_legal_page']) && $page['is_legal_page'] == 1) {
                return ['success' => false, 'error' => 'Rechtliche Pflichtseiten können nicht gelöscht werden.'];
            }
            
            if ($page['is_deletable'] == 0) {
                return ['success' => false, 'error' => 'Diese Seite kann nicht gelöscht werden.'];
            }
            
            // Delete child pages first
            db()->execute("DELETE FROM pages WHERE parent_id = ?", [$pageId]);
            
            // Delete page blocks
            db()->execute("DELETE FROM page_blocks WHERE page_slug = ?", [$page['slug']]);
            
            // Delete page
            db()->execute("DELETE FROM pages WHERE id = ?", [$pageId]);
            
            // Clear cache
            self::clearCache();
            
            if (class_exists('Logger')) {
                Logger::info("Page deleted: {$page['slug']} (ID: {$pageId})");
            }
            
            return ['success' => true];
        } catch (Exception $e) {
            error_log("PageManager::deletePage Error: " . $e->getMessage());
            if (class_exists('Logger')) {
                Logger::error('Failed to delete page', [
                    'page_id' => $pageId,
                    'error' => $e->getMessage()
                ]);
            }
            return ['success' => false, 'error' => 'Fehler beim Löschen der Seite.'];
        }
    }
    
    /**
     * Toggle menu visibility
     */
    public static function toggleMenuVisibility($pageId, $showInMenu) {
        try {
            db()->execute(
                "UPDATE pages SET show_in_menu = ? WHERE id = ?",
                [(int)$showInMenu, (int)$pageId]
            );
            
            // If disabling parent, disable children too
            if ($showInMenu == 0) {
                db()->execute("UPDATE pages SET show_in_menu = 0 WHERE parent_id = ?", [(int)$pageId]);
            }
            
            // Clear cache
            self::clearCache();
            
            return true;
        } catch (Exception $e) {
            error_log("PageManager::toggleMenuVisibility Error: " . $e->getMessage());
            if (class_exists('Logger')) {
                Logger::error('Failed to toggle menu visibility', [
                    'page_id' => $pageId,
                    'error' => $e->getMessage()
                ]);
            }
            return false;
        }
    }
    
    /**
     * Reorder menu items
     */
    public static function reorderMenu($pageIds, $parentId = null) {
        try {
            foreach ($pageIds as $order => $pageId) {
                db()->execute(
                    "UPDATE pages SET menu_order = ? WHERE id = ?",
                    [$order, (int)$pageId]
                );
            }
            
            // Clear cache
            self::clearCache();
            
            return true;
        } catch (Exception $e) {
            error_log("PageManager::reorderMenu Error: " . $e->getMessage());
            if (class_exists('Logger')) {
                Logger::error('Failed to reorder menu', [
                    'page_ids' => $pageIds,
                    'error' => $e->getMessage()
                ]);
            }
            return false;
        }
    }
    
    /**
     * Set page parent (for dropdown menus)
     */
    public static function setPageParent($pageId, $parentId) {
        try {
            db()->execute(
                "UPDATE pages SET parent_id = ? WHERE id = ?",
                [$parentId, (int)$pageId]
            );
            return true;
        } catch (Exception $e) {
            error_log("PageManager::setPageParent Error: " . $e->getMessage());
            if (class_exists('Logger')) {
                Logger::error('Failed to set page parent', [
                    'page_id' => $pageId,
                    'parent_id' => $parentId,
                    'error' => $e->getMessage()
                ]);
            }
            return false;
        }
    }
    
    /**
     * Validate slug
     */
    public static function validateSlug($slug, $excludeId = null) {
        try {
            if (empty($slug)) {
                return ['valid' => false, 'error' => 'Slug ist erforderlich.'];
            }
            
            if (!preg_match('/^[a-z0-9-]+$/', $slug)) {
                return ['valid' => false, 'error' => 'Slug darf nur Kleinbuchstaben, Zahlen und Bindestriche enthalten.'];
            }
            
            // Check if slug exists
            if ($excludeId) {
                $existing = db()->fetchOne(
                    "SELECT id FROM pages WHERE slug = ? AND id != ?",
                    [$slug, $excludeId]
                );
            } else {
                $existing = db()->fetchOne(
                    "SELECT id FROM pages WHERE slug = ?",
                    [$slug]
                );
            }
            
            if ($existing) {
                return ['valid' => false, 'error' => 'Dieser Slug wird bereits verwendet.'];
            }
            
            return ['valid' => true];
        } catch (Exception $e) {
            error_log("PageManager::validateSlug Error: " . $e->getMessage());
            return ['valid' => false, 'error' => 'Fehler bei der Validierung.'];
        }
    }
    
    /**
     * Get available icons
     */
    public static function getAvailableIcons() {
        return [
            '📄', '🏠', '📧', '📞', '🎨', '🐕', '🧠', '👥', '✍️', '📝', 
            '💼', '🎓', '🏆', '⭐', '🌟', '❤️', '💡', '🔧', '⚙️', '🚀',
            '🎯', '📊', '📈', '💰', '🛒', '🎁', '📱', '💻', '🖥️', '⌚',
            '📷', '🎵', '🎬', '🎮', '🏃', '⚽', '🍕', '☕', '🌍', '✨'
        ];
    }
    
    /**
     * Clear cache when pages are modified
     */
    public static function clearCache() {
        try {
            // Clear PersistentCache if available
            if (class_exists('PersistentCache')) {
                PersistentCache::delete('all_pages');
                PersistentCache::delete('published_pages');
            }
            
            // Also clear QueryCache
            QueryCache::delete('all_pages');
            QueryCache::delete('published_pages');
            QueryCache::delete('menu_pages');
            QueryCache::clear();
        } catch (Exception $e) {
            error_log("PageManager::clearCache Error: " . $e->getMessage());
            // Non-critical, continue
        }
    }
    
    /**
     * Create page file template (DEPRECATED - using page.php router)
     * Kept for backward compatibility
     */
    public static function createPageFile($slug, $pageType = 'flexible') {
        return ['success' => true, 'message' => 'Using dynamic router'];
    }
    
    /**
     * Get flexible page template (DEPRECATED)
     */
    private static function getFlexiblePageTemplate($slug) {
        return <<<PHP
<?php
/** DEPRECATED: This template is no longer used. 
 * All pages are now handled dynamically by page.php router.
 */
require_once __DIR__ . '/../../backend/core/init.php';
header('Location: /{$slug}');
exit;
?>
PHP;
    }
    
    /**
     * Get standard page template (DEPRECATED)
     */
    private static function getStandardPageTemplate($slug) {
        return self::getFlexiblePageTemplate($slug);
    }
}