<?php
/** cms-core/backend/core/RichTextField.php
 * Rich Text Field Component - MODERN COMPACT STYLE with LIVE PREVIEW
 * Inline toolbar with bold, italic, underline, hyperlink, and real-time preview
 */

class RichTextField {
    
    /**
     * Render a rich text field with inline controls and live preview
     */
    public static function render($fieldId, $fieldName, $value, $label, $required = false) {
        // Parse stored data - HANDLE LEGACY TEXT DATA
        if (is_string($value)) {
            $decoded = json_decode($value, true);
            if (is_array($decoded) && isset($decoded['text'])) {
                $data = $decoded;
            } else {
                $data = [
                    'text' => $value,
                    'alignment' => 'left',
                    'fontSize' => 0,
                    'color' => '',
                    'bold' => false,
                    'italic' => false,
                    'underline' => false,
                    'link' => ''
                ];
            }
        } elseif (is_array($value)) {
            $data = $value;
        } else {
            $data = [
                'text' => '',
                'alignment' => 'left',
                'fontSize' => 0,
                'color' => '',
                'bold' => false,
                'italic' => false,
                'underline' => false,
                'link' => ''
            ];
        }
    
        $text = $data['text'] ?? '';
        $alignment = $data['alignment'] ?? 'left';
        $fontSize = $data['fontSize'] ?? 0;
        $color = $data['color'] ?? '';
        $bold = $data['bold'] ?? false;
        $italic = $data['italic'] ?? false;
        $underline = $data['underline'] ?? false;
        $link = $data['link'] ?? '';
        
        $requiredAttr = $required ? 'required' : '';
        $requiredMark = $required ? '<span class="required-mark">*</span>' : '';
        
        // Calculate font size in pixels (base 15px + fontSize increment)
        $fontSizePx = 15 + ($fontSize * 2);
        
        ob_start();
        ?>
        <div class="form-group rich-text-field-modern">
            <label class="form-label">
                <?= e($label) ?> <?= $requiredMark ?>
            </label>
            
            <!-- Container with integrated toolbar -->
            <div class="rich-text-container" style="border: 1px solid #e0e6ed; border-radius: 12px; overflow: hidden; background: #ffffff; box-shadow: 0 2px 4px rgba(0,0,0,0.05);">
                <!-- Toolbar -->
                <div class="rich-text-toolbar" data-field="<?= e($fieldId) ?>" style="display: flex; align-items: center; gap: 8px; padding: 8px 12px; background: linear-gradient(180deg, #f8fafc, #f1f5f9); border-bottom: 1px solid #e0e6ed; flex-wrap: wrap;">
                    <!-- Alignment -->
                    <button type="button" class="toolbar-btn <?= $alignment === 'left' ? 'active' : '' ?>" onclick="setAlignment('<?= e($fieldId) ?>', 'left', event)" title="Align Left" data-align="left">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M3 5h18M3 9h14M3 13h18M3 17h14"/>
                        </svg>
                    </button>
                    <button type="button" class="toolbar-btn <?= $alignment === 'center' ? 'active' : '' ?>" onclick="setAlignment('<?= e($fieldId) ?>', 'center', event)" title="Align Center" data-align="center">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M3 5h18M5 9h14M3 13h18M5 17h14"/>
                        </svg>
                    </button>
                    <button type="button" class="toolbar-btn <?= $alignment === 'right' ? 'active' : '' ?>" onclick="setAlignment('<?= e($fieldId) ?>', 'right', event)" title="Align Right" data-align="right">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M3 5h18M7 9h14M3 13h18M7 17h14"/>
                        </svg>
                    </button>
                    
                    <div class="toolbar-divider"></div>
                    
                    <!-- Text Styles -->
                    <button type="button" class="toolbar-btn <?= $bold ? 'active' : '' ?>" onclick="toggleStyle('<?= e($fieldId) ?>', 'bold', event)" title="Bold" data-style="bold">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M6 4h8a4 4 0 0 1 4 4v1a4 4 0 0 1-4 4H6m0 0h9a4 4 0 0 1 4 4v1a4 4 0 0 1-4 4H6"/>
                        </svg>
                    </button>
                    <button type="button" class="toolbar-btn <?= $italic ? 'active' : '' ?>" onclick="toggleStyle('<?= e($fieldId) ?>', 'italic', event)" title="Italic" data-style="italic">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M19 4h-8l-2 16h8"/>
                        </svg>
                    </button>
                    <button type="button" class="toolbar-btn <?= $underline ? 'active' : '' ?>" onclick="toggleStyle('<?= e($fieldId) ?>', 'underline', event)" title="Underline" data-style="underline">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M6 3v10a6 6 0 0 0 12 0V3M4 21h16"/>
                        </svg>
                    </button>
                    
                    <div class="toolbar-divider"></div>
                    
                    <!-- Font Size -->
                    <button type="button" class="toolbar-btn" onclick="adjustFontSize('<?= e($fieldId) ?>', -1, event)" title="Decrease Font Size">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M9 11H15M12 8V14M3 3h18"/>
                        </svg>
                    </button>
                    <span class="font-size-display" id="fontSize-<?= e($fieldId) ?>" style="min-width: 32px; text-align: center; font-weight: 500; color: #4f46e5; font-size: 12px;"><?= $fontSize > 0 ? '+' . $fontSize : $fontSize ?></span>
                    <button type="button" class="toolbar-btn" onclick="adjustFontSize('<?= e($fieldId) ?>', 1, event)" title="Increase Font Size">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M9 11H15M12 8V14M3 3h18M12 8V14"/>
                        </svg>
                    </button>
                    
                    <div class="toolbar-divider"></div>
                    
                    <!-- Color -->
                    <label class="toolbar-btn color-picker" title="Text Color">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 18c-4.42 0-8-3.58-8-8s3.58-8 8-8 8 3.58 8 8-3.58 8-8 8z"/>
                            <circle cx="12" cy="12" r="4" fill="currentColor"/>
                        </svg>
                        <input type="color" 
                               id="color-<?= e($fieldId) ?>"
                               value="<?= $color ?: '#000000' ?>"
                               onchange="setTextColor('<?= e($fieldId) ?>', this.value, event)"
                               style="display: none;">
                    </label>
                    <?php if ($color): ?>
                    <button type="button" class="toolbar-btn" onclick="resetTextColor('<?= e($fieldId) ?>', event)" title="Reset Color">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M6 18L18 6M6 6l12 12"/>
                        </svg>
                    </button>
                    <?php endif; ?>
                    
                    <div class="toolbar-divider"></div>
                    
                    <!-- Hyperlink -->
                    <button type="button" class="toolbar-btn <?= $link ? 'active' : '' ?>" onclick="setHyperlink('<?= e($fieldId) ?>', event)" title="Add/Remove Hyperlink" data-style="link">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M10 13a5 5 0 0 0 7.54.54l3-3a5 5 0 0 0-7.07-7.07l-1.72 1.71"/>
                            <path d="M14 11a5 5 0 0 0-7.54-.54l-3 3a5 5 0 0 0 7.07 7.07l1.71-1.71"/>
                        </svg>
                    </button>
                </div>
                
                <!-- Contenteditable Input with Live Preview -->
                <div 
                    id="<?= e($fieldId) ?>" 
                    class="rich-text-input"
                    contenteditable="true"
                    oninput="updateRichTextField('<?= e($fieldId) ?>')"
                    style="width: 100%; min-height: 80px; padding: 12px 16px; border: none; font-size: <?= $fontSizePx ?>px; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; outline: none; line-height: 1.5; background: #ffffff; text-align: <?= e($alignment) ?>; <?= $bold ? 'font-weight: bold;' : '' ?> <?= $italic ? 'font-style: italic;' : '' ?> <?= $underline ? 'text-decoration: underline;' : '' ?> <?= $color ? 'color: ' . e($color) . ';' : '' ?>"
                    <?= $requiredAttr ?>>
                    <?php if ($link): ?>
                        <a href="<?= e($link) ?>" style="color: #4f46e5; text-decoration: underline;" target="_blank"><?= e($text) ?></a>
                    <?php else: ?>
                        <?= e($text) ?>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Hidden inputs for data storage -->
            <input type="hidden" 
                   id="<?= e($fieldId) ?>-text" 
                   name="<?= e($fieldName) ?>[text]" 
                   value="<?= e($text) ?>">
            <input type="hidden" 
                   id="<?= e($fieldId) ?>-alignment" 
                   name="<?= e($fieldName) ?>[alignment]" 
                   value="<?= e($alignment) ?>">
            <input type="hidden" 
                   id="<?= e($fieldId) ?>-fontSize" 
                   name="<?= e($fieldName) ?>[fontSize]" 
                   value="<?= e($fontSize) ?>">
            <input type="hidden" 
                   id="<?= e($fieldId) ?>-color" 
                   name="<?= e($fieldName) ?>[color]" 
                   value="<?= e($color) ?>">
            <input type="hidden" 
                   id="<?= e($fieldId) ?>-bold" 
                   name="<?= e($fieldName) ?>[bold]" 
                   value="<?= $bold ? '1' : '0' ?>">
            <input type="hidden" 
                   id="<?= e($fieldId) ?>-italic" 
                   name="<?= e($fieldName) ?>[italic]" 
                   value="<?= $italic ? '1' : '0' ?>">
            <input type="hidden" 
                   id="<?= e($fieldId) ?>-underline" 
                   name="<?= e($fieldName) ?>[underline]" 
                   value="<?= $underline ? '1' : '0' ?>">
            <input type="hidden" 
                   id="<?= e($fieldId) ?>-link" 
                   name="<?= e($fieldName) ?>[link]" 
                   value="<?= e($link) ?>">
        </div>
        
        <style>
            .rich-text-field-modern {
                margin-bottom: 24px;
            }

            .form-label {
                font-size: 14px;
                font-weight: 600;
                color: #1e293b;
                margin-bottom: 8px;
                display: block;
            }

            .required-mark {
                color: #e11d48;
                font-weight: bold;
            }

            .rich-text-container {
                transition: all 0.2s ease;
            }

            .rich-text-container:hover {
                box-shadow: 0 4px 8px rgba(0,0,0,0.1);
            }

            .toolbar-btn {
                padding: 6px;
                background: #ffffff;
                border: 1px solid #e0e6ed;
                border-radius: 6px;
                cursor: pointer;
                transition: all 0.2s ease;
                display: flex;
                align-items: center;
                justify-content: center;
                width: 32px;
                height: 32px;
                color: #475569;
            }

            .toolbar-btn:hover {
                background: #4f46e5;
                border-color: #4f46e5;
                color: #ffffff;
                transform: translateY(-1px);
            }

            .toolbar-btn.active {
                background: #4f46e5;
                border-color: #4f46e5;
                color: #ffffff;
            }

            .toolbar-divider {
                width: 1px;
                height: 24px;
                background: #e0e6ed;
                margin: 0 8px;
            }

            .font-size-display {
                display: flex;
                align-items: center;
                justify-content: center;
                padding: 0 8px;
                background: #eef2ff;
                border-radius: 6px;
                font-weight: 500;
            }

            .rich-text-input {
                transition: background 0.2s ease;
                overflow-y: auto;
                resize: vertical;
            }

            .rich-text-input:focus {
                background: #f8fafc;
            }

            .rich-text-input a {
                color: #4f46e5;
                text-decoration: underline;
                cursor: pointer;
            }

            .rich-text-input a:hover {
                text-decoration: none;
            }
        </style>
        
        <script>
            function setAlignment(fieldId, alignment, event) {
                event.preventDefault();
                event.stopPropagation();
                
                const alignmentInput = document.getElementById(`${fieldId}-alignment`);
                alignmentInput.value = alignment;
                document.getElementById(fieldId).style.textAlign = alignment;
                
                const toolbar = document.querySelector(`[data-field="${fieldId}"]`);
                toolbar.querySelectorAll('[data-align]').forEach(btn => btn.classList.remove('active'));
                toolbar.querySelector(`[data-align="${alignment}"]`).classList.add('active');
            }
            
            function toggleStyle(fieldId, style, event) {
                event.preventDefault();
                event.stopPropagation();
                
                const styleInput = document.getElementById(`${fieldId}-${style}`);
                const editor = document.getElementById(fieldId);
                const isActive = styleInput.value === '1';
                
                styleInput.value = isActive ? '0' : '1';
                
                const button = document.querySelector(`[data-field="${fieldId}"] [data-style="${style}"]`);
                button.classList.toggle('active', !isActive);
                
                // Update editor styles
                if (style === 'bold') {
                    editor.style.fontWeight = isActive ? 'normal' : 'bold';
                } else if (style === 'italic') {
                    editor.style.fontStyle = isActive ? 'normal' : 'italic';
                } else if (style === 'underline') {
                    editor.style.textDecoration = isActive ? 'none' : 'underline';
                }
            }
            
            function adjustFontSize(fieldId, delta, event) {
                event.preventDefault();
                event.stopPropagation();
                
                const fontSizeInput = document.getElementById(`${fieldId}-fontSize`);
                let currentSize = parseInt(fontSizeInput.value) || 0;
                let newSize = Math.max(-4, Math.min(4, currentSize + delta));
                
                fontSizeInput.value = newSize;
                document.getElementById(`fontSize-${fieldId}`).textContent = newSize > 0 ? `+${newSize}` : newSize;
                document.getElementById(fieldId).style.fontSize = `${15 + (newSize * 2)}px`;
            }
            
            function setTextColor(fieldId, color, event) {
                event.preventDefault();
                event.stopPropagation();
                
                const colorInput = document.getElementById(`${fieldId}-color`);
                colorInput.value = color;
                document.getElementById(fieldId).style.color = color;
                
                // Dynamically add reset button if not present
                if (!document.querySelector(`[onclick="resetTextColor('${fieldId}', event)"]`)) {
                    const toolbar = document.querySelector(`[data-field="${fieldId}"]`);
                    const resetButton = document.createElement('button');
                    resetButton.type = 'button';
                    resetButton.className = 'toolbar-btn';
                    resetButton.title = 'Reset Color';
                    resetButton.innerHTML = `<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M6 18L18 6M6 6l12 12"/></svg>`;
                    resetButton.onclick = (e) => resetTextColor(fieldId, e);
                    toolbar.insertBefore(resetButton, toolbar.querySelector('[data-style="link"]'));
                }
            }
            
            function resetTextColor(fieldId, event) {
                event.preventDefault();
                event.stopPropagation();
                
                document.getElementById(`${fieldId}-color`).value = '';
                document.getElementById(`color-${fieldId}`).value = '#000000';
                document.getElementById(fieldId).style.color = '';
                
                event.target.remove();
            }
            
            function setHyperlink(fieldId, event) {
                event.preventDefault();
                event.stopPropagation();
                
                const linkInput = document.getElementById(`${fieldId}-link`);
                const editor = document.getElementById(fieldId);
                const button = document.querySelector(`[data-field="${fieldId}"] [data-style="link"]`);
                const isActive = linkInput.value !== '';
                
                if (isActive) {
                    // Remove link
                    linkInput.value = '';
                    const text = editor.textContent;
                    editor.innerHTML = text;
                    button.classList.remove('active');
                } else {
                    // Add link
                    const url = prompt('Enter URL (e.g., https://example.com):');
                    if (url && /^https?:\/\/.+/.test(url)) {
                        linkInput.value = url;
                        const text = editor.textContent || 'Link';
                        editor.innerHTML = `<a href="${url}" style="color: #4f46e5; text-decoration: underline;" target="_blank">${text}</a>`;
                        button.classList.add('active');
                    } else if (url) {
                        alert('Please enter a valid URL starting with http:// or https://');
                    }
                }
                updateRichTextField(fieldId);
            }
            
            function updateRichTextField(fieldId) {
                const editor = document.getElementById(fieldId);
                const textInput = document.getElementById(`${fieldId}-text`);
                // Extract plain text, removing HTML tags if any
                textInput.value = editor.textContent;
            }
            
            // Initialize on load
            document.addEventListener('DOMContentLoaded', () => {
                const fieldId = '<?= e($fieldId) ?>';
                const editor = document.getElementById(fieldId);
                if (editor) {
                    updateRichTextField(fieldId);
                    // Ensure cursor remains in editor
                    editor.addEventListener('keydown', (e) => {
                        if (e.key === 'Enter') {
                            e.preventDefault();
                            document.execCommand('insertHTML', false, '<div><br></div>');
                        }
                    });
                }
            });
        </script>
        <?php
        return ob_get_clean();
    }
}