<?php
/**
 * cms-core/backend/core/SitemapGenerator.php
 * SEO-Optimized Sitemap Generator with blog_enabled check
 * Follows Google's best practices for high ranking
 */

class SitemapGenerator {
    
    private $urls = [];
    private $baseUrl;
    
    /**
     * Initialize with base URL
     */
    public function __construct() {
        $this->baseUrl = rtrim(defined('APP_URL') ? APP_URL : BASE_URL, '/');
    }
    
    /**
     * Generate complete sitemap
     * FIXED: Changed from static to instance method
     */
    public function generate() {
        $this->urls = [];
        
        // Add homepage (highest priority)
        $this->addUrl('/', 1.0, 'daily');
        
        // Add all published pages
        $this->addPages();
        
        // Add all published blog posts (ONLY if blog is enabled)
        $this->addBlogPosts();
        
        // Generate XML
        return $this->buildXML();
    }
    
    /**
     * Add all published pages to sitemap
     */
 /**
 * Add all published pages to sitemap
 */
private function addPages() {
    // FIXED: Use last_modified column
    $pages = db()->fetchAll(
        "SELECT slug, last_modified, created_at FROM pages 
         WHERE status = 'published' 
         ORDER BY slug"
    );
    
    foreach ($pages as $page) {
        // Skip home page (already added as /)
        if ($page['slug'] === 'home') {
            continue;
        }
        
        // Determine priority based on page
        $priority = $this->calculatePagePriority($page['slug']);
        
        // Determine change frequency
        $changefreq = $this->calculateChangeFrequency($page['slug']);
        
        // Use last_modified if available, otherwise created_at
        $lastmod = $page['last_modified'] ?? $page['created_at'];
        
        // Add to sitemap
        $this->addUrl(
            '/' . $page['slug'],
            $priority,
            $changefreq,
            $lastmod
        );
    }
}
    
    /**
     * Add all blog posts to sitemap
     * UPDATED: Only includes blog if blog_enabled setting is true
     */
/**
 * Add all blog posts to sitemap
 * UPDATED: Only includes blog if blog_enabled setting is true
 */
private function addBlogPosts() {
    // Check if blog is enabled in settings
    if (!get_setting('blog_enabled', true)) {
        Logger::info('Blog is disabled in settings, skipping blog in sitemap');
        return;
    }
    
    // Check if blog_posts table exists
    try {
        // FIXED: Use created_at and published_at
        $posts = db()->fetchAll(
            "SELECT slug, published_at, created_at FROM blog_posts 
             WHERE status = 'published' 
             ORDER BY published_at DESC"
        );
        
        foreach ($posts as $post) {
            // Use published_at if available, otherwise created_at
            $lastmod = $post['published_at'] ?? $post['created_at'];
            
            $this->addUrl(
                '/blog/' . $post['slug'],
                0.6, // Blog posts have medium-high priority
                'weekly',
                $lastmod
            );
        }
        
        // Add blog overview page
        if (!empty($posts)) {
            $this->addUrl('/blog', 0.7, 'daily');
        }
        
    } catch (Exception $e) {
        // Blog table doesn't exist yet, skip
        Logger::info('Blog table not found, skipping blog in sitemap: ' . $e->getMessage());
    }
}
    
    /**
     * Add single URL to sitemap
     */
    private function addUrl($path, $priority = 0.5, $changefreq = 'monthly', $lastmod = null) {
        $this->urls[] = [
            'loc' => $this->baseUrl . $path,
            'lastmod' => $lastmod ? date('Y-m-d', strtotime($lastmod)) : date('Y-m-d'),
            'changefreq' => $changefreq,
            'priority' => number_format($priority, 1)
        ];
    }
    
    /**
     * Calculate priority based on page importance
     */
    private function calculatePagePriority($slug) {
        // High priority pages
        $highPriority = ['kontakt', 'contact', 'about', 'ueber-uns', 'services', 'leistungen'];
        if (in_array($slug, $highPriority)) {
            return 0.9;
        }
        
        // Medium priority pages
        $mediumPriority = ['impressum', 'datenschutz', 'privacy', 'imprint'];
        if (in_array($slug, $mediumPriority)) {
            return 0.3;
        }
        
        // Default priority
        return 0.7;
    }
    
    /**
     * Calculate change frequency based on page type
     */
    private function calculateChangeFrequency($slug) {
        // Frequently updated pages
        $daily = ['news', 'blog', 'aktuelles'];
        if (in_array($slug, $daily)) {
            return 'daily';
        }
        
        // Occasionally updated pages
        $weekly = ['services', 'leistungen', 'portfolio', 'projekte'];
        if (in_array($slug, $weekly)) {
            return 'weekly';
        }
        
        // Static pages
        $yearly = ['impressum', 'datenschutz', 'privacy', 'imprint', 'agb'];
        if (in_array($slug, $yearly)) {
            return 'yearly';
        }
        
        // Default
        return 'monthly';
    }
    
    /**
     * Build XML sitemap
     */
    private function buildXML() {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
        
        foreach ($this->urls as $url) {
            $xml .= '  <url>' . "\n";
            $xml .= '    <loc>' . htmlspecialchars($url['loc']) . '</loc>' . "\n";
            $xml .= '    <lastmod>' . $url['lastmod'] . '</lastmod>' . "\n";
            $xml .= '    <changefreq>' . $url['changefreq'] . '</changefreq>' . "\n";
            $xml .= '    <priority>' . $url['priority'] . '</priority>' . "\n";
            $xml .= '  </url>' . "\n";
        }
        
        $xml .= '</urlset>';
        
        return $xml;
    }
    
    /**
     * Save sitemap to file
     */
    public function save() {
        $xml = $this->generate();
        $filepath = APP_ROOT . '/sitemap.xml';
        
        if (file_put_contents($filepath, $xml)) {
            Logger::info('Sitemap generated successfully: ' . count($this->urls) . ' URLs');
            return [
                'success' => true,
                'pages' => count($this->urls),
                'file' => $filepath
            ];
        } else {
            Logger::error('Failed to save sitemap to: ' . $filepath);
            return [
                'success' => false,
                'error' => 'Failed to write sitemap file'
            ];
        }
    }
    
    /**
     * Get sitemap statistics
     */
    public function getStats() {
        $filepath = APP_ROOT . '/sitemap.xml';
        
        if (!file_exists($filepath)) {
            return [
                'exists' => false,
                'urls' => 0,
                'last_generated' => null,
                'file_size' => 0
            ];
        }
        
        return [
            'exists' => true,
            'urls' => count($this->urls),
            'last_generated' => date('d.m.Y H:i', filemtime($filepath)),
            'file_size' => filesize($filepath)
        ];
    }
    
    /**
     * Ping search engines about sitemap update
     */
    public function pingSearchEngines() {
        $sitemapUrl = urlencode($this->baseUrl . '/sitemap.xml');
        
        $searchEngines = [
            'Google' => "https://www.google.com/ping?sitemap={$sitemapUrl}",
            'Bing' => "https://www.bing.com/ping?sitemap={$sitemapUrl}"
        ];
        
        $results = [];
        
        foreach ($searchEngines as $engine => $url) {
            try {
                $response = @file_get_contents($url);
                $results[$engine] = $response !== false;
                
                if ($results[$engine]) {
                    Logger::info("Pinged {$engine} about sitemap update");
                } else {
                    Logger::warning("Failed to ping {$engine}");
                }
            } catch (Exception $e) {
                $results[$engine] = false;
                Logger::error("Error pinging {$engine}: " . $e->getMessage());
            }
        }
        
        return $results;
    }
    
    /**
     * Auto-generate sitemap on page/post save
     */
    public static function autoGenerate() {
        // Check if enough time has passed since last generation (prevent spam)
        $lastGenFile = APP_ROOT . '/sitemap.xml';
        
        if (file_exists($lastGenFile)) {
            $lastMod = filemtime($lastGenFile);
            $timeSince = time() - $lastMod;
            
            // Only regenerate if more than 5 minutes passed
            if ($timeSince < 300) {
                return false;
            }
        }
        
        $generator = new self();
        return $generator->save();
    }
}