<?php
/**
 * cms-core/backend/core/Translation.php
 * Smart Translation System with Auto-Translate - FIXED VERSION
 * 
 * Features:
 * - Write ONCE in German, auto-translate to English
 * - Cache translations in JSON file
 * - Manual override possible
 * - Simple function: t('Text auf Deutsch')
 * - Language switch in install + admin panel
 */

class Translation {
    private static $currentLang = 'de';
    private static $translations = [];
    private static $translationFile = __DIR__ . '/../../translations.json';
    private static $fallbackLang = 'de';
    
    /**
     * Initialize translation system
     */
    public static function init() {
        $sessionLang = $_SESSION['language'] ?? null;
        $configLang = defined('APP_LANGUAGE') ? APP_LANGUAGE : null;
        $envLang = getenv('APP_LANGUAGE') ?: null;
        
        self::$currentLang = $sessionLang ?? $configLang ?? $envLang ?? 'de';
        
        // Load translations from cache
        if (file_exists(self::$translationFile)) {
            $content = @file_get_contents(self::$translationFile);
            if ($content) {
                self::$translations = json_decode($content, true) ?: [];
            }
        }
    }
    
    /**
     * Set language (also updates session)
     */
    public static function setLanguage($lang) {
        if (in_array($lang, ['de', 'en'])) {
            self::$currentLang = $lang;
            $_SESSION['language'] = $lang;
        }
    }
    
    /**
     * Main translation function
     */
    public static function t($germanText, $vars = []) {
        // If German is selected, just return the text
        if (self::$currentLang === 'de') {
            return self::replaceVars($germanText, $vars);
        }
        
        // Check if translation exists in cache
        $hash = md5($germanText);
        
        if (isset(self::$translations[$hash][self::$currentLang])) {
            $translated = self::$translations[$hash][self::$currentLang];
            return self::replaceVars($translated, $vars);
        }
        
        // Translation not found - auto-translate and cache it
        $translated = self::autoTranslate($germanText, self::$currentLang);
        
        // Cache the translation
        self::cacheTranslation($germanText, self::$currentLang, $translated);
        
        return self::replaceVars($translated, $vars);
    }
    
    /**
     * Auto-translate using EXACT MATCH ONLY
     * NO word-by-word replacement to avoid partial matches!
     */
    private static function autoTranslate($text, $targetLang) {
        if ($targetLang === 'de') return $text;
        
        // Complete translation dictionary - EVERY text must be here!
        $translations = [
            // Navigation sections
            'Dashboard' => 'Dashboard',
            'Seiten' => 'Pages',
            'Blog' => 'Blog',
            'Medien' => 'Media',
            'Design' => 'Design',
            'Benutzer' => 'Users',
            'System' => 'System',
            
            // Page links
            'Alle Seiten' => 'All Pages',
            'Neue Seite' => 'New Page',
            'Alle Posts' => 'All Posts',
            'Neuer Beitrag' => 'New Post',
            'Media Bibliothek' => 'Media Library',
            'Farbe & Schrift' => 'Color & Font',
            'Benutzerverwaltung' => 'User Management',
            'Einstellungen' => 'Settings',
            'Integrationen' => 'Integrations',
            'Sitemap' => 'Sitemap',
            'Error Logs' => 'Error Logs',
            'System-Update' => 'System Update',
            
            // Custom Addons
            'Custom-Addons' => 'Custom Add-ons',
            'Farb-Phänomen' => 'Color Phenomenon',
            'Color Experiment' => 'Color Experiment',
            
            // Actions
            'Abmelden' => 'Logout',
            'Passwort ändern' => 'Change Password',
            'Website ansehen' => 'View Website',
            'Speichern' => 'Save',
            'Löschen' => 'Delete',
            'Bearbeiten' => 'Edit',
            'Hinzufügen' => 'Add',
            'Erstellen' => 'Create',
            'Zurück' => 'Back',
            'Weiter' => 'Next',
            'Abbrechen' => 'Cancel',
            'Bestätigen' => 'Confirm',
            'Suchen' => 'Search',
            'Filtern' => 'Filter',
            'Sortieren' => 'Sort',
            'Hochladen' => 'Upload',
            'Herunterladen' => 'Download',
            'Exportieren' => 'Export',
            'Importieren' => 'Import',
            'Aktualisieren' => 'Update',
            'Aktivieren' => 'Activate',
            'Deaktivieren' => 'Deactivate',
            
            // Messages
            'Erfolgreich gespeichert' => 'Successfully saved',
            'Erfolgreich gelöscht' => 'Successfully deleted',
            'Erfolgreich erstellt' => 'Successfully created',
            'Erfolgreich aktualisiert' => 'Successfully updated',
            'Fehler beim Speichern' => 'Error saving',
            'Fehler beim Löschen' => 'Error deleting',
            'Fehler beim Erstellen' => 'Error creating',
            'Bitte alle Felder ausfüllen' => 'Please fill all fields',
            'Keine Ergebnisse gefunden' => 'No results found',
            'Laden...' => 'Loading...',
            'Bitte warten...' => 'Please wait...',
            'Möchten Sie wirklich löschen?' => 'Are you sure you want to delete?',
            
            // Maintenance
            'WARTUNGSMODUS AKTIV' => 'MAINTENANCE MODE ACTIVE',
            'Die Website ist für Besucher nicht erreichbar.' => 'The website is not accessible to visitors.',
            'Deaktivieren' => 'Deactivate',
            
            // Language
            'Sprache wechseln' => 'Switch Language',
            'Update verfügbar' => 'Update Available',
            
            // Forms
            'Titel' => 'Title',
            'Inhalt' => 'Content',
            'Beschreibung' => 'Description',
            'Bild' => 'Image',
            'Bilder' => 'Images',
            'Datum' => 'Date',
            'Status' => 'Status',
            'Autor' => 'Author',
            'Kategorie' => 'Category',
            'Tags' => 'Tags',
            'URL' => 'URL',
            'Slug' => 'Slug',
            'Meta-Titel' => 'Meta Title',
            'Meta-Beschreibung' => 'Meta Description',
            
            // Installation
            'Installation' => 'Installation',
            'Datenbank-Konfiguration' => 'Database Configuration',
            'Seiten-Konfiguration' => 'Site Configuration',
            'Website-URL' => 'Website URL',
            'Website-Name' => 'Website Name',
            'Admin-Email' => 'Admin Email',
            'Admin-Benutzername' => 'Admin Username',
            'Admin-Passwort' => 'Admin Password',
            'Passwort bestätigen' => 'Confirm Password',
            'Installation starten' => 'Start Installation',
            'Installation erfolgreich' => 'Installation Successful',
            
            // Time
            'Heute' => 'Today',
            'Gestern' => 'Yesterday',
            'Diese Woche' => 'This Week',
            'Dieser Monat' => 'This Month',
            'Dieses Jahr' => 'This Year',
            
            // Common
            'von' => 'from',
            'bis' => 'to',
            'und' => 'and',
            'oder' => 'or',
            'ja' => 'yes',
            'nein' => 'no',
        ];
        
        // EXACT MATCH ONLY - no word replacement!
        $textLower = mb_strtolower($text);
        
        foreach ($translations as $de => $en) {
            if (mb_strtolower($de) === $textLower) {
                // Preserve original capitalization if possible
                if ($text === $de) {
                    return $en;
                }
                // If all uppercase, return uppercase
                if (mb_strtoupper($de) === $text) {
                    return mb_strtoupper($en);
                }
                // Otherwise return as-is
                return $en;
            }
        }
        
        // No translation found - return original
        return $text;
    }
    
    /**
     * Cache translation to JSON file
     */
    private static function cacheTranslation($original, $lang, $translated) {
        $hash = md5($original);
        
        if (!isset(self::$translations[$hash])) {
            self::$translations[$hash] = [
                'de' => $original
            ];
        }
        
        self::$translations[$hash][$lang] = $translated;
        
        @file_put_contents(
            self::$translationFile, 
            json_encode(self::$translations, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)
        );
    }
    
    /**
     * Replace variables in text
     */
    private static function replaceVars($text, $vars) {
        foreach ($vars as $key => $value) {
            $text = str_replace(':' . $key, $value, $text);
        }
        return $text;
    }
    
    /**
     * Get current language
     */
    public static function getCurrentLang() {
        return self::$currentLang;
    }
    
    /**
     * Get all available languages
     */
    public static function getAvailableLanguages() {
        return [
            'de' => '🇩🇪 Deutsch',
            'en' => '🇬🇧 English'
        ];
    }
    
    /**
     * Manual override for specific translations
     */
    public static function setTranslation($germanText, $lang, $translation) {
        $hash = md5($germanText);
        
        if (!isset(self::$translations[$hash])) {
            self::$translations[$hash] = ['de' => $germanText];
        }
        
        self::$translations[$hash][$lang] = $translation;
        
        @file_put_contents(
            self::$translationFile,
            json_encode(self::$translations, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)
        );
    }
    
    /**
     * Export all translations for review
     */
    public static function exportTranslations() {
        return self::$translations;
    }
}

/**
 * Global helper function
 */
function t($text, $vars = []) {
    return Translation::t($text, $vars);
}

/**
 * Get current language
 */
function current_lang() {
    return Translation::getCurrentLang();
}

/**
 * Check if current language is X
 */
function is_lang($lang) {
    return Translation::getCurrentLang() === $lang;
}