<?php
/**
 * cms-core/backend/core/helpers.php
 * Helper-Funktionen - WITH CDN & WebP SUPPORT
 */

/**
 * Redirect zu URL - FIXED VERSION
 */
function redirect($url, $statusCode = 302) {
    // Clean any output buffers
    while (ob_get_level() > 0) {
        ob_end_clean();
    }
    
    // Make URL absolute if needed
    if (strpos($url, 'http') !== 0) {
        // Handle protocol
        $forceSsl = defined('FORCE_SSL') ? FORCE_SSL : true;
        $protocol = $forceSsl ? 'https://' : 'http://';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        
        // Remove leading slash for consistency
        $url = ltrim($url, '/');
        
        // Build full URL
        $url = $protocol . $host . '/' . $url;
    }
    
    // Check if headers already sent
    if (headers_sent($file, $line)) {
        error_log("❌ REDIRECT FAILED: Headers already sent in {$file} on line {$line}");
        // Fallback to JavaScript redirect
        echo "<script>window.location.href = '{$url}';</script>";
        echo "<noscript><meta http-equiv='refresh' content='0;url={$url}'></noscript>";
        exit;
    }
    
    // Send redirect header
    header("Location: " . $url, true, $statusCode);
    exit;
}

/**
 * Überprüft ob Request eine POST-Anfrage ist
 */
function is_post() {
    return $_SERVER['REQUEST_METHOD'] === 'POST';
}

/**
 * Überprüft ob Request eine GET-Anfrage ist
 */
function is_get() {
    return $_SERVER['REQUEST_METHOD'] === 'GET';
}

/**
 * Holt POST-Daten mit optionalem Default-Wert
 */
function post($key, $default = null) {
    return $_POST[$key] ?? $default;
}

/**
 * Holt GET-Daten mit optionalem Default-Wert
 */
function get_param($key, $default = null) {
    return $_GET[$key] ?? $default;
}

/**
 * Escape HTML für sichere Ausgabe
 */
function e($string) {
    // Safety check for arrays
    if (is_array($string)) {
        error_log("WARNING: e() received array - " . json_encode($string));
        return APP_DEBUG ? '[ARRAY: ' . json_encode($string) . ']' : '';
    }
    
    // Safety check for null
    if ($string === null) {
        return '';
    }
    
    // Safety check for objects
    if (is_object($string)) {
        error_log("WARNING: e() received object - " . get_class($string));
        return APP_DEBUG ? '[OBJECT: ' . get_class($string) . ']' : '';
    }
    
    return htmlspecialchars((string)$string, ENT_QUOTES, 'UTF-8');
}

/**
 * Erstellt sicheren Slug aus String
 */
function create_slug($string) {
    $string = strtolower(trim($string));
    
    // Deutsche Umlaute ersetzen
    $umlauts = ['ä' => 'ae', 'ö' => 'oe', 'ü' => 'ue', 'ß' => 'ss'];
    $string = strtr($string, $umlauts);
    
    // Nur alphanumerische Zeichen und Bindestriche
    $string = preg_replace('/[^a-z0-9-]/', '-', $string);
    $string = preg_replace('/-+/', '-', $string);
    $string = trim($string, '-');
    
    return $string;
}

/**
 * Gibt die Base URL der Website zurück
 * RESPECTS FORCE_SSL SETTING FROM .env
 */
function base_url() {
    // Use constant if defined (from config.php)
    if (defined('BASE_URL')) {
        return BASE_URL;
    }
    
    // Fallback: construct from server variables
    // Check FORCE_SSL setting
    $forceSsl = defined('FORCE_SSL') ? FORCE_SSL : true;
    
    // Determine protocol
    if ($forceSsl) {
        $protocol = 'https';
    } else {
        // Use current protocol if FORCE_SSL is false
        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    }
    
    $host = $_SERVER['HTTP_HOST'];
    return $protocol . '://' . $host;
}

/**
 * Erstellt URL für Frontend-Seite
 */
function site_url($path = '') {
    $base = base_url();
    $path = ltrim($path, '/');
    return $base . '/' . $path;
}

/**
 * Erstellt URL für Admin-Bereich
 */
function admin_url($path = '') {
    $base = base_url();
    $path = ltrim($path, '/');
    return $base . '/admin/' . $path;
}

/**
 * Get CDN-optimized URL for media files
 * Returns CDN URL if enabled, otherwise returns standard MEDIA_URL
 */
function cdn_url($path) {
    // Check if CDN is enabled
    $cdnEnabled = get_setting('cloudflare_cdn_enabled', false);
    
    if (!$cdnEnabled) {
        return MEDIA_URL . '/' . ltrim($path, '/');
    }
    
    // Get CDN URL from settings
    $cdnUrl = get_setting('cloudflare_cdn_url', '');
    
    if (empty($cdnUrl)) {
        // If no custom CDN URL, use base URL (Cloudflare works transparently)
        return MEDIA_URL . '/' . ltrim($path, '/');
    }
    
    // Return CDN URL
    return rtrim($cdnUrl, '/') . '/media/' . ltrim($path, '/');
}

/**
 * Erstellt URL für Media-Dateien (with CDN support)
 */
function media_url($path = '') {
    $path = ltrim($path, '/');
    return cdn_url($path);
}

/**
 * Get asset URL (CSS/JS) - can also use CDN
 */
function asset_url($path = '') {
    $cdnEnabled = get_setting('cloudflare_cdn_enabled', false);
    $cdnMediaOnly = get_setting('cloudflare_cdn_media_only', true);
    $path = ltrim($path, '/');
    
    // If CDN is enabled and not media-only mode, use CDN for assets too
    if ($cdnEnabled && !$cdnMediaOnly) {
        $cdnUrl = get_setting('cloudflare_cdn_url', '');
        if (!empty($cdnUrl)) {
            return rtrim($cdnUrl, '/') . '/assets/' . $path;
        }
    }
    
    // Standard asset URL
    $base = base_url();
    return $base . '/assets/' . $path;
}

/**
 * Erstellt URL für benutzerdefinierte Seiten
 */
function page_url($slug) {
    $slug = ltrim($slug, '/');
    return base_url() . '/' . $slug;
}

/**
 * Erstellt URL für Blog-Übersicht
 */
function blog_url() {
    return base_url() . '/blog';
}

/**
 * Erstellt URL für Blog-Post
 */
function blog_post_url($slug) {
    $slug = ltrim($slug, '/');
    return base_url() . '/blog/post/' . $slug;
}

/**
 * Filesystem Helper: Get absolute path from APP_ROOT
 */
function app_path($path = '') {
    return APP_ROOT . ($path ? '/' . ltrim($path, '/') : '');
}

/**
 * Filesystem Helper: Get path to media directory
 */
function media_path($path = '') {
    return MEDIA_ROOT . ($path ? '/' . ltrim($path, '/') : '');
}

/**
 * Filesystem Helper: Get path to logs directory
 */
function logs_path($path = '') {
    return LOGS_ROOT . ($path ? '/' . ltrim($path, '/') : '');
}

/**
 * Filesystem Helper: Get path to customizations directory
 */
function custom_path($path = '') {
    return CUSTOM_ROOT . ($path ? '/' . ltrim($path, '/') : '');
}

/**
 * Filesystem Helper: Get path to blog posts directory
 */
function blog_path($path = '') {
    return CUSTOM_BLOG_PATH . ($path ? '/' . ltrim($path, '/') : '');
}

/**
 * Get current page slug from URL
 */
function get_current_slug() {
    $uri = $_SERVER['REQUEST_URI'] ?? '';
    $uri = strtok($uri, '?'); // Remove query string
    $uri = trim($uri, '/');
    
    // If empty or homepage
    if (empty($uri) || $uri === 'index.php') {
        return 'home';
    }
    
    // If it's a blog post
    if (preg_match('#^blog/post/([a-z0-9-]+)#', $uri, $matches)) {
        return 'blog-post';
    }
    
    // If it's blog overview
    if ($uri === 'blog') {
        return 'blog';
    }
    
    // Remove file extension if present
    if (substr($uri, -4) === '.php') {
        $uri = substr($uri, 0, -4);
    }
    
    return $uri;
}

/**
 * Check if we're on homepage
 */
function is_homepage() {
    $slug = get_current_slug();
    return $slug === 'home' || $slug === '';
}

/**
 * Check if current request is for admin area
 */
function is_admin_area() {
    $uri = $_SERVER['REQUEST_URI'] ?? '';
    return strpos($uri, '/admin/') !== false;
}

/**
 * Check if current request is AJAX
 */
function is_ajax() {
    return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
           strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
}

/**
 * Formatiert Datum im deutschen Format
 */
function format_date($date, $format = 'd.m.Y') {
    if (empty($date)) return '';
    
    $timestamp = is_numeric($date) ? $date : strtotime($date);
    return date($format, $timestamp);
}

/**
 * Formatiert Datum mit Uhrzeit
 */
function format_datetime($datetime) {
    return format_date($datetime, 'd.m.Y H:i');
}

/**
 * Kürzt Text auf bestimmte Länge
 */
function truncate($text, $length = 100, $suffix = '...') {
    if (mb_strlen($text) <= $length) {
        return $text;
    }
    
    return mb_substr($text, 0, $length) . $suffix;
}

/**
 * Generiert zufälligen Token
 */
function generate_token($length = 32) {
    return bin2hex(random_bytes($length / 2));
}

/**
 * Sicheres Array-Get mit Default-Wert
 */
function array_get($array, $key, $default = null) {
    if (!is_array($array)) {
        return $default;
    }
    
    return $array[$key] ?? $default;
}

/**
 * Prüft ob String mit bestimmtem Wert startet
 */
function starts_with($haystack, $needle) {
    return strpos($haystack, $needle) === 0;
}

/**
 * Prüft ob String mit bestimmtem Wert endet
 */
function ends_with($haystack, $needle) {
    return substr($haystack, -strlen($needle)) === $needle;
}

/**
 * Generiert zufälligen String
 */
function random_string($length = 16) {
    return bin2hex(random_bytes($length / 2));
}

/**
 * Debug-Funktion: Gibt Variable aus und beendet Script
 */
function dd($var) {
    echo '<pre style="background:#1e1e1e;color:#00ff00;padding:20px;margin:20px;border-radius:8px;font-family:monospace;font-size:14px;">';
    var_dump($var);
    echo '</pre>';
    exit;
}

/**
 * Debug-Funktion: Gibt Variable aus ohne Script zu beenden
 */
function dump($var) {
    echo '<pre style="background:#1e1e1e;color:#00ff00;padding:20px;margin:20px;border-radius:8px;font-family:monospace;font-size:14px;">';
    var_dump($var);
    echo '</pre>';
}

/**
 * Überprüft ob User eingeloggt ist
 */
function is_logged_in() {
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

/**
 * Holt aktuellen User aus Session
 */
if (!function_exists('current_user')) {
    function current_user() {
        if (!isset($_SESSION['user_id'])) {
            return null;
        }
        
        $user = db()->fetchOne(
            "SELECT id, username, role, email FROM users WHERE id = ?",
            [$_SESSION['user_id']]
        );
        
        return $user;
    }
}

/**
 * Überprüft ob User Admin ist
 */
if (!function_exists('is_admin')) {
    function is_admin() {
        $user = current_user();
        return $user && in_array($user['role'], ['admin', 'super_admin']);
    }
}

/**
 * Erzwingt Login (redirect zu Login-Seite)
 */
if (!function_exists('require_login')) {
    function require_login() {
        if (!is_logged_in()) {
            $_SESSION['redirect_after_login'] = $_SERVER['REQUEST_URI'];
            redirect('/admin/login.php');
        }
    }
}

/**
 * Erzwingt Admin-Rechte
 */
if (!function_exists('require_admin')) {
    function require_admin() {
        require_login();
        
        if (!is_admin()) {
            http_response_code(403);
            die('Zugriff verweigert. Admin-Rechte erforderlich.');
        }
    }
}

/**
 * Flash-Message setzen
 */
function set_flash($type, $message) {
    $_SESSION['flash'][] = [
        'type' => $type, // success, error, warning, info
        'message' => $message
    ];
}

/**
 * Prüft ob Flash-Messages vorhanden sind
 */
function has_flash($type = null) {
    $messages = $_SESSION['flash'] ?? [];
    
    if ($type === null) {
        return !empty($messages);
    }
    
    foreach ($messages as $msg) {
        if ($msg['type'] === $type) {
            return true;
        }
    }
    
    return false;
}

/**
 * Flash-Messages abrufen und löschen
 */
function get_flash() {
    $messages = $_SESSION['flash'] ?? [];
    unset($_SESSION['flash']);
    return $messages;
}

/**
 * Holt Setting aus Datenbank
 */
function get_setting($key, $default = null) {
    $setting = db()->fetchOne(
        "SELECT setting_value, setting_type FROM settings WHERE setting_key = ?",
        [$key]
    );
    
    if (!$setting) {
        return $default;
    }
    
    $value = $setting['setting_value'];
    
    // Type-Casting
    if ($setting['setting_type'] === 'boolean') {
        return filter_var($value, FILTER_VALIDATE_BOOLEAN);
    } elseif ($setting['setting_type'] === 'json') {
        return json_decode($value, true);
    }
    
    return $value;
}

/**
 * Setzt Setting in Datenbank
 */
function set_setting($key, $value, $type = 'text') {
    // Convert boolean to string
    if ($type === 'boolean') {
        $value = $value ? '1' : '0';
    } elseif ($type === 'json' && is_array($value)) {
        $value = json_encode($value);
    }
    
    // Check if exists
    $existing = db()->fetchOne(
        "SELECT id FROM settings WHERE setting_key = ?",
        [$key]
    );
    
    if ($existing) {
        // Update
        return db()->execute(
            "UPDATE settings SET setting_value = ?, setting_type = ? WHERE setting_key = ?",
            [$value, $type, $key]
        );
    } else {
        // Insert
        return db()->insert(
            "INSERT INTO settings (setting_key, setting_value, setting_type) VALUES (?, ?, ?)",
            [$key, $value, $type]
        );
    }
}

/**
 * File Upload Handler with Automatic Optimization
 */
function handle_file_upload($fileInputName, $uploadDir, $allowedTypes = null, $optimize = true) {
    if (!isset($_FILES[$fileInputName]) || $_FILES[$fileInputName]['error'] === UPLOAD_ERR_NO_FILE) {
        return ['success' => false, 'error' => 'Keine Datei hochgeladen'];
    }
    
    $file = $_FILES[$fileInputName];
    
    if ($file['error'] !== UPLOAD_ERR_OK) {
        return ['success' => false, 'error' => 'Upload-Fehler: ' . $file['error']];
    }
    
    // Check file size
    if (defined('MAX_UPLOAD_SIZE') && $file['size'] > MAX_UPLOAD_SIZE) {
        return ['success' => false, 'error' => 'Datei zu groß (max ' . (MAX_UPLOAD_SIZE / 1024 / 1024) . ' MB)'];
    }
    
    // Check file type
    if ($allowedTypes === null && defined('ALLOWED_IMAGE_TYPES')) {
        $allowedTypes = ALLOWED_IMAGE_TYPES;
    }
    
    // Convert string to array if necessary
    if (is_string($allowedTypes)) {
        $allowedTypes = explode(',', $allowedTypes);
        $allowedTypes = array_map('trim', $allowedTypes);
    }
    
    $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    
    if ($allowedTypes && !in_array($extension, $allowedTypes)) {
        return ['success' => false, 'error' => 'Dateityp nicht erlaubt. Erlaubte Typen: ' . implode(', ', $allowedTypes)];
    }
    
    // Generate unique filename
    $filename = uniqid() . '_' . time() . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '', basename($file['name']));
    $destination = $uploadDir . '/' . $filename;
    
    // Create directory if doesn't exist
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    // Move file
    if (!move_uploaded_file($file['tmp_name'], $destination)) {
        return ['success' => false, 'error' => 'Fehler beim Speichern der Datei'];
    }
    
    $result = [
        'success' => true, 
        'filename' => $filename, 
        'path' => $destination,
        'size' => filesize($destination)
    ];
    
    // Optimize images if enabled
    if ($optimize && in_array($extension, ['jpg', 'jpeg', 'png', 'gif'])) {
        require_once BACKEND_ROOT . '/performance/image_optimizer.php';
        
        $optimization = ImageOptimizer::processUpload($destination);
        $result['optimization'] = $optimization;
        
        // If WebP was created, return WebP filename as primary
        if (!empty($optimization['webp'])) {
            $result['webp_filename'] = $optimization['webp'];
            $result['original_filename'] = $filename;
            
            // Optional: Use WebP as default filename
            if (get_setting('prefer_webp', true)) {
                $result['filename'] = $optimization['webp'];
            }
        }
    }
    
    return $result;
}

/**
 * Delete file securely
 */
function delete_file($filepath) {
    if (file_exists($filepath) && is_file($filepath)) {
        return unlink($filepath);
    }
    return false;
}

/**
 * Render RichTextField data to HTML
 * SMART: nl2br only for paragraph tags, not for headings
 */
function render_richtext($data, $tag = 'div') {
    // Handle old string format
    if (is_string($data)) {
        return "<{$tag}>" . nl2br(e($data)) . "</{$tag}>";
    }
    
    // Handle new array format from RichTextField
    if (!is_array($data)) {
        return '';
    }
    
    // Extract values
    $text = $data['text'] ?? '';
    $fontSize = isset($data['fontSize']) ? (int)$data['fontSize'] : 0;
    $bold = !empty($data['bold']) && $data['bold'] !== '0';
    $italic = !empty($data['italic']) && $data['italic'] !== '0';
    $underline = !empty($data['underline']) && $data['underline'] !== '0';
    $color = $data['color'] ?? '';
    $alignment = $data['alignment'] ?? 'left';
    $link = $data['link'] ?? '';
    
    // Trim text to remove leading/trailing whitespace
    $text = trim($text);
    
    // Skip if no text
    if (empty($text)) {
        return '';
    }
    
    // Build inline style
    $styles = [];
    $styles[] = "text-align: {$alignment}";
    
    // Font size
    if ($fontSize !== 0) {
        $styles[] = "font-size: calc(1em + {$fontSize} * 0.15em)";
    }
    
    if ($bold) {
        $styles[] = "font-weight: bold";
    }
    
    if ($italic) {
        $styles[] = "font-style: italic";
    }
    
    if ($underline) {
        $styles[] = "text-decoration: underline";
    }
    
    if ($color) {
        $styles[] = "color: {$color}";
    }
    
    $styleAttr = !empty($styles) ? ' style="' . implode('; ', $styles) . '"' : '';
    
    // Escape text
    $formattedText = e($text);
    
    // SMART nl2br: Only for paragraph/div/span tags, NOT for headings
    // This prevents unwanted <br> tags in H1, H2, H3
    $paragraphTags = ['p', 'div', 'span', 'td', 'li', 'article', 'section'];
    if (in_array(strtolower($tag), $paragraphTags)) {
        $formattedText = nl2br($formattedText);
    }
    
    // Wrap in link if provided
    if ($link) {
        $formattedText = '<a href="' . e($link) . '">' . $formattedText . '</a>';
    }
    
    // Return wrapped in tag
    return "<{$tag}{$styleAttr}>{$formattedText}</{$tag}>";
}

/**
 * Check if data is RichTextField format
 */
function is_richtext_data($data) {
    return is_array($data) && isset($data['text']);
}

/**
 * Extract plain text from RichTextField data
 * Returns trimmed text without formatting
 */
function get_richtext_plain($data) {
    if (is_string($data)) {
        return trim($data);
    }
    
    if (is_array($data) && isset($data['text'])) {
        return trim($data['text']);
    }
    
    return '';
}

/**
 * Get menu items for navigation
 */
function get_menu_items() {
    if (class_exists('MenuManager')) {
        return MenuManager::getMenuPagesWithHierarchy();
    }
    return [];
}

/**
 * Get menu items (alias for backwards compatibility)
 */
function get_navigation_menu() {
    return get_menu_items();
}