<?php
/** cms-core/backend/core/init.php - OPTIMIZED WITH TRANSLATION SYSTEM */

// ============================================
// CRITICAL FIX: Start Output Buffering FIRST
// ============================================
if (ob_get_level() === 0) {
    ob_start();
}

// ============================================
// STEP 1: Define APP_ROOT (Project Root Directory)
// ============================================
if (!defined('APP_ROOT')) {
    define('APP_ROOT', dirname(dirname(dirname(__DIR__))));
}

// ============================================
// STEP 2: Load Config (which defines all other paths)
// ============================================
require_once APP_ROOT . '/cms-core/config/config.php';

// Also load database.php if not already loaded
if (!class_exists('Database')) {
    require_once APP_ROOT . '/cms-core/config/database.php';
}

// ============================================
// STEP 2.5: Load Performance Modules EARLY
// ============================================
$performanceDir = APP_ROOT . '/cms-core/backend/performance/';

// Load QueryCache (in-memory cache)
if (file_exists($performanceDir . 'QueryCache.php')) {
    require_once $performanceDir . 'QueryCache.php';
}

// Load PersistentCache (file-based cache)
if (file_exists($performanceDir . 'PersistentCache.php')) {
    require_once $performanceDir . 'PersistentCache.php';
    PersistentCache::init();
}

// Load AdvancedCacheManager (Redis, Memcached, APCu) - NEW!
if (file_exists($performanceDir . 'AdvancedCacheManager.php')) {
    require_once $performanceDir . 'AdvancedCacheManager.php';
    // Will auto-init when first used
}

// Load PerformanceOptimizer
if (file_exists($performanceDir . 'PerformanceOptimizer.php')) {
    require_once $performanceDir . 'PerformanceOptimizer.php';
    PerformanceOptimizer::init();
}

// ============================================
// STEP 2.8: Load Translation System 🌍
// ============================================
if (file_exists(__DIR__ . '/Translation.php')) {
    require_once __DIR__ . '/Translation.php';
    Translation::init();
}

// ============================================
// STEP 3: Load Core Functions
// ============================================
require_once __DIR__ . '/helpers.php';
require_once __DIR__ . '/PageManager.php';
require_once __DIR__ . '/MenuManager.php';
require_once __DIR__ . '/LegalPagesManager.php';
require_once __DIR__ . '/../settings/site_info_manager.php';
require_once __DIR__ . '/../settings/farb_font.php';

// Optional: Flexible Blocks
if (file_exists(__DIR__ . '/FlexibleBlocks.php')) {
    require_once __DIR__ . '/FlexibleBlocks.php';
}

// Optional: RichTextField
if (file_exists(__DIR__ . '/RichTextField.php')) {
    require_once __DIR__ . '/RichTextField.php';
}

// ============================================
// CHECK FOR ACCIDENTAL OUTPUT
// ============================================
$accidentalOutput = ob_get_contents();
if (!empty(trim($accidentalOutput))) {
    error_log("WARNING: " . strlen($accidentalOutput) . " bytes of output detected before session start");
    ob_clean();
}

// ============================================
// STEP 4: Load Security Modules
// ============================================
require_once APP_ROOT . '/cms-core/backend/security/csrf.php';
require_once APP_ROOT . '/cms-core/backend/security/xss.php';
require_once APP_ROOT . '/cms-core/backend/security/validation.php';
require_once APP_ROOT . '/cms-core/backend/security/csp.php';

// ============================================
// STEP 4.5: Load User Roles System
// ============================================
if (file_exists(APP_ROOT . '/cms-core/backend/auth/UserRoles.php')) {
    require_once APP_ROOT . '/cms-core/backend/auth/UserRoles.php';
}

// ============================================
// STEP 5: Load Session & Auth (BEFORE SecurityMonitor!)
// ============================================
require_once APP_ROOT . '/cms-core/backend/auth/session.php';

// ============================================
// STEP 6: Load Error Handlers
// ============================================
require_once APP_ROOT . '/cms-core/backend/error/error_handler.php';
require_once APP_ROOT . '/cms-core/backend/error/logger.php';

// ============================================
// STEP 7: Load Maintenance Check
// ============================================
require_once __DIR__ . '/maintenance_check.php';

// ============================================
// STEP 8: Start Session (NOW SAFE!)
// ============================================
if (session_status() === PHP_SESSION_NONE) {
    SessionManager::start();
}

// ============================================
// STEP 9: Load SecurityMonitor (AFTER Session!)
// ============================================
require_once APP_ROOT . '/cms-core/backend/security/SecurityMonitor.php';

// Optional: Request Monitoring aktivieren (AFTER session is started!)
if (getenv('APP_ENV') === 'production') {
    // Disable for now until we're sure it works
    // SecurityMonitor::monitorRequest();
}

// ============================================
// STEP 10: Set Security Headers (NOW SAFE!)
// ============================================
CSP::setHeaders();

// ============================================
// STEP 11: Register Error Handler
// ============================================
ErrorHandler::register();

// ============================================
// STEP 12: Check Maintenance Mode (Frontend Only)
// ============================================
$requestUri = $_SERVER['REQUEST_URI'] ?? '';
$isAdminArea = strpos($requestUri, '/admin/') !== false;

if (!$isAdminArea) {
    check_maintenance_mode();
}

// ============================================
// STEP 13: Load Custom Features
// ============================================
if (defined('CUSTOM_FEATURES_PATH') && is_dir(CUSTOM_FEATURES_PATH)) {
    foreach (glob(CUSTOM_FEATURES_PATH . '/*.php') as $featureFile) {
        require_once $featureFile;
        // Only log in debug mode to avoid log spam
        if (defined('APP_DEBUG') && APP_DEBUG) {
            Logger::debug('Loaded custom feature: ' . basename($featureFile));
        }
    }
}

$customConfigPath = APP_ROOT . '/customizations/config/custom.php';
if (file_exists($customConfigPath)) {
    require_once $customConfigPath;
    Logger::info('Loaded custom config');
}

// ============================================
// STEP 14: Cleanup expired cache (occasionally)
// ============================================
// Run cache cleanup 1% of the time to keep cache fresh
if (class_exists('PersistentCache') && mt_rand(1, 100) === 1) {
    PersistentCache::cleanExpired();
}

// ============================================
// OUTPUT BUFFERING REMAINS ACTIVE
// ============================================
// The buffer will be flushed automatically at the end of script execution
// This ensures no premature output breaks sessions or headers