<?php
/** cms-core/backend/encryption/password_utils.php */

/**
 * Password Utilities
 * Sichere Passwort-Verarbeitung
 */

class PasswordUtils {
    
    /**
     * Hash-Algorithmus
     */
    const ALGORITHM = PASSWORD_ARGON2ID;
    
    /**
     * Erstellt Passwort-Hash
     */
    public static function hash($password) {
        return password_hash($password, self::ALGORITHM, [
            'memory_cost' => 65536,
            'time_cost' => 4,
            'threads' => 3
        ]);
    }
    
    /**
     * Verifiziert Passwort gegen Hash
     */
    public static function verify($password, $hash) {
        return password_verify($password, $hash);
    }
    
    /**
     * Prüft ob Hash neu erstellt werden muss (Rehashing)
     */
    public static function needsRehash($hash) {
        return password_needs_rehash($hash, self::ALGORITHM);
    }
    
    /**
     * Generiert sicheres Zufalls-Passwort
     */
    public static function generate($length = 16) {
        $chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789!@#$%^&*';
        $password = '';
        $max = strlen($chars) - 1;
        
        for ($i = 0; $i < $length; $i++) {
            $password .= $chars[random_int(0, $max)];
        }
        
        return $password;
    }
    
    /**
     * Validiert Passwort-Stärke
     */
    public static function validateStrength($password, $minLength = 8) {
        $errors = [];
        
        if (strlen($password) < $minLength) {
            $errors[] = "Passwort muss mindestens $minLength Zeichen lang sein.";
        }
        
        if (!preg_match('/[A-Z]/', $password)) {
            $errors[] = "Passwort muss mindestens einen Großbuchstaben enthalten.";
        }
        
        if (!preg_match('/[a-z]/', $password)) {
            $errors[] = "Passwort muss mindestens einen Kleinbuchstaben enthalten.";
        }
        
        if (!preg_match('/[0-9]/', $password)) {
            $errors[] = "Passwort muss mindestens eine Zahl enthalten.";
        }
        
        if (!preg_match('/[!@#$%^&*(),.?":{}|<>]/', $password)) {
            $errors[] = "Passwort sollte mindestens ein Sonderzeichen enthalten.";
        }
        
        return [
            'valid' => empty($errors),
            'errors' => $errors
        ];
    }
}