<?php
/** cms-core/backend/error/error_handler.php - FIXED */

/**
 * Error Handler
 * Zentrale Fehlerbehandlung
 */

class ErrorHandler {
    
    /**
     * Registriert Error Handler
     */
    public static function register() {
        set_error_handler([self::class, 'handleError']);
        set_exception_handler([self::class, 'handleException']);
        register_shutdown_function([self::class, 'handleShutdown']);
    }
    
    /**
     * Check if debug mode is enabled
     * Handles multiple formats: true, 'true', 1, '1', yes, 'yes'
     */
    private static function isDebugMode() {
        // Check constant first (set by config.php)
        if (defined('APP_DEBUG')) {
            return APP_DEBUG === true || APP_DEBUG === 1;
        }
        
        // Fallback to environment variable
        $debug = getenv('APP_DEBUG');
        
        // Handle various formats
        if ($debug === true || $debug === 1) {
            return true;
        }
        
        if (is_string($debug)) {
            $debug = strtolower(trim($debug));
            return in_array($debug, ['true', '1', 'yes', 'on'], true);
        }
        
        return false;
    }
    
    /**
     * Behandelt PHP-Fehler
     */
    public static function handleError($errno, $errstr, $errfile, $errline) {
        // Ignoriere unterdrückte Fehler (@)
        if (!(error_reporting() & $errno)) {
            return false;
        }
        
        $errorTypes = [
            E_ERROR => 'ERROR',
            E_WARNING => 'WARNING',
            E_PARSE => 'PARSE ERROR',
            E_NOTICE => 'NOTICE',
            E_CORE_ERROR => 'CORE ERROR',
            E_CORE_WARNING => 'CORE WARNING',
            E_COMPILE_ERROR => 'COMPILE ERROR',
            E_COMPILE_WARNING => 'COMPILE WARNING',
            E_USER_ERROR => 'USER ERROR',
            E_USER_WARNING => 'USER WARNING',
            E_USER_NOTICE => 'USER NOTICE',
            E_STRICT => 'STRICT',
            E_RECOVERABLE_ERROR => 'RECOVERABLE ERROR',
            E_DEPRECATED => 'DEPRECATED',
            E_USER_DEPRECATED => 'USER DEPRECATED'
        ];
        
        $type = $errorTypes[$errno] ?? 'UNKNOWN';
        
        // Logge Fehler
        if (class_exists('Logger')) {
            Logger::error("[$type] $errstr in $errfile:$errline");
        } else {
            error_log("[$type] $errstr in $errfile:$errline");
        }
        
        // Bei kritischen Fehlern, zeige Fehlerseite
        if (in_array($errno, [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
            self::showErrorPage('Ein kritischer Fehler ist aufgetreten.');
        }
        
        return true;
    }
    
    /**
     * Behandelt Exceptions
     */
    public static function handleException($exception) {
        // Logge Exception
        $logMessage = 'Exception: ' . $exception->getMessage() . 
                     ' in ' . $exception->getFile() . ':' . $exception->getLine() .
                     "\nStack trace:\n" . $exception->getTraceAsString();
        
        if (class_exists('Logger')) {
            Logger::error($logMessage);
        } else {
            error_log($logMessage);
        }
        
        // Zeige Fehlerseite
        self::showErrorPage('Ein Fehler ist aufgetreten.');
    }
    
    /**
     * Behandelt Fatal Errors beim Shutdown
     */
    public static function handleShutdown() {
        $error = error_get_last();
        
        if ($error !== null && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
            // Logge Fatal Error
            $logMessage = 'Fatal Error: ' . $error['message'] . 
                         ' in ' . $error['file'] . ':' . $error['line'];
            
            if (class_exists('Logger')) {
                Logger::error($logMessage);
            } else {
                error_log($logMessage);
            }
            
            // Zeige Fehlerseite (falls nicht bereits output gesendet)
            if (!headers_sent()) {
                self::showErrorPage('Ein kritischer Fehler ist aufgetreten.');
            }
        }
    }
    
    /**
     * Zeigt Fehlerseite
     */
    private static function showErrorPage($message) {
        http_response_code(500);
        
        // Im Debug-Modus zeige Details (FIXED: proper debug check)
        if (self::isDebugMode()) {
            return; // PHP zeigt eigene Fehlermeldungen
        }
        
        // Production: Generische Fehlerseite
        echo '<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Fehler</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        .error-container {
            background: white;
            padding: 60px 40px;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            max-width: 500px;
            text-align: center;
        }
        .error-icon {
            font-size: 80px;
            margin-bottom: 20px;
        }
        h1 {
            color: #333;
            margin-bottom: 15px;
            font-size: 28px;
        }
        p {
            color: #666;
            line-height: 1.6;
            margin-bottom: 30px;
        }
        .btn {
            display: inline-block;
            background: #667eea;
            color: white;
            padding: 12px 30px;
            border-radius: 25px;
            text-decoration: none;
            transition: transform 0.2s;
        }
        .btn:hover {
            transform: translateY(-2px);
            background: #5568d3;
        }
    </style>
</head>
<body>
    <div class="error-container">
        <div class="error-icon">⚠️</div>
        <h1>Oops! Ein Fehler ist aufgetreten</h1>
        <p>' . htmlspecialchars($message) . '</p>
        <p>Bitte versuchen Sie es später erneut oder kontaktieren Sie den Support.</p>
        <a href="/" class="btn">← Zurück zur Startseite</a>
    </div>
</body>
</html>';
        exit;
    }
}