<?php
/** cms-core/backend/error/logger.php */

/**
 * Logger
 * Schreibt Logs in Dateien
 */

class Logger {
    
    /**
     * Log-Level
     */
    const DEBUG = 'DEBUG';
    const INFO = 'INFO';
    const WARNING = 'WARNING';
    const ERROR = 'ERROR';
    const CRITICAL = 'CRITICAL';
    
    /**
     * Log-Datei
     */
    private static $logFile = null;
    
    /**
     * Initialisiert Logger
     */
    private static function init() {
    if (self::$logFile === null) {
        // CRITICAL: Use /logs/ outside cms-core!
        $logDir = defined('LOGS_PATH') ? LOGS_PATH : APP_ROOT . '/logs';
        
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
        
        self::$logFile = $logDir . '/error.log';
    }
}
    
    /**
     * Schreibt Log-Eintrag
     */
    private static function write($level, $message, $context = []) {
        self::init();
        
        // Formatiere Log-Eintrag
        $timestamp = date('Y-m-d H:i:s');
        $contextStr = !empty($context) ? ' ' . json_encode($context) : '';
        $logEntry = "[$timestamp] [$level] $message$contextStr\n";
        
        // Schreibe in Datei
        file_put_contents(self::$logFile, $logEntry, FILE_APPEND | LOCK_EX);
    }
    
    /**
     * Debug-Log
     */
    public static function debug($message, $context = []) {
        if (getenv('APP_DEBUG') === 'true') {
            self::write(self::DEBUG, $message, $context);
        }
    }
    
    /**
     * Info-Log
     */
    public static function info($message, $context = []) {
        self::write(self::INFO, $message, $context);
    }
    
    /**
     * Warning-Log
     */
    public static function warning($message, $context = []) {
        self::write(self::WARNING, $message, $context);
    }
    
    /**
     * Error-Log
     */
    public static function error($message, $context = []) {
        self::write(self::ERROR, $message, $context);
    }
    
    /**
     * Critical-Log
     */
    public static function critical($message, $context = []) {
        self::write(self::CRITICAL, $message, $context);
    }
    
    /**
     * Loggt Login-Versuche
     */
    public static function logLogin($username, $success, $ip = null) {
        $ip = $ip ?: ($_SERVER['REMOTE_ADDR'] ?? 'unknown');
        $status = $success ? 'SUCCESS' : 'FAILED';
        
        self::info("Login $status for user '$username' from IP $ip");
    }
    
    /**
     * Loggt Datei-Operationen
     */
    public static function logFileOperation($operation, $file, $user = null) {
        $userStr = $user ? " by user $user" : '';
        self::info("File $operation: $file$userStr");
    }
    
    /**
     * Löscht alte Logs (älter als X Tage)
     */
    public static function cleanup($days = 30) {
        self::init();
        
        $logFiles = glob(dirname(self::$logFile) . '/*.log');
        $cutoffTime = time() - ($days * 86400);
        
        foreach ($logFiles as $file) {
            if (filemtime($file) < $cutoffTime) {
                unlink($file);
            }
        }
    }
}