<?php
/** cms-core/backend/integrations/integration_helper.php */

/**
 * Integration Helper - ENTERPRISE EDITION
 * Verwaltet externe Integrationen + Performance Caching
 */

class IntegrationHelper {
    
    /**
     * Prüft ob Integration aktiv ist
     */
    public static function isActive($integration) {
        $settingKey = $integration . '_enabled';
        return get_setting($settingKey, false);
    }
    
    /**
     * Aktiviert Integration
     */
    public static function enable($integration) {
        $settingKey = $integration . '_enabled';
        return set_setting($settingKey, true, 'boolean');
    }
    
    /**
     * Deaktiviert Integration
     */
    public static function disable($integration) {
        $settingKey = $integration . '_enabled';
        return set_setting($settingKey, false, 'boolean');
    }
    
    /**
     * Holt Integration-Konfiguration
     */
    public static function getConfig($integration) {
        $config = [];
        
        switch ($integration) {
            case 'google_analytics':
                $config['tracking_id'] = get_setting('google_analytics_id', '');
                $config['enabled'] = self::isActive('google_analytics');
                break;
                
            case 'tiktok_pixel':
                $config['pixel_id'] = get_setting('tiktok_pixel_id', '');
                $config['enabled'] = self::isActive('tiktok_pixel');
                break;
                
            case 'instagram_pixel':
                $config['pixel_id'] = get_setting('instagram_pixel_id', '');
                $config['enabled'] = self::isActive('instagram_pixel');
                break;
                
            case 'deepl':
                $config['api_key'] = get_setting('deepl_api_key', '');
                break;

            case 'mailchimp':
                $config['api_key'] = get_setting('mailchimp_api_key', '');
                $config['list_id'] = get_setting('mailchimp_list_id', '');
                $config['enabled'] = self::isActive('mailchimp');
                break;
                
            case 'cloudflare_cdn':
                $config['cdn_url'] = get_setting('cloudflare_cdn_url', '');
                $config['cdn_media_only'] = get_setting('cloudflare_cdn_media_only', true);
                $config['enabled'] = self::isActive('cloudflare_cdn');
                break;
                
            case 'webp_optimization':
                $config['max_image_width'] = get_setting('max_image_width', 2000);
                $config['webp_quality'] = get_setting('webp_quality', 85);
                $config['prefer_webp'] = get_setting('prefer_webp', true);
                $config['enabled'] = get_setting('auto_convert_webp', true);
                break;
                
            case 'redis_cache':
                $config['host'] = get_setting('redis_cache_host', '127.0.0.1');
                $config['port'] = get_setting('redis_cache_port', 6379);
                $config['password'] = get_setting('redis_cache_password', '');
                $config['ttl'] = get_setting('redis_cache_ttl', 3600);
                $config['enabled'] = self::isActive('redis_cache');
                break;
                
            case 'memcached':
                $config['host'] = get_setting('memcached_host', '127.0.0.1');
                $config['port'] = get_setting('memcached_port', 11211);
                $config['ttl'] = get_setting('memcached_ttl', 3600);
                $config['enabled'] = self::isActive('memcached');
                break;
                
            case 'varnish_cache':
                $config['host'] = get_setting('varnish_cache_host', '127.0.0.1');
                $config['port'] = get_setting('varnish_cache_port', 6081);
                $config['enabled'] = self::isActive('varnish_cache');
                break;
                
            case 'elasticsearch':
                $config['host'] = get_setting('elasticsearch_host', 'localhost');
                $config['port'] = get_setting('elasticsearch_port', 9200);
                $config['index'] = get_setting('elasticsearch_index', 'cms_content');
                $config['enabled'] = self::isActive('elasticsearch');
                break;
        }
        
        return $config;
    }
    
    /**
     * Speichert Integration-Konfiguration
     */
    public static function saveConfig($integration, $config) {
        foreach ($config as $key => $value) {
            $settingKey = $integration . '_' . $key;
            $type = is_bool($value) ? 'boolean' : 'text';
            set_setting($settingKey, $value, $type);
        }
        
        return true;
    }
    
    /**
     * Test Integration Connection
     */
    public static function testConnection($integration) {
        $config = self::getConfig($integration);
        
        switch ($integration) {
            case 'redis_cache':
                if (!class_exists('Redis')) {
                    return ['success' => false, 'message' => 'Redis PHP Extension nicht installiert'];
                }
                
                try {
                    $redis = new Redis();
                    $redis->connect($config['host'], $config['port'], 2);
                    
                    if (!empty($config['password'])) {
                        $redis->auth($config['password']);
                    }
                    
                    $redis->ping();
                    return ['success' => true, 'message' => 'Verbindung erfolgreich'];
                } catch (Exception $e) {
                    return ['success' => false, 'message' => 'Verbindung fehlgeschlagen: ' . $e->getMessage()];
                }
                break;
                
            case 'memcached':
                if (!class_exists('Memcached')) {
                    return ['success' => false, 'message' => 'Memcached PHP Extension nicht installiert'];
                }
                
                try {
                    $memcached = new Memcached();
                    $memcached->addServer($config['host'], $config['port']);
                    $memcached->set('test_key', 'test_value', 10);
                    
                    if ($memcached->get('test_key') === 'test_value') {
                        return ['success' => true, 'message' => 'Verbindung erfolgreich'];
                    }
                    
                    return ['success' => false, 'message' => 'Verbindung fehlgeschlagen'];
                } catch (Exception $e) {
                    return ['success' => false, 'message' => $e->getMessage()];
                }
                break;
                
            case 'elasticsearch':
                try {
                    $url = "http://{$config['host']}:{$config['port']}";
                    $response = @file_get_contents($url);
                    
                    if ($response !== false) {
                        return ['success' => true, 'message' => 'Verbindung erfolgreich'];
                    }
                    
                    return ['success' => false, 'message' => 'Verbindung fehlgeschlagen'];
                } catch (Exception $e) {
                    return ['success' => false, 'message' => $e->getMessage()];
                }
                break;
                
            default:
                return ['success' => false, 'message' => 'Test nicht verfügbar'];
        }
    }
    
    /**
     * Gibt alle verfügbaren Integrationen zurück
     */
    public static function getAvailable() {
        return [
            // ========== PERFORMANCE & CACHING ==========
            'redis_cache' => [
                'name' => 'Redis Cache',
                'description' => 'Ultra-schneller In-Memory Cache (empfohlen für High-Traffic)',
                'icon' => '🔴',
                'category' => 'performance',
                'config_fields' => [
                    'host' => [
                        'label' => 'Redis Host',
                        'placeholder' => '127.0.0.1',
                        'default' => '127.0.0.1'
                    ],
                    'port' => [
                        'label' => 'Redis Port',
                        'type' => 'number',
                        'placeholder' => '6379',
                        'default' => 6379
                    ],
                    'password' => [
                        'label' => 'Password (optional)',
                        'type' => 'password',
                        'placeholder' => 'Leer lassen wenn kein Password'
                    ],
                    'ttl' => [
                        'label' => 'Cache TTL (Sekunden)',
                        'type' => 'number',
                        'default' => 3600,
                        'description' => 'Cache-Lebensdauer in Sekunden (Standard: 3600 = 1 Stunde)'
                    ]
                ]
            ],
            'memcached' => [
                'name' => 'Memcached',
                'description' => 'Distributed Object Caching System',
                'icon' => '🐘',
                'category' => 'performance',
                'config_fields' => [
                    'host' => [
                        'label' => 'Memcached Host',
                        'placeholder' => '127.0.0.1',
                        'default' => '127.0.0.1'
                    ],
                    'port' => [
                        'label' => 'Memcached Port',
                        'type' => 'number',
                        'placeholder' => '11211',
                        'default' => 11211
                    ],
                    'ttl' => [
                        'label' => 'Cache TTL (Sekunden)',
                        'type' => 'number',
                        'default' => 3600
                    ]
                ]
            ],
            'varnish_cache' => [
                'name' => 'Varnish Cache',
                'description' => 'HTTP Accelerator für extreme Performance',
                'icon' => '⚡',
                'category' => 'performance',
                'config_fields' => [
                    'host' => [
                        'label' => 'Varnish Host',
                        'placeholder' => '127.0.0.1',
                        'default' => '127.0.0.1'
                    ],
                    'port' => [
                        'label' => 'Varnish Port',
                        'type' => 'number',
                        'placeholder' => '6081',
                        'default' => 6081
                    ]
                ]
            ],
            'elasticsearch' => [
                'name' => 'Elasticsearch',
                'description' => 'Leistungsstarke Volltext-Suche',
                'icon' => '🔍',
                'category' => 'performance',
                'config_fields' => [
                    'host' => [
                        'label' => 'Elasticsearch Host',
                        'placeholder' => 'localhost',
                        'default' => 'localhost'
                    ],
                    'port' => [
                        'label' => 'Elasticsearch Port',
                        'type' => 'number',
                        'placeholder' => '9200',
                        'default' => 9200
                    ],
                    'index' => [
                        'label' => 'Index Name',
                        'placeholder' => 'cms_content',
                        'default' => 'cms_content'
                    ]
                ]
            ],
            
            // ========== CDN & OPTIMIZATION ==========
            'cloudflare_cdn' => [
                'name' => 'Cloudflare CDN',
                'description' => 'Globales Content Delivery Network - bis zu 3x schneller',
                'icon' => '🌐',
                'category' => 'cdn',
                'config_fields' => [
                    'cdn_url' => [
                        'label' => 'CDN URL (optional)',
                        'placeholder' => 'https://cdn.beispiel.de',
                        'description' => 'Leer lassen für automatische Cloudflare-Konfiguration'
                    ],
                    'cdn_media_only' => [
                        'label' => 'Nur Media-Dateien über CDN',
                        'type' => 'checkbox',
                        'description' => 'Nur Bilder über CDN, nicht CSS/JS (empfohlen)'
                    ]
                ]
            ],
            'webp_optimization' => [
                'name' => 'WebP Auto-Konvertierung',
                'description' => 'Automatische Bildoptimierung beim Upload (bis zu 70% kleiner)',
                'icon' => '🖼️',
                'category' => 'cdn',
                'config_fields' => [
                    'max_image_width' => [
                        'label' => 'Maximale Bildbreite (px)',
                        'type' => 'number',
                        'min' => 800,
                        'max' => 4000,
                        'default' => 2000,
                        'description' => 'Größere Bilder werden automatisch verkleinert'
                    ],
                    'webp_quality' => [
                        'label' => 'WebP Qualität (%)',
                        'type' => 'number',
                        'min' => 60,
                        'max' => 100,
                        'default' => 85,
                        'description' => '85% ist optimal (kaum sichtbarer Unterschied)'
                    ],
                    'prefer_webp' => [
                        'label' => 'WebP bevorzugen',
                        'type' => 'checkbox',
                        'description' => 'Moderne Browser nutzen automatisch WebP (empfohlen)'
                    ]
                ]
            ],
            
            // ========== TRACKING & ANALYTICS ==========
            'google_analytics' => [
                'name' => 'Google Analytics',
                'description' => 'Website-Traffic und Benutzerverhalten analysieren',
                'icon' => '📊',
                'category' => 'tracking',
                'config_fields' => [
                    'tracking_id' => [
                        'label' => 'Tracking ID',
                        'placeholder' => 'G-XXXXXXXXXX'
                    ]
                ]
            ],
            'tiktok_pixel' => [
                'name' => 'TikTok Pixel',
                'description' => 'TikTok Ads Conversion Tracking',
                'icon' => '🎵',
                'category' => 'tracking',
                'config_fields' => [
                    'pixel_id' => [
                        'label' => 'Pixel ID',
                        'placeholder' => 'Deine TikTok Pixel ID'
                    ]
                ]
            ],
            'instagram_pixel' => [
                'name' => 'Instagram Pixel',
                'description' => 'Instagram Ads Tracking',
                'icon' => '📷',
                'category' => 'tracking',
                'config_fields' => [
                    'pixel_id' => [
                        'label' => 'Pixel ID',
                        'placeholder' => 'Deine Instagram Pixel ID'
                    ]
                ]
            ],
            
            // ========== MARKETING ==========
            'mailchimp' => [
                'name' => 'Mailchimp',
                'description' => 'Newsletter-Marketing und E-Mail-Kampagnen',
                'icon' => '📧',
                'category' => 'marketing',
                'config_fields' => [
                    'api_key' => [
                        'label' => 'Mailchimp API Key',
                        'placeholder' => 'abc123def456-us19'
                    ],
                    'list_id' => [
                        'label' => 'Audience/List ID',
                        'placeholder' => 'a1b2c3d4e5'
                    ]
                ]
            ]
        ];
    }
    
    /**
     * Lädt Integration-Script
     */
    public static function loadScript($integration) {
        $integrationFile = dirname(__FILE__) . '/' . $integration . '.php';
        
        if (file_exists($integrationFile)) {
            require_once $integrationFile;
            return true;
        }
        
        return false;
    }
}