<?php
/**
 * NEW FILE: cms-core/backend/performance/AdvancedCacheManager.php
 * Unified Cache Manager mit Support für Redis, Memcached, APCu
 */

class AdvancedCacheManager {
    
    private static $driver = null;
    private static $config = [];
    
    // Available cache drivers
    const DRIVER_FILE = 'file';
    const DRIVER_REDIS = 'redis';
    const DRIVER_MEMCACHED = 'memcached';
    const DRIVER_APCU = 'apcu';
    
    /**
     * Initialize cache with best available driver
     */
    public static function init() {
        // Check which driver to use from settings
        $preferredDriver = get_setting('cache_driver', 'auto');
        
        if ($preferredDriver === 'auto') {
            // Auto-detect best available driver
            self::$driver = self::detectBestDriver();
        } else {
            self::$driver = $preferredDriver;
        }
        
        // Load configuration
        self::$config = [
            'redis' => [
                'host' => get_setting('redis_host', '127.0.0.1'),
                'port' => get_setting('redis_port', 6379),
                'password' => get_setting('redis_password', ''),
                'database' => get_setting('redis_database', 0),
                'timeout' => 2.5,
            ],
            'memcached' => [
                'host' => get_setting('memcached_host', '127.0.0.1'),
                'port' => get_setting('memcached_port', 11211),
            ]
        ];
        
        // Initialize connection
        self::connect();
    }
    
    /**
     * Auto-detect best available cache driver
     */
    private static function detectBestDriver() {
        // Priority: Redis > Memcached > APCu > File
        
        if (extension_loaded('redis')) {
            $redis = new Redis();
            try {
                if (@$redis->connect('127.0.0.1', 6379, 1)) {
                    $redis->close();
                    return self::DRIVER_REDIS;
                }
            } catch (Exception $e) {
                // Redis not available
            }
        }
        
        if (extension_loaded('memcached')) {
            $memcached = new Memcached();
            $memcached->addServer('127.0.0.1', 11211);
            if ($memcached->getVersion() !== false) {
                return self::DRIVER_MEMCACHED;
            }
        }
        
        if (extension_loaded('apcu') && ini_get('apc.enabled')) {
            return self::DRIVER_APCU;
        }
        
        return self::DRIVER_FILE;
    }
    
    /**
     * Connect to cache backend
     */
    private static function connect() {
        switch (self::$driver) {
            case self::DRIVER_REDIS:
                return self::connectRedis();
            case self::DRIVER_MEMCACHED:
                return self::connectMemcached();
            case self::DRIVER_APCU:
            case self::DRIVER_FILE:
                return true; // No connection needed
        }
        return false;
    }
    
    /**
     * Redis connection
     */
    private static $redisConnection = null;
    
    private static function connectRedis() {
        if (self::$redisConnection !== null) {
            return true;
        }
        
        try {
            $redis = new Redis();
            $config = self::$config['redis'];
            
            $connected = $redis->connect(
                $config['host'],
                $config['port'],
                $config['timeout']
            );
            
            if (!$connected) {
                throw new Exception('Redis connection failed');
            }
            
            if (!empty($config['password'])) {
                $redis->auth($config['password']);
            }
            
            if ($config['database'] > 0) {
                $redis->select($config['database']);
            }
            
            self::$redisConnection = $redis;
            return true;
            
        } catch (Exception $e) {
            error_log('Redis connection failed: ' . $e->getMessage());
            // Fallback to file cache
            self::$driver = self::DRIVER_FILE;
            return false;
        }
    }
    
    /**
     * Memcached connection
     */
    private static $memcachedConnection = null;
    
    private static function connectMemcached() {
        if (self::$memcachedConnection !== null) {
            return true;
        }
        
        try {
            $memcached = new Memcached();
            $config = self::$config['memcached'];
            
            $memcached->addServer($config['host'], $config['port']);
            
            // Test connection
            $version = $memcached->getVersion();
            if ($version === false) {
                throw new Exception('Memcached connection failed');
            }
            
            self::$memcachedConnection = $memcached;
            return true;
            
        } catch (Exception $e) {
            error_log('Memcached connection failed: ' . $e->getMessage());
            // Fallback to file cache
            self::$driver = self::DRIVER_FILE;
            return false;
        }
    }
    
    /**
     * Get cached value
     */
    public static function get($key) {
        switch (self::$driver) {
            case self::DRIVER_REDIS:
                return self::getRedis($key);
            case self::DRIVER_MEMCACHED:
                return self::getMemcached($key);
            case self::DRIVER_APCU:
                return self::getAPCu($key);
            case self::DRIVER_FILE:
            default:
                return PersistentCache::get($key);
        }
    }
    
    /**
     * Set cached value
     */
    public static function set($key, $value, $ttl = 3600) {
        switch (self::$driver) {
            case self::DRIVER_REDIS:
                return self::setRedis($key, $value, $ttl);
            case self::DRIVER_MEMCACHED:
                return self::setMemcached($key, $value, $ttl);
            case self::DRIVER_APCU:
                return self::setAPCu($key, $value, $ttl);
            case self::DRIVER_FILE:
            default:
                return PersistentCache::set($key, $value, $ttl);
        }
    }
    
    /**
     * Delete cached value
     */
    public static function delete($key) {
        switch (self::$driver) {
            case self::DRIVER_REDIS:
                return self::$redisConnection->del($key);
            case self::DRIVER_MEMCACHED:
                return self::$memcachedConnection->delete($key);
            case self::DRIVER_APCU:
                return apcu_delete($key);
            case self::DRIVER_FILE:
            default:
                return PersistentCache::delete($key);
        }
    }
    
    /**
     * Clear all cache
     */
    public static function clear() {
        switch (self::$driver) {
            case self::DRIVER_REDIS:
                return self::$redisConnection->flushDB();
            case self::DRIVER_MEMCACHED:
                return self::$memcachedConnection->flush();
            case self::DRIVER_APCU:
                return apcu_clear_cache();
            case self::DRIVER_FILE:
            default:
                return PersistentCache::clear();
        }
    }
    
    /**
     * Redis-specific methods
     */
    private static function getRedis($key) {
        $value = self::$redisConnection->get($key);
        return $value !== false ? unserialize($value) : null;
    }
    
    private static function setRedis($key, $value, $ttl) {
        return self::$redisConnection->setex($key, $ttl, serialize($value));
    }
    
    /**
     * Memcached-specific methods
     */
    private static function getMemcached($key) {
        $value = self::$memcachedConnection->get($key);
        return $value !== false ? $value : null;
    }
    
    private static function setMemcached($key, $value, $ttl) {
        return self::$memcachedConnection->set($key, $value, $ttl);
    }
    
    /**
     * APCu-specific methods
     */
    private static function getAPCu($key) {
        $success = false;
        $value = apcu_fetch($key, $success);
        return $success ? $value : null;
    }
    
    private static function setAPCu($key, $value, $ttl) {
        return apcu_store($key, $value, $ttl);
    }
    
    /**
     * Get cache statistics
     */
    public static function getStats() {
        $stats = [
            'driver' => self::$driver,
            'driver_name' => self::getDriverName(),
        ];
        
        switch (self::$driver) {
            case self::DRIVER_REDIS:
                if (self::$redisConnection) {
                    $info = self::$redisConnection->info();
                    $stats['memory_used'] = $info['used_memory_human'] ?? 'N/A';
                    $stats['keys'] = self::$redisConnection->dbSize();
                    $stats['hits'] = $info['keyspace_hits'] ?? 0;
                    $stats['misses'] = $info['keyspace_misses'] ?? 0;
                    $stats['uptime'] = $info['uptime_in_seconds'] ?? 0;
                }
                break;
                
            case self::DRIVER_MEMCACHED:
                if (self::$memcachedConnection) {
                    $stats_data = self::$memcachedConnection->getStats();
                    $server = array_keys($stats_data)[0] ?? null;
                    if ($server) {
                        $data = $stats_data[$server];
                        $stats['memory_used'] = self::formatBytes($data['bytes'] ?? 0);
                        $stats['keys'] = $data['curr_items'] ?? 0;
                        $stats['hits'] = $data['get_hits'] ?? 0;
                        $stats['misses'] = $data['get_misses'] ?? 0;
                        $stats['uptime'] = $data['uptime'] ?? 0;
                    }
                }
                break;
                
            case self::DRIVER_APCU:
                $info = apcu_cache_info();
                $stats['memory_used'] = self::formatBytes($info['mem_size'] ?? 0);
                $stats['keys'] = $info['num_entries'] ?? 0;
                $stats['hits'] = $info['num_hits'] ?? 0;
                $stats['misses'] = $info['num_misses'] ?? 0;
                break;
                
            case self::DRIVER_FILE:
                $stats = array_merge($stats, PersistentCache::getStats());
                break;
        }
        
        return $stats;
    }
    
    /**
     * Get current driver name
     */
    public static function getDriverName() {
        $names = [
            self::DRIVER_FILE => 'File Cache',
            self::DRIVER_REDIS => 'Redis',
            self::DRIVER_MEMCACHED => 'Memcached',
            self::DRIVER_APCU => 'APCu',
        ];
        return $names[self::$driver] ?? 'Unknown';
    }
    
    /**
     * Check if driver is available
     */
    public static function isDriverAvailable($driver) {
        switch ($driver) {
            case self::DRIVER_REDIS:
                return extension_loaded('redis');
            case self::DRIVER_MEMCACHED:
                return extension_loaded('memcached');
            case self::DRIVER_APCU:
                return extension_loaded('apcu') && ini_get('apc.enabled');
            case self::DRIVER_FILE:
                return true;
        }
        return false;
    }
    
    /**
     * Test connection to cache backend
     */
    public static function testConnection($driver, $config = []) {
        try {
            switch ($driver) {
                case self::DRIVER_REDIS:
                    $redis = new Redis();
                    $connected = $redis->connect(
                        $config['host'] ?? '127.0.0.1',
                        $config['port'] ?? 6379,
                        2.5
                    );
                    if (!$connected) return false;
                    
                    if (!empty($config['password'])) {
                        $redis->auth($config['password']);
                    }
                    
                    // Test write/read
                    $testKey = 'cms_test_' . time();
                    $redis->set($testKey, 'test');
                    $result = $redis->get($testKey);
                    $redis->del($testKey);
                    $redis->close();
                    
                    return $result === 'test';
                    
                case self::DRIVER_MEMCACHED:
                    $memcached = new Memcached();
                    $memcached->addServer(
                        $config['host'] ?? '127.0.0.1',
                        $config['port'] ?? 11211
                    );
                    
                    // Test connection
                    $version = $memcached->getVersion();
                    if ($version === false) return false;
                    
                    // Test write/read
                    $testKey = 'cms_test_' . time();
                    $memcached->set($testKey, 'test', 10);
                    $result = $memcached->get($testKey);
                    $memcached->delete($testKey);
                    
                    return $result === 'test';
                    
                case self::DRIVER_APCU:
                    if (!function_exists('apcu_store')) return false;
                    
                    $testKey = 'cms_test_' . time();
                    apcu_store($testKey, 'test', 10);
                    $result = apcu_fetch($testKey);
                    apcu_delete($testKey);
                    
                    return $result === 'test';
                    
                case self::DRIVER_FILE:
                    return true;
            }
        } catch (Exception $e) {
            error_log('Cache driver test failed: ' . $e->getMessage());
            return false;
        }
        
        return false;
    }
    
    /**
     * Format bytes to human readable
     */
    private static function formatBytes($bytes) {
        $units = ['B', 'KB', 'MB', 'GB'];
        $i = 0;
        
        while ($bytes >= 1024 && $i < count($units) - 1) {
            $bytes /= 1024;
            $i++;
        }
        
        return round($bytes, 2) . ' ' . $units[$i];
    }
    
    /**
     * Remember pattern with cache
     */
    public static function remember($key, $ttl, $callback) {
        $value = self::get($key);
        
        if ($value !== null) {
            return $value;
        }
        
        $value = $callback();
        self::set($key, $value, $ttl);
        
        return $value;
    }
    
    /**
     * Get current driver
     */
    public static function getDriver() {
        return self::$driver;
    }
}