<?php
/**
 * cms-core/backend/performance/PerformanceOptimizer.php
 * Automatic Performance Optimization
 * 
 * ✅ FIXED: Cache now in ROOT/cache/assets/ instead of cms-core/backend/cache/
 * ✅ FIXED: All URLs now point to /cache/assets/ instead of /cms-core/backend/cache/assets/
 * 
 * Features:
 * - CSS/JS minification
 * - Browser caching headers
 * - GZIP compression
 * - Asset bundling
 * - Image lazy loading
 * 
 * NO USER CONFIGURATION NEEDED - Works automatically!
 */

class PerformanceOptimizer {
    
    private static $cacheDir;
    private static $isProduction;
    
    /**
     * Initialize optimizer
     * ✅ FIXED: Cache location changed to ROOT/cache/assets/
     */
    public static function init() {
        // ✅ FIXED: Cache outside cms-core for update-safety
        self::$cacheDir = APP_ROOT . '/cache/assets/';
        self::$isProduction = (APP_ENV === 'production');
        
        // Create cache directory if not exists
        if (!is_dir(self::$cacheDir)) {
            mkdir(self::$cacheDir, 0755, true);
        }
        
        // Enable GZIP compression if not already enabled
        if (self::$isProduction && !ob_get_level()) {
            ob_start('ob_gzhandler');
        }
    }
    
    /**
     * Minify CSS content
     */
    public static function minifyCSS($css) {
        // Remove comments
        $css = preg_replace('!/\*[^*]*\*+([^/][^*]*\*+)*/!', '', $css);
        
        // Remove whitespace
        $css = str_replace(["\r\n", "\r", "\n", "\t", '  ', '    ', '    '], '', $css);
        
        // Remove spaces around selectors
        $css = preg_replace('/\s*([\{\}:;,])\s*/', '$1', $css);
        
        // Remove last semicolon in block
        $css = str_replace(';}', '}', $css);
        
        return trim($css);
    }
    
    /**
     * Minify JavaScript content
     */
    public static function minifyJS($js) {
        // Remove single-line comments (but preserve URLs)
        $js = preg_replace('~//[^\n]*~', '', $js);
        
        // Remove multi-line comments
        $js = preg_replace('~/\*.*?\*/~s', '', $js);
        
        // Remove whitespace
        $js = preg_replace('/\s+/', ' ', $js);
        
        // Remove spaces around operators and punctuation
        $js = preg_replace('/\s*([\{\}\(\);,:\+\-\*\/=])\s*/', '$1', $js);
        
        return trim($js);
    }
    
    /**
     * Get minified CSS file
     * Auto-generates minified version if doesn't exist or source changed
     */
    public static function getMinifiedCSS($cssFile) {
        $sourcePath = APP_ROOT . $cssFile;
        
        if (!file_exists($sourcePath)) {
            return $cssFile; // Return original if not found
        }
        
        // In development, return original
        if (!self::$isProduction) {
            return $cssFile;
        }
        
        $sourceHash = md5_file($sourcePath);
        $fileName = basename($cssFile, '.css');
        $minifiedPath = self::$cacheDir . $fileName . '.' . $sourceHash . '.min.css';
        
        // ✅ FIXED: URL now points to /cache/assets/ instead of /cms-core/backend/cache/assets/
        $minifiedUrl = '/cache/assets/' . basename($minifiedPath);
        
        // Generate minified version if doesn't exist
        if (!file_exists($minifiedPath)) {
            $css = file_get_contents($sourcePath);
            $minified = self::minifyCSS($css);
            file_put_contents($minifiedPath, $minified);
            
            // Clean old versions
            self::cleanOldVersions($fileName, '*.min.css', $sourceHash);
        }
        
        return $minifiedUrl . '?v=' . $sourceHash;
    }
    
    /**
     * Get minified JS file
     */
    public static function getMinifiedJS($jsFile) {
        $sourcePath = APP_ROOT . $jsFile;
        
        if (!file_exists($sourcePath)) {
            return $jsFile;
        }
        
        // In development, return original
        if (!self::$isProduction) {
            return $jsFile;
        }
        
        $sourceHash = md5_file($sourcePath);
        $fileName = basename($jsFile, '.js');
        $minifiedPath = self::$cacheDir . $fileName . '.' . $sourceHash . '.min.js';
        
        // ✅ FIXED: URL now points to /cache/assets/ instead of /cms-core/backend/cache/assets/
        $minifiedUrl = '/cache/assets/' . basename($minifiedPath);
        
        if (!file_exists($minifiedPath)) {
            $js = file_get_contents($sourcePath);
            $minified = self::minifyJS($js);
            file_put_contents($minifiedPath, $minified);
            
            self::cleanOldVersions($fileName, '*.min.js', $sourceHash);
        }
        
        return $minifiedUrl . '?v=' . $sourceHash;
    }
    
    /**
     * Clean old cached versions
     */
    private static function cleanOldVersions($fileName, $pattern, $keepHash) {
        $files = glob(self::$cacheDir . $fileName . '.' . $pattern);
        foreach ($files as $file) {
            if (strpos($file, $keepHash) === false) {
                unlink($file);
            }
        }
    }
    
    /**
     * Set browser cache headers for static assets
     */
    public static function setCacheHeaders($type = 'asset', $maxAge = 31536000) {
        // Only in production
        if (!self::$isProduction) {
            return;
        }
        
        // Set cache control
        header('Cache-Control: public, max-age=' . $maxAge);
        header('Expires: ' . gmdate('D, d M Y H:i:s', time() + $maxAge) . ' GMT');
        
        // Set ETag
        if (isset($_SERVER['REQUEST_URI'])) {
            $etag = md5($_SERVER['REQUEST_URI']);
            header('ETag: "' . $etag . '"');
            
            // Check if browser has cached version
            if (isset($_SERVER['HTTP_IF_NONE_MATCH']) && $_SERVER['HTTP_IF_NONE_MATCH'] === '"' . $etag . '"') {
                header('HTTP/1.1 304 Not Modified');
                exit;
            }
        }
    }
    
    /**
     * Enable lazy loading for images
     * Call this to get the lazy loading attribute
     */
    public static function lazyLoadAttr() {
        return self::$isProduction ? 'loading="lazy"' : '';
    }
    
    /**
     * Bundle multiple CSS files into one
     */
    public static function bundleCSS(array $cssFiles) {
        if (!self::$isProduction) {
            // In development, return separate files
            $output = '';
            foreach ($cssFiles as $file) {
                $output .= '<link rel="stylesheet" href="' . $file . '">' . "\n";
            }
            return $output;
        }
        
        $bundleHash = md5(implode('', $cssFiles));
        $bundlePath = self::$cacheDir . 'bundle.' . $bundleHash . '.css';
        
        // ✅ FIXED: Bundle URL now points to /cache/assets/
        $bundleUrl = '/cache/assets/bundle.' . $bundleHash . '.css';
        
        if (!file_exists($bundlePath)) {
            $combined = '';
            foreach ($cssFiles as $file) {
                $sourcePath = APP_ROOT . $file;
                if (file_exists($sourcePath)) {
                    $css = file_get_contents($sourcePath);
                    $combined .= $css . "\n";
                }
            }
            
            $minified = self::minifyCSS($combined);
            file_put_contents($bundlePath, $minified);
        }
        
        return '<link rel="stylesheet" href="' . $bundleUrl . '?v=' . $bundleHash . '">';
    }
    
    /**
     * Bundle multiple JS files into one
     */
    public static function bundleJS(array $jsFiles) {
        if (!self::$isProduction) {
            $output = '';
            foreach ($jsFiles as $file) {
                $output .= '<script src="' . $file . '"></script>' . "\n";
            }
            return $output;
        }
        
        $bundleHash = md5(implode('', $jsFiles));
        $bundlePath = self::$cacheDir . 'bundle.' . $bundleHash . '.js';
        
        // ✅ FIXED: Bundle URL now points to /cache/assets/
        $bundleUrl = '/cache/assets/bundle.' . $bundleHash . '.js';
        
        if (!file_exists($bundlePath)) {
            $combined = '';
            foreach ($jsFiles as $file) {
                $sourcePath = APP_ROOT . $file;
                if (file_exists($sourcePath)) {
                    $js = file_get_contents($sourcePath);
                    $combined .= $js . ";\n";
                }
            }
            
            $minified = self::minifyJS($combined);
            file_put_contents($bundlePath, $minified);
        }
        
        return '<script src="' . $bundleUrl . '?v=' . $bundleHash . '"></script>';
    }
    
    /**
     * Clear all cached assets
     */
    public static function clearCache() {
        $files = glob(self::$cacheDir . '*');
        foreach ($files as $file) {
            if (is_file($file)) {
                unlink($file);
            }
        }
        
        return [
            'success' => true,
            'files_deleted' => count($files)
        ];
    }
    
    /**
     * Get cache statistics
     */
    public static function getCacheStats() {
        $files = glob(self::$cacheDir . '*');
        $totalSize = 0;
        
        foreach ($files as $file) {
            if (is_file($file)) {
                $totalSize += filesize($file);
            }
        }
        
        return [
            'files' => count($files),
            'size' => $totalSize,
            'size_formatted' => self::formatBytes($totalSize),
            'location' => self::$cacheDir
        ];
    }
    
    /**
     * Format bytes to human readable
     */
    private static function formatBytes($bytes) {
        $units = ['B', 'KB', 'MB', 'GB'];
        $i = 0;
        
        while ($bytes >= 1024 && $i < count($units) - 1) {
            $bytes /= 1024;
            $i++;
        }
        
        return round($bytes, 2) . ' ' . $units[$i];
    }
}