<?php
/**
 * cms-core/backend/performance/PersistentCache.php
 * File-based persistent cache that survives across requests
 * Much faster than database queries, no external dependencies
 */

class PersistentCache {
    
    private static $cacheDir;
    private static $enabled = true;
    private static $defaultTTL = 3600; // 1 hour
    
    /**
     * Initialize cache
     */
    public static function init() {
        // BETTER: Store cache in ROOT/cache/ instead of cms-core
        // This survives CMS updates!
        self::$cacheDir = APP_ROOT . '/cache/queries/';
        
        // Create cache directory if not exists
        if (!is_dir(self::$cacheDir)) {
            mkdir(self::$cacheDir, 0755, true);
        }
    }
    
    /**
     * Get cached value
     */
    public static function get($key) {
        if (!self::$enabled) {
            return null;
        }
        
        $filename = self::$cacheDir . md5($key) . '.cache';
        
        if (!file_exists($filename)) {
            return null;
        }
        
        $data = @file_get_contents($filename);
        if ($data === false) {
            return null;
        }
        
        $cache = @unserialize($data);
        if ($cache === false) {
            return null;
        }
        
        // Check if expired
        if (isset($cache['expires']) && $cache['expires'] < time()) {
            @unlink($filename);
            return null;
        }
        
        return $cache['value'] ?? null;
    }
    
    /**
     * Set cached value
     */
    public static function set($key, $value, $ttl = null) {
        if (!self::$enabled) {
            return false;
        }
        
        if ($ttl === null) {
            $ttl = self::$defaultTTL;
        }
        
        $filename = self::$cacheDir . md5($key) . '.cache';
        
        $cache = [
            'value' => $value,
            'expires' => time() + $ttl,
            'created' => time()
        ];
        
        $data = serialize($cache);
        return @file_put_contents($filename, $data, LOCK_EX) !== false;
    }
    
    /**
     * Delete cached value
     */
    public static function delete($key) {
        $filename = self::$cacheDir . md5($key) . '.cache';
        
        if (file_exists($filename)) {
            return @unlink($filename);
        }
        
        return true;
    }
    
    /**
     * Clear all cache
     */
    public static function clear() {
        $files = glob(self::$cacheDir . '*.cache');
        $deleted = 0;
        
        foreach ($files as $file) {
            if (is_file($file)) {
                if (@unlink($file)) {
                    $deleted++;
                }
            }
        }
        
        return $deleted;
    }
    
    /**
     * Clean expired cache entries
     */
    public static function cleanExpired() {
        $files = glob(self::$cacheDir . '*.cache');
        $deleted = 0;
        
        foreach ($files as $file) {
            if (!is_file($file)) continue;
            
            $data = @file_get_contents($file);
            if ($data === false) continue;
            
            $cache = @unserialize($data);
            if ($cache === false) continue;
            
            // Delete if expired
            if (isset($cache['expires']) && $cache['expires'] < time()) {
                if (@unlink($file)) {
                    $deleted++;
                }
            }
        }
        
        return $deleted;
    }
    
    /**
     * Get cache statistics
     */
    public static function getStats() {
        $files = glob(self::$cacheDir . '*.cache');
        $totalSize = 0;
        $expired = 0;
        
        foreach ($files as $file) {
            if (is_file($file)) {
                $totalSize += filesize($file);
                
                $data = @file_get_contents($file);
                if ($data !== false) {
                    $cache = @unserialize($data);
                    if ($cache !== false && isset($cache['expires']) && $cache['expires'] < time()) {
                        $expired++;
                    }
                }
            }
        }
        
        return [
            'enabled' => self::$enabled,
            'files' => count($files),
            'size' => $totalSize,
            'size_formatted' => self::formatBytes($totalSize),
            'expired' => $expired,
            'location' => self::$cacheDir
        ];
    }
    
    /**
     * Format bytes to human readable
     */
    private static function formatBytes($bytes) {
        $units = ['B', 'KB', 'MB', 'GB'];
        $i = 0;
        
        while ($bytes >= 1024 && $i < count($units) - 1) {
            $bytes /= 1024;
            $i++;
        }
        
        return round($bytes, 2) . ' ' . $units[$i];
    }
    
    /**
     * Enable/disable cache
     */
    public static function setEnabled($enabled) {
        self::$enabled = $enabled;
    }
    
    /**
     * Remember: Execute a callback and cache the result
     * Usage: PersistentCache::remember('key', 3600, function() { return expensive_operation(); });
     */
    public static function remember($key, $ttl, $callback) {
        $value = self::get($key);
        
        if ($value !== null) {
            return $value;
        }
        
        $value = $callback();
        self::set($key, $value, $ttl);
        
        return $value;
    }
}