<?php
/**
 * cms-core/backend/core/QueryCache.php
 * Simple in-memory query cache to reduce database load
 * Caches results for the duration of the request
 */

class QueryCache {
    
    private static $cache = [];
    private static $enabled = true;
    private static $hits = 0;
    private static $misses = 0;
    
    /**
     * Get cached query result
     */
    public static function get($key) {
        if (!self::$enabled) {
            return null;
        }
        
        if (isset(self::$cache[$key])) {
            self::$hits++;
            return self::$cache[$key];
        }
        
        self::$misses++;
        return null;
    }
    
    /**
     * Store query result in cache
     */
    public static function set($key, $value, $ttl = 300) {
        if (!self::$enabled) {
            return;
        }
        
        self::$cache[$key] = $value;
    }
    
    /**
     * Clear specific cache entry
     */
    public static function delete($key) {
        unset(self::$cache[$key]);
    }
    
    /**
     * Clear all cache
     */
    public static function clear() {
        self::$cache = [];
        self::$hits = 0;
        self::$misses = 0;
    }
    
    /**
     * Get cache statistics
     */
    public static function getStats() {
        return [
            'enabled' => self::$enabled,
            'items' => count(self::$cache),
            'hits' => self::$hits,
            'misses' => self::$misses,
            'hit_rate' => self::$hits + self::$misses > 0 
                ? round((self::$hits / (self::$hits + self::$misses)) * 100, 2) . '%'
                : '0%'
        ];
    }
    
    /**
     * Enable/disable cache
     */
    public static function setEnabled($enabled) {
        self::$enabled = $enabled;
    }
    
    /**
     * Generate cache key from SQL and params
     */
    public static function generateKey($sql, $params = []) {
        return md5($sql . serialize($params));
    }
}