<?php
/**
 * cms-core/backend/performance/image_optimizer.php
 * Image Optimization & WebP Conversion
 */

class ImageOptimizer {
    
    /**
     * Process uploaded image: resize + convert to WebP
     */
    public static function processUpload($imagePath) {
        // Check if optimization is enabled
        if (!get_setting('auto_convert_webp', true)) {
            return ['optimized' => false];
        }
        
        // Check if file exists
        if (!file_exists($imagePath)) {
            return ['error' => 'File not found'];
        }
        
        // Get settings
        $maxWidth = (int)get_setting('max_image_width', 2000);
        $quality = (int)get_setting('webp_quality', 85);
        
        $result = [
            'optimized' => true,
            'original_size' => filesize($imagePath)
        ];
        
        // Get image info
        $imageInfo = getimagesize($imagePath);
        if (!$imageInfo) {
            return ['error' => 'Invalid image'];
        }
        
        list($width, $height, $type) = $imageInfo;
        
        // Load image based on type
        $image = null;
        switch ($type) {
            case IMAGETYPE_JPEG:
                $image = @imagecreatefromjpeg($imagePath);
                break;
            case IMAGETYPE_PNG:
                $image = @imagecreatefrompng($imagePath);
                break;
            case IMAGETYPE_GIF:
                $image = @imagecreatefromgif($imagePath);
                break;
            default:
                return ['error' => 'Unsupported image type'];
        }
        
        if (!$image) {
            return ['error' => 'Failed to load image'];
        }
        
        // Resize if needed
        if ($width > $maxWidth) {
            $newHeight = (int)($height * ($maxWidth / $width));
            $resized = imagecreatetruecolor($maxWidth, $newHeight);
            
            // Preserve transparency for PNG/GIF
            if ($type === IMAGETYPE_PNG || $type === IMAGETYPE_GIF) {
                imagealphablending($resized, false);
                imagesavealpha($resized, true);
                $transparent = imagecolorallocatealpha($resized, 255, 255, 255, 127);
                imagefilledrectangle($resized, 0, 0, $maxWidth, $newHeight, $transparent);
            }
            
            imagecopyresampled($resized, $image, 0, 0, 0, 0, $maxWidth, $newHeight, $width, $height);
            imagedestroy($image);
            $image = $resized;
            
            $result['resized'] = true;
            $result['new_dimensions'] = "{$maxWidth}x{$newHeight}";
        }
        
        // Save optimized original
        switch ($type) {
            case IMAGETYPE_JPEG:
                imagejpeg($image, $imagePath, $quality);
                break;
            case IMAGETYPE_PNG:
                // PNG compression level (0-9)
                $pngQuality = (int)(9 - ($quality / 100 * 9));
                imagepng($image, $imagePath, $pngQuality);
                break;
            case IMAGETYPE_GIF:
                imagegif($image, $imagePath);
                break;
        }
        
        $result['optimized_size'] = filesize($imagePath);
        
        // Create WebP version
        $webpPath = preg_replace('/\.(jpg|jpeg|png|gif)$/i', '.webp', $imagePath);
        
        if (function_exists('imagewebp')) {
            if (imagewebp($image, $webpPath, $quality)) {
                $result['webp'] = basename($webpPath);
                $result['webp_size'] = filesize($webpPath);
                $result['webp_savings'] = (int)((1 - ($result['webp_size'] / $result['original_size'])) * 100);
            }
        } else {
            $result['webp_error'] = 'WebP not supported by PHP installation';
        }
        
        imagedestroy($image);
        
        return $result;
    }
    
    /**
     * Batch convert existing images to WebP
     */
    public static function batchConvert($directory, $limit = 50) {
        $converted = 0;
        $errors = [];
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($directory)
        );
        
        foreach ($iterator as $file) {
            if ($converted >= $limit) break;
            
            if ($file->isFile() && preg_match('/\.(jpg|jpeg|png|gif)$/i', $file->getFilename())) {
                $webpPath = preg_replace('/\.(jpg|jpeg|png|gif)$/i', '.webp', $file->getPathname());
                
                // Skip if WebP already exists
                if (file_exists($webpPath)) {
                    continue;
                }
                
                $result = self::processUpload($file->getPathname());
                
                if (isset($result['webp'])) {
                    $converted++;
                } elseif (isset($result['error'])) {
                    $errors[] = $file->getFilename() . ': ' . $result['error'];
                }
            }
        }
        
        return [
            'converted' => $converted,
            'errors' => $errors
        ];
    }
}