<?php
/** cms-core/backend/security/SecurityMonitor.php */

/**
 * Security Monitor
 * 
 * Überwacht und loggt verdächtige Aktivitäten
 * Features:
 * - Brute-Force Detection
 * - Rate Limiting
 * - SQL Injection Detection
 * - XSS Attempt Detection
 * - Path Traversal Detection
 * - Suspicious Request Pattern Detection
 */

class SecurityMonitor {
    
    private static $db = null;
    private static $logFile = null;
    
    /**
     * Initialisiert Security Monitor
     */
    public static function init() {
        self::$logFile = APP_ROOT . '/logs/security.log';
        
        // Erstelle logs Verzeichnis falls nicht vorhanden
        $logDir = dirname(self::$logFile);
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
        
        // Prüfe ob Tabelle existiert
        self::ensureTable();
    }
    
    /**
     * Erstellt Security-Log Tabelle falls nicht vorhanden
     */
    private static function ensureTable() {
        try {
            $db = Database::getInstance()->getConnection();
            
            $sql = "CREATE TABLE IF NOT EXISTS security_logs (
                id INT AUTO_INCREMENT PRIMARY KEY,
                event_type VARCHAR(50) NOT NULL,
                severity ENUM('low', 'medium', 'high', 'critical') DEFAULT 'medium',
                ip_address VARCHAR(45) NOT NULL,
                user_agent TEXT,
                request_uri TEXT,
                user_id INT NULL,
                details TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                INDEX idx_event_type (event_type),
                INDEX idx_ip (ip_address),
                INDEX idx_created (created_at)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
            
            $db->exec($sql);
            
        } catch (Exception $e) {
            // Wenn DB nicht verfügbar, nur File-Logging
            error_log('SecurityMonitor: Could not create table - ' . $e->getMessage());
        }
    }
    
    /**
     * Loggt Security-Event
     */
    public static function log($eventType, $severity = 'medium', $details = []) {
        $ip = self::getClientIP();
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
        $requestUri = $_SERVER['REQUEST_URI'] ?? 'unknown';
        $userId = $_SESSION['user_id'] ?? null;
        
        $logEntry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'event_type' => $eventType,
            'severity' => $severity,
            'ip' => $ip,
            'user_agent' => $userAgent,
            'request_uri' => $requestUri,
            'user_id' => $userId,
            'details' => $details
        ];
        
        // Log to file
        self::logToFile($logEntry);
        
        // Log to database
        self::logToDatabase($logEntry);
        
        // Bei kritischen Events: Email-Benachrichtigung
        if ($severity === 'critical') {
            self::sendAlert($logEntry);
        }
    }
    
    /**
     * Loggt zu File
     */
    private static function logToFile($entry) {
        $line = sprintf(
            "[%s] %s (%s) - IP: %s - %s\n",
            $entry['timestamp'],
            strtoupper($entry['event_type']),
            strtoupper($entry['severity']),
            $entry['ip'],
            json_encode($entry['details'])
        );
        
        file_put_contents(self::$logFile, $line, FILE_APPEND | LOCK_EX);
    }
    
    /**
     * Loggt zu Database
     */
    private static function logToDatabase($entry) {
        try {
            $db = Database::getInstance()->getConnection();
            
            $stmt = $db->prepare("
                INSERT INTO security_logs 
                (event_type, severity, ip_address, user_agent, request_uri, user_id, details)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $entry['event_type'],
                $entry['severity'],
                $entry['ip'],
                $entry['user_agent'],
                $entry['request_uri'],
                $entry['user_id'],
                json_encode($entry['details'])
            ]);
            
        } catch (Exception $e) {
            // Stiller Fehler - wir wollen normale Operations nicht stören
            error_log('SecurityMonitor DB Error: ' . $e->getMessage());
        }
    }
    
    /**
     * Prüft auf Brute-Force Login-Versuche
     */
    public static function checkBruteForce($username) {
        $ip = self::getClientIP();
        
        try {
            $db = Database::getInstance()->getConnection();
            
            // Zähle fehlgeschlagene Login-Versuche in letzten 15 Minuten
            $stmt = $db->prepare("
                SELECT COUNT(*) as attempts
                FROM security_logs
                WHERE event_type = 'login_failed'
                AND ip_address = ?
                AND created_at > DATE_SUB(NOW(), INTERVAL 15 MINUTE)
            ");
            
            $stmt->execute([$ip]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            $attempts = $result['attempts'] ?? 0;
            
            // Max. 5 Versuche in 15 Minuten
            if ($attempts >= 5) {
                self::log('brute_force_detected', 'high', [
                    'username' => $username,
                    'attempts' => $attempts
                ]);
                return true; // Blockieren
            }
            
            return false;
            
        } catch (Exception $e) {
            // Im Fehlerfall: Sicher sein und blockieren
            return true;
        }
    }
    
    /**
     * Rate Limiting Check
     */
    public static function checkRateLimit($action = 'general', $maxRequests = 60, $timeWindow = 60) {
        $ip = self::getClientIP();
        $cacheKey = "ratelimit_{$action}_{$ip}";
        
        // Verwende APCu wenn verfügbar, sonst Session
        if (function_exists('apcu_fetch')) {
            $requests = apcu_fetch($cacheKey, $success);
            if (!$success) {
                $requests = 1;
                apcu_store($cacheKey, $requests, $timeWindow);
                return false;
            }
            
            if ($requests >= $maxRequests) {
                self::log('rate_limit_exceeded', 'medium', [
                    'action' => $action,
                    'requests' => $requests,
                    'limit' => $maxRequests
                ]);
                return true; // Blockieren
            }
            
            apcu_inc($cacheKey);
            return false;
            
        } else {
            // Fallback auf Session-basiertes Rate Limiting
            $sessionKey = '_rate_limit_' . $action;
            
            if (!isset($_SESSION[$sessionKey])) {
                $_SESSION[$sessionKey] = [
                    'count' => 1,
                    'start' => time()
                ];
                return false;
            }
            
            $data = $_SESSION[$sessionKey];
            
            // Reset wenn Zeitfenster abgelaufen
            if (time() - $data['start'] > $timeWindow) {
                $_SESSION[$sessionKey] = [
                    'count' => 1,
                    'start' => time()
                ];
                return false;
            }
            
            // Prüfe Limit
            if ($data['count'] >= $maxRequests) {
                self::log('rate_limit_exceeded', 'medium', [
                    'action' => $action,
                    'requests' => $data['count']
                ]);
                return true;
            }
            
            $_SESSION[$sessionKey]['count']++;
            return false;
        }
    }
    
    /**
     * Prüft auf SQL Injection Versuche
     */
    public static function detectSQLInjection($input) {
        $patterns = [
            '/(\bUNION\b.*\bSELECT\b)/i',
            '/(\bSELECT\b.*\bFROM\b)/i',
            '/(\bINSERT\b.*\bINTO\b)/i',
            '/(\bDELETE\b.*\bFROM\b)/i',
            '/(\bDROP\b.*\bTABLE\b)/i',
            '/(\bUPDATE\b.*\bSET\b)/i',
            '/(\'|\").*(\bOR\b|\bAND\b).*(\=)/i',
            '/(\-\-|#|\/\*|\*\/)/i'
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $input)) {
                self::log('sql_injection_attempt', 'critical', [
                    'input' => substr($input, 0, 200),
                    'pattern' => $pattern
                ]);
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Prüft auf XSS Versuche
     */
    public static function detectXSS($input) {
        $patterns = [
            '/<script\b[^>]*>(.*?)<\/script>/is',
            '/javascript:/i',
            '/on\w+\s*=\s*["\'].*["\']/i',
            '/<iframe/i',
            '/<embed/i',
            '/<object/i'
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $input)) {
                self::log('xss_attempt', 'high', [
                    'input' => substr($input, 0, 200),
                    'pattern' => $pattern
                ]);
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Prüft auf Path Traversal Versuche
     */
    public static function detectPathTraversal($path) {
        $patterns = [
            '/\.\.\//',
            '/\.\.\\\\/',
            '/%2e%2e%2f/i',
            '/%2e%2e\//i',
            '/\.\.\%2f/i'
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $path)) {
                self::log('path_traversal_attempt', 'high', [
                    'path' => $path,
                    'pattern' => $pattern
                ]);
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Überwacht Request
     */
    public static function monitorRequest() {
        // Prüfe Query String
        if (!empty($_SERVER['QUERY_STRING'])) {
            if (self::detectSQLInjection($_SERVER['QUERY_STRING'])) {
                self::blockRequest('SQL Injection attempt in query string');
            }
            if (self::detectXSS($_SERVER['QUERY_STRING'])) {
                self::blockRequest('XSS attempt in query string');
            }
        }
        
        // Prüfe POST Daten
        foreach ($_POST as $key => $value) {
            if (is_string($value)) {
                if (self::detectSQLInjection($value)) {
                    self::blockRequest('SQL Injection attempt in POST data');
                }
            }
        }
        
        // Prüfe Request URI
        if (self::detectPathTraversal($_SERVER['REQUEST_URI'])) {
            self::blockRequest('Path traversal attempt');
        }
        
        // Rate Limiting für API-Calls
        if (strpos($_SERVER['REQUEST_URI'], '/api/') !== false) {
            if (self::checkRateLimit('api', 100, 60)) {
                self::blockRequest('API rate limit exceeded');
            }
        }
    }
    
    /**
     * Blockiert Request
     */
    private static function blockRequest($reason) {
        self::log('request_blocked', 'critical', ['reason' => $reason]);
        
        http_response_code(403);
        die('Access Denied');
    }
    
    /**
     * Holt Client IP
     */
    private static function getClientIP() {
        $headers = [
            'HTTP_CF_CONNECTING_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_REAL_IP',
            'REMOTE_ADDR'
        ];
        
        foreach ($headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = $_SERVER[$header];
                if (strpos($ip, ',') !== false) {
                    $ips = explode(',', $ip);
                    $ip = trim($ips[0]);
                }
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return 'unknown';
    }
    
    /**
     * Sendet Alert bei kritischen Events
     */
    private static function sendAlert($entry) {
        // TODO: Email-Benachrichtigung implementieren
        // Für jetzt: Nur Error Log
        error_log(sprintf(
            "SECURITY ALERT [%s]: %s from IP %s",
            $entry['severity'],
            $entry['event_type'],
            $entry['ip']
        ));
    }
    
    /**
     * Gibt Security-Statistiken zurück
     */
    public static function getStats($days = 7) {
        try {
            $db = Database::getInstance()->getConnection();
            
            $stmt = $db->prepare("
                SELECT 
                    event_type,
                    severity,
                    COUNT(*) as count
                FROM security_logs
                WHERE created_at > DATE_SUB(NOW(), INTERVAL ? DAY)
                GROUP BY event_type, severity
                ORDER BY count DESC
            ");
            
            $stmt->execute([$days]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
            
        } catch (Exception $e) {
            return [];
        }
    }
    
    /**
     * Gibt letzte Security-Events zurück
     */
    public static function getRecentEvents($limit = 50) {
        try {
            $db = Database::getInstance()->getConnection();
            
            $stmt = $db->prepare("
                SELECT *
                FROM security_logs
                ORDER BY created_at DESC
                LIMIT ?
            ");
            
            $stmt->execute([$limit]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
            
        } catch (Exception $e) {
            return [];
        }
    }
}

// Auto-Initialize
SecurityMonitor::init();