<?php
/** cms-core/backend/security/csp.php - FIXED */

/**
 * Content Security Policy
 * Verhindert XSS und andere Code-Injection-Angriffe
 */

class CSP {
    
    /**
     * Setzt CSP-Header
     */
    public static function setHeaders() {
        // CRITICAL: Skip headers if in test mode or headers already sent
        if (defined('SKIP_CSP_HEADERS') || headers_sent()) {
            return;
        }
        
        // WICHTIG: Prüfe ob Vorschau-Modus aktiv ist
        $isPreview = isset($_COOKIE['color_variant_preview']) && 
                     isset($_SESSION['user_role']) && 
                     $_SESSION['user_role'] === 'admin';
        
        // Frame-Ancestors: Erlaube iFrames im Preview-Modus
        $frameAncestors = $isPreview ? "'self'" : "'none'";
        
        $directives = [
            'default-src' => "'self'",
            'script-src' => "'self' 'unsafe-inline' https://www.googletagmanager.com https://www.google-analytics.com https://connect.facebook.net https://analytics.tiktok.com https://cdn.quilljs.com",
            'style-src' => "'self' 'unsafe-inline' https://cdn.quilljs.com",
            'img-src' => "'self' data: https: http:",
            'font-src' => "'self' data:",
            'connect-src' => "'self' https://www.google-analytics.com https://analytics.tiktok.com",
            'frame-ancestors' => $frameAncestors,
            'base-uri' => "'self'"
        ];
        
        // Baue CSP-String
        $cspParts = [];
        foreach ($directives as $directive => $value) {
            $cspParts[] = "$directive $value";
        }
        
        $csp = implode('; ', $cspParts);
        
        // Setze Header
        header("Content-Security-Policy: $csp");
    }
    
    /**
     * Generiert Nonce für inline Scripts
     */
    public static function generateNonce() {
        if (!isset($_SESSION['csp_nonce'])) {
            $_SESSION['csp_nonce'] = base64_encode(random_bytes(16));
        }
        return $_SESSION['csp_nonce'];
    }
    
    /**
     * Gibt Nonce-Attribut für Script-Tags zurück
     */
    public static function nonceAttr() {
        return 'nonce="' . self::generateNonce() . '"';
    }
}

/**
 * Zusätzliche Security Header
 */
function set_security_headers() {
    // CRITICAL: Skip headers if in test mode or headers already sent
    if (defined('SKIP_CSP_HEADERS') || headers_sent()) {
        return;
    }
    
    // WICHTIG: Prüfe Vorschau-Modus
    $isPreview = isset($_COOKIE['color_variant_preview']) && 
                 isset($_SESSION['user_role']) && 
                 $_SESSION['user_role'] === 'admin';
    
    // X-Frame-Options: SAMEORIGIN im Preview, sonst DENY
    if ($isPreview) {
        header('X-Frame-Options: SAMEORIGIN');
    } else {
        header('X-Frame-Options: DENY');
    }
    
    // Aktiviere XSS-Filter im Browser
    header('X-XSS-Protection: 1; mode=block');
    
    // Verhindere MIME-Type-Sniffing
    header('X-Content-Type-Options: nosniff');
    
    // Referrer Policy
    header('Referrer-Policy: strict-origin-when-cross-origin');
    
    // Permissions Policy
    header('Permissions-Policy: geolocation=(), microphone=(), camera=()');
    
    // HTTPS erzwingen (nur wenn SSL aktiviert)
    if (getenv('FORCE_HTTPS') === 'true' || getenv('APP_ENV') === 'production') {
        // Redirect zu HTTPS wenn nicht bereits HTTPS
        if (empty($_SERVER['HTTPS']) || $_SERVER['HTTPS'] === 'off') {
            $redirect = 'https://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
            header('Location: ' . $redirect, true, 301);
            exit;
        }
        
        // HSTS Header setzen
        header('Strict-Transport-Security: max-age=31536000; includeSubDomains; preload');
    }
}