<?php
/** cms-core/backend/security/csrf.php - ENHANCED */

/**
 * CSRF Protection - OWASP Compliant
 * Cross-Site Request Forgery Protection with Token Expiry
 */

class CSRF {
    
    // Token expires after 1 hour
    const TOKEN_EXPIRY = 3600;
    
    /**
     * Generiert CSRF-Token und speichert in Session
     */
    public static function generateToken() {
        if (empty($_SESSION[CSRF_TOKEN_NAME]) || self::isExpired()) {
            $_SESSION[CSRF_TOKEN_NAME] = bin2hex(random_bytes(32));
            $_SESSION[CSRF_TOKEN_NAME . '_time'] = time();
        }
        return $_SESSION[CSRF_TOKEN_NAME];
    }
    
    /**
     * Gibt HTML-Input-Feld mit CSRF-Token zurück
     */
    public static function field() {
        $token = self::generateToken();
        return '<input type="hidden" name="' . CSRF_TOKEN_NAME . '" value="' . htmlspecialchars($token) . '">';
    }
    
    /**
     * Validiert CSRF-Token
     */
    public static function validate($token = null) {
        // Hole Token aus POST oder Parameter
        if ($token === null) {
            $token = $_POST[CSRF_TOKEN_NAME] ?? $_GET[CSRF_TOKEN_NAME] ?? '';
        }
        
        // Prüfe ob Session-Token existiert
        if (empty($_SESSION[CSRF_TOKEN_NAME])) {
            return false;
        }
        
        // Prüfe ob Token abgelaufen ist
        if (self::isExpired()) {
            return false;
        }
        
        // Vergleiche Tokens (timing-attack safe)
        return hash_equals($_SESSION[CSRF_TOKEN_NAME], $token);
    }
    
    /**
     * Prüft ob Token abgelaufen ist
     */
    public static function isExpired() {
        if (!isset($_SESSION[CSRF_TOKEN_NAME . '_time'])) {
            return true;
        }
        
        $tokenAge = time() - $_SESSION[CSRF_TOKEN_NAME . '_time'];
        return $tokenAge > self::TOKEN_EXPIRY;
    }
    
    /**
     * Erzwingt CSRF-Validierung (sterben bei Fehler)
     */
    public static function require() {
        if (!self::validate()) {
            http_response_code(403);
            die('CSRF-Token ungültig oder abgelaufen. Bitte versuchen Sie es erneut.');
        }
    }
    
    /**
     * Regeneriert Token (nach Login empfohlen)
     */
    public static function regenerate() {
        unset($_SESSION[CSRF_TOKEN_NAME]);
        unset($_SESSION[CSRF_TOKEN_NAME . '_time']);
        return self::generateToken();
    }
    
    /**
     * Get token age in seconds
     */
    public static function getTokenAge() {
        if (!isset($_SESSION[CSRF_TOKEN_NAME . '_time'])) {
            return null;
        }
        return time() - $_SESSION[CSRF_TOKEN_NAME . '_time'];
    }
}