<?php
/** cms-core/backend/security/rate_limit.php */

/**
 * Rate Limiting
 * Verhindert Brute-Force-Angriffe
 */

class RateLimit {
    
    /**
     * Prüft Login-Versuche
     */
    public static function checkLoginAttempts($username = null) {
        $ip = self::getClientIP();
        $timeWindow = time() - LOGIN_LOCKOUT_TIME;
        
        // Lösche alte Versuche
        db()->execute(
            "DELETE FROM login_attempts WHERE attempt_time < FROM_UNIXTIME(?)",
            [$timeWindow]
        );
        
        // Zähle aktuelle Versuche
        $sql = "SELECT COUNT(*) as count FROM login_attempts WHERE ip_address = ? AND attempt_time > FROM_UNIXTIME(?)";
        $params = [$ip, $timeWindow];
        
        if ($username) {
            $sql .= " AND username = ?";
            $params[] = $username;
        }
        
        $result = db()->fetchOne($sql, $params);
        $attempts = $result['count'] ?? 0;
        
        return [
            'allowed' => $attempts < MAX_LOGIN_ATTEMPTS,
            'attempts' => $attempts,
            'remaining' => max(0, MAX_LOGIN_ATTEMPTS - $attempts)
        ];
    }
    
    /**
     * Registriert Login-Versuch
     */
    public static function recordLoginAttempt($username = null) {
        $ip = self::getClientIP();
        
        db()->insert(
            "INSERT INTO login_attempts (ip_address, username, attempt_time) VALUES (?, ?, NOW())",
            [$ip, $username]
        );
    }
    
    /**
     * Löscht Login-Versuche nach erfolgreichem Login
     */
    public static function clearLoginAttempts($username = null) {
        $ip = self::getClientIP();
        
        if ($username) {
            db()->execute(
                "DELETE FROM login_attempts WHERE ip_address = ? AND username = ?",
                [$ip, $username]
            );
        } else {
            db()->execute(
                "DELETE FROM login_attempts WHERE ip_address = ?",
                [$ip]
            );
        }
    }
    
    /**
     * Holt Client-IP (berücksichtigt Proxy)
     */
    private static function getClientIP() {
        $ipKeys = [
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_X_CLUSTER_CLIENT_IP',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR'
        ];
        
        foreach ($ipKeys as $key) {
            if (!empty($_SERVER[$key])) {
                $ips = explode(',', $_SERVER[$key]);
                $ip = trim($ips[0]);
                
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
    
    /**
     * Generisches Rate Limiting (z.B. für API-Calls)
     */
    public static function checkLimit($key, $maxRequests, $timeWindow) {
        $cacheKey = 'rate_limit_' . md5($key);
        
        if (!isset($_SESSION[$cacheKey])) {
            $_SESSION[$cacheKey] = [
                'count' => 0,
                'reset_time' => time() + $timeWindow
            ];
        }
        
        $data = $_SESSION[$cacheKey];
        
        // Reset wenn Zeitfenster abgelaufen
        if (time() > $data['reset_time']) {
            $_SESSION[$cacheKey] = [
                'count' => 1,
                'reset_time' => time() + $timeWindow
            ];
            return true;
        }
        
        // Prüfe Limit
        if ($data['count'] >= $maxRequests) {
            return false;
        }
        
        // Inkrementiere Counter
        $_SESSION[$cacheKey]['count']++;
        return true;
    }
}