<?php
/** cms-core/backend/security/session_hijack.php */

/**
 * SessionSecurity - Full Featured Session Hijacking Prevention
 * 
 * Features:
 * - Session Fingerprinting (Browser + Language + Encoding)
 * - Automatic Session Regeneration (every 5 minutes)
 * - Session Lifetime Management
 * - Activity Timeout (2 hours default)
 * - Optional IP Validation
 * - User Agent Validation
 * - Secure Session Destruction
 */

class SessionSecurity {
    
    private static $regenerateInterval = 300; // 5 Minuten
    private static $activityTimeout = 7200; // 2 Stunden
    
    /**
     * Validiert Session-Integrität
     */
    public static function validate() {
        // Neue Session? Initialisieren
        if (!isset($_SESSION['_fingerprint'])) {
            self::initialize();
            return true;
        }
        
        // 1. Fingerprint Check
        if (!self::validateFingerprint()) {
            self::logSecurityEvent('fingerprint_mismatch');
            return false;
        }
        
        // 2. Session Lifetime Check
        if (!self::validateLifetime()) {
            self::logSecurityEvent('session_expired');
            return false;
        }
        
        // 3. Activity Timeout Check
        if (!self::validateActivity()) {
            self::logSecurityEvent('session_timeout');
            return false;
        }
        
        // 4. Optional: IP Validation
        if (defined('SESSION_CHECK_IP') && SESSION_CHECK_IP) {
            if (!self::validateIP()) {
                self::logSecurityEvent('ip_changed');
                return false;
            }
        }
        
        // 5. User Agent Check
        if (!self::validateUserAgent()) {
            self::logSecurityEvent('user_agent_changed');
            return false;
        }
        
        // 6. Auto-Regeneration Check
        self::checkRegeneration();
        
        // 7. Update Activity
        self::updateActivity();
        
        return true;
    }
    
    /**
     * Initialisiert neue Session
     */
    private static function initialize() {
        $_SESSION['_fingerprint'] = self::generateFingerprint();
        $_SESSION['_created'] = time();
        $_SESSION['_last_regenerate'] = time();
        $_SESSION['_last_activity'] = time();
        $_SESSION['_ip'] = self::getClientIP();
        $_SESSION['_user_agent'] = $_SERVER['HTTP_USER_AGENT'] ?? '';
    }
    
    /**
     * Validiert Fingerprint
     */
    private static function validateFingerprint() {
        $current = self::generateFingerprint();
        $stored = $_SESSION['_fingerprint'] ?? '';
        return hash_equals($stored, $current);
    }
    
    /**
     * Validiert Session Lifetime
     */
    private static function validateLifetime() {
        if (!isset($_SESSION['_created'])) {
            return false;
        }
        
        $age = time() - $_SESSION['_created'];
        $maxLifetime = defined('SESSION_LIFETIME') ? SESSION_LIFETIME : 86400;
        
        return $age <= $maxLifetime;
    }
    
    /**
     * Validiert Activity Timeout
     */
    private static function validateActivity() {
        if (!isset($_SESSION['_last_activity'])) {
            return true;
        }
        
        $idle = time() - $_SESSION['_last_activity'];
        return $idle <= self::$activityTimeout;
    }
    
    /**
     * Validiert IP-Adresse
     */
    private static function validateIP() {
        $current = self::getClientIP();
        $stored = $_SESSION['_ip'] ?? '';
        return $current === $stored;
    }
    
    /**
     * Validiert User Agent
     */
    private static function validateUserAgent() {
        $current = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $stored = $_SESSION['_user_agent'] ?? '';
        return $current === $stored;
    }
    
    /**
     * Generiert Browser-Fingerprint
     */
    private static function generateFingerprint() {
        $data = [
            $_SERVER['HTTP_USER_AGENT'] ?? '',
            $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '',
            $_SERVER['HTTP_ACCEPT_ENCODING'] ?? ''
        ];
        return hash('sha256', implode('|', $data));
    }
    
    /**
     * Holt sichere Client-IP (Proxy-kompatibel)
     */
    private static function getClientIP() {
        $headers = [
            'HTTP_CF_CONNECTING_IP', // Cloudflare
            'HTTP_X_FORWARDED_FOR',  // Standard Proxy
            'HTTP_X_REAL_IP',        // Nginx
            'REMOTE_ADDR'            // Fallback
        ];
        
        foreach ($headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = $_SERVER[$header];
                
                // X-Forwarded-For kann mehrere IPs enthalten
                if (strpos($ip, ',') !== false) {
                    $ips = explode(',', $ip);
                    $ip = trim($ips[0]);
                }
                
                // Validiere IP-Format
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return 'unknown';
    }
    
    /**
     * Prüft ob Session regeneriert werden muss
     */
    private static function checkRegeneration() {
        $lastRegenerate = $_SESSION['_last_regenerate'] ?? 0;
        
        if (time() - $lastRegenerate > self::$regenerateInterval) {
            self::regenerate();
        }
    }
    
    /**
     * Update Activity Timestamp
     */
    private static function updateActivity() {
        $_SESSION['_last_activity'] = time();
    }
    
    /**
     * Regeneriert Session-ID sicher
     */
    public static function regenerate() {
        // Backup kritischer Daten
        $backup = [
            '_fingerprint' => $_SESSION['_fingerprint'] ?? null,
            '_created' => $_SESSION['_created'] ?? time(),
            '_ip' => $_SESSION['_ip'] ?? null,
            '_user_agent' => $_SESSION['_user_agent'] ?? null,
            'user_id' => $_SESSION['user_id'] ?? null,
            'user_role' => $_SESSION['user_role'] ?? null,
            'user_email' => $_SESSION['user_email'] ?? null
        ];
        
        // Regeneriere Session-ID
        session_regenerate_id(true);
        
        // Restore Daten
        foreach ($backup as $key => $value) {
            if ($value !== null) {
                $_SESSION[$key] = $value;
            }
        }
        
        $_SESSION['_last_regenerate'] = time();
        $_SESSION['_last_activity'] = time();
    }
    
    /**
     * Zerstört Session komplett
     */
    public static function destroy() {
        $_SESSION = [];
        
        // Lösche Session-Cookie
        if (isset($_COOKIE[session_name()])) {
            $params = session_get_cookie_params();
            setcookie(
                session_name(),
                '',
                time() - 3600,
                $params['path'],
                $params['domain'],
                $params['secure'],
                $params['httponly']
            );
        }
        
        if (session_status() === PHP_SESSION_ACTIVE) {
            session_destroy();
        }
    }
    
    /**
     * Loggt Security-Event
     */
    private static function logSecurityEvent($event) {
        // Nutze SecurityMonitor wenn verfügbar
        if (class_exists('SecurityMonitor')) {
            SecurityMonitor::log('session_' . $event, 'medium', [
                'ip' => self::getClientIP(),
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
                'session_id' => session_id()
            ]);
        }
        
        // Fallback: Error Log
        error_log(sprintf(
            'Session Security [%s]: %s from IP %s',
            $event,
            $_SERVER['REQUEST_URI'] ?? 'unknown',
            self::getClientIP()
        ));
    }
    
    /**
     * Session-Info für Debugging
     */
    public static function getInfo() {
        if (!isset($_SESSION['_created'])) {
            return ['status' => 'not_initialized'];
        }
        
        return [
            'status' => 'active',
            'session_id' => session_id(),
            'created' => date('Y-m-d H:i:s', $_SESSION['_created']),
            'last_activity' => date('Y-m-d H:i:s', $_SESSION['_last_activity']),
            'last_regenerate' => date('Y-m-d H:i:s', $_SESSION['_last_regenerate']),
            'ip' => $_SESSION['_ip'] ?? 'unknown',
            'fingerprint' => substr($_SESSION['_fingerprint'] ?? '', 0, 16) . '...',
            'time_remaining' => self::$activityTimeout - (time() - $_SESSION['_last_activity']),
            'regenerate_in' => self::$regenerateInterval - (time() - $_SESSION['_last_regenerate']),
            'user_id' => $_SESSION['user_id'] ?? null,
            'user_role' => $_SESSION['user_role'] ?? null
        ];
    }
}