<?php
/** cms-core/backend/security/validation.php - COMPREHENSIVE VERSION */

/**
 * Input Validation - Enterprise Grade
 * Validates ALL user inputs with strict rules
 */

class Validator {
    
    private $errors = [];
    private $data = [];
    
    public function __construct($data = []) {
        $this->data = $data;
    }
    
    /**
     * Required field
     */
    public function required($field, $message = null) {
        if (!isset($this->data[$field]) || trim($this->data[$field]) === '') {
            $this->errors[$field] = $message ?: "Das Feld '$field' ist erforderlich.";
        }
        return $this;
    }
    
    /**
     * Email validation with DNS check
     */
    public function email($field, $checkDNS = false, $message = null) {
        if (!empty($this->data[$field])) {
            $email = filter_var($this->data[$field], FILTER_VALIDATE_EMAIL);
            
            if (!$email) {
                $this->errors[$field] = $message ?: "Ungültige E-Mail-Adresse.";
                return $this;
            }
            
            // Optional DNS check
            if ($checkDNS) {
                $domain = substr(strrchr($email, "@"), 1);
                if (!checkdnsrr($domain, "MX") && !checkdnsrr($domain, "A")) {
                    $this->errors[$field] = "E-Mail-Domain existiert nicht.";
                }
            }
        }
        return $this;
    }
    
    /**
     * Min length
     */
    public function minLength($field, $min, $message = null) {
        if (!empty($this->data[$field])) {
            if (mb_strlen($this->data[$field]) < $min) {
                $this->errors[$field] = $message ?: "Mindestens $min Zeichen erforderlich.";
            }
        }
        return $this;
    }
    
    /**
     * Max length
     */
    public function maxLength($field, $max, $message = null) {
        if (!empty($this->data[$field])) {
            if (mb_strlen($this->data[$field]) > $max) {
                $this->errors[$field] = $message ?: "Maximal $max Zeichen erlaubt.";
            }
        }
        return $this;
    }
    
    /**
     * Exact length
     */
    public function exactLength($field, $length, $message = null) {
        if (!empty($this->data[$field])) {
            if (mb_strlen($this->data[$field]) !== $length) {
                $this->errors[$field] = $message ?: "Muss genau $length Zeichen haben.";
            }
        }
        return $this;
    }
    
    /**
     * Numeric value
     */
    public function numeric($field, $message = null) {
        if (!empty($this->data[$field])) {
            if (!is_numeric($this->data[$field])) {
                $this->errors[$field] = $message ?: "Muss eine Zahl sein.";
            }
        }
        return $this;
    }
    
    /**
     * Integer value
     */
    public function integer($field, $message = null) {
        if (!empty($this->data[$field])) {
            if (!filter_var($this->data[$field], FILTER_VALIDATE_INT)) {
                $this->errors[$field] = $message ?: "Muss eine ganze Zahl sein.";
            }
        }
        return $this;
    }
    
    /**
     * Float value
     */
    public function float($field, $message = null) {
        if (!empty($this->data[$field])) {
            if (!filter_var($this->data[$field], FILTER_VALIDATE_FLOAT)) {
                $this->errors[$field] = $message ?: "Muss eine Dezimalzahl sein.";
            }
        }
        return $this;
    }
    
    /**
     * Min value (numeric)
     */
    public function min($field, $min, $message = null) {
        if (!empty($this->data[$field])) {
            if ($this->data[$field] < $min) {
                $this->errors[$field] = $message ?: "Muss mindestens $min sein.";
            }
        }
        return $this;
    }
    
    /**
     * Max value (numeric)
     */
    public function max($field, $max, $message = null) {
        if (!empty($this->data[$field])) {
            if ($this->data[$field] > $max) {
                $this->errors[$field] = $message ?: "Darf maximal $max sein.";
            }
        }
        return $this;
    }
    
    /**
     * Between values
     */
    public function between($field, $min, $max, $message = null) {
        if (!empty($this->data[$field])) {
            $value = $this->data[$field];
            if ($value < $min || $value > $max) {
                $this->errors[$field] = $message ?: "Muss zwischen $min und $max liegen.";
            }
        }
        return $this;
    }
    
    /**
     * URL validation
     */
    public function url($field, $message = null) {
        if (!empty($this->data[$field])) {
            if (!XSS::validateURL($this->data[$field])) {
                $this->errors[$field] = $message ?: "Ungültige URL.";
            }
        }
        return $this;
    }
    
    /**
     * IP address validation
     */
    public function ip($field, $message = null) {
        if (!empty($this->data[$field])) {
            if (!filter_var($this->data[$field], FILTER_VALIDATE_IP)) {
                $this->errors[$field] = $message ?: "Ungültige IP-Adresse.";
            }
        }
        return $this;
    }
    
    /**
     * Alphanumeric only
     */
    public function alphanumeric($field, $message = null) {
        if (!empty($this->data[$field])) {
            if (!ctype_alnum($this->data[$field])) {
                $this->errors[$field] = $message ?: "Nur Buchstaben und Zahlen erlaubt.";
            }
        }
        return $this;
    }
    
    /**
     * Alpha only (letters)
     */
    public function alpha($field, $message = null) {
        if (!empty($this->data[$field])) {
            if (!ctype_alpha($this->data[$field])) {
                $this->errors[$field] = $message ?: "Nur Buchstaben erlaubt.";
            }
        }
        return $this;
    }
    
    /**
     * Slug validation (lowercase, numbers, hyphens)
     */
    public function slug($field, $message = null) {
        if (!empty($this->data[$field])) {
            if (!preg_match('/^[a-z0-9-]+$/', $this->data[$field])) {
                $this->errors[$field] = $message ?: "Nur Kleinbuchstaben, Zahlen und Bindestriche.";
            }
        }
        return $this;
    }
    
    /**
     * Username validation
     */
    public function username($field, $message = null) {
        if (!empty($this->data[$field])) {
            if (!preg_match('/^[a-zA-Z0-9_.-]{3,20}$/', $this->data[$field])) {
                $this->errors[$field] = $message ?: "3-20 Zeichen: Buchstaben, Zahlen, _, -, .";
            }
        }
        return $this;
    }
    
    /**
     * Password strength validation
     */
    public function password($field, $message = null) {
        if (!empty($this->data[$field])) {
            $password = $this->data[$field];
            
            // Min 8 characters
            if (strlen($password) < 8) {
                $this->errors[$field] = "Passwort muss mindestens 8 Zeichen haben.";
                return $this;
            }
            
            // At least one uppercase
            if (!preg_match('/[A-Z]/', $password)) {
                $this->errors[$field] = "Passwort muss einen Großbuchstaben enthalten.";
                return $this;
            }
            
            // At least one lowercase
            if (!preg_match('/[a-z]/', $password)) {
                $this->errors[$field] = "Passwort muss einen Kleinbuchstaben enthalten.";
                return $this;
            }
            
            // At least one number
            if (!preg_match('/[0-9]/', $password)) {
                $this->errors[$field] = "Passwort muss eine Zahl enthalten.";
                return $this;
            }
            
            // Optional: Special character
            // if (!preg_match('/[!@#$%^&*(),.?":{}|<>]/', $password)) {
            //     $this->errors[$field] = "Passwort muss ein Sonderzeichen enthalten.";
            // }
        }
        return $this;
    }
    
    /**
     * Match another field (password confirmation)
     */
    public function matches($field, $matchField, $message = null) {
        if (!empty($this->data[$field]) && !empty($this->data[$matchField])) {
            if ($this->data[$field] !== $this->data[$matchField]) {
                $this->errors[$field] = $message ?: "Die Felder stimmen nicht überein.";
            }
        }
        return $this;
    }
    
    /**
     * Pattern matching (regex)
     */
    public function pattern($field, $pattern, $message = null) {
        if (!empty($this->data[$field])) {
            if (!preg_match($pattern, $this->data[$field])) {
                $this->errors[$field] = $message ?: "Ungültiges Format.";
            }
        }
        return $this;
    }
    
    /**
     * In array (whitelist)
     */
    public function in($field, array $allowed, $message = null) {
        if (!empty($this->data[$field])) {
            if (!in_array($this->data[$field], $allowed, true)) {
                $this->errors[$field] = $message ?: "Ungültiger Wert.";
            }
        }
        return $this;
    }
    
    /**
     * Not in array (blacklist)
     */
    public function notIn($field, array $forbidden, $message = null) {
        if (!empty($this->data[$field])) {
            if (in_array($this->data[$field], $forbidden, true)) {
                $this->errors[$field] = $message ?: "Dieser Wert ist nicht erlaubt.";
            }
        }
        return $this;
    }
    
    /**
     * Date validation
     */
    public function date($field, $format = 'Y-m-d', $message = null) {
        if (!empty($this->data[$field])) {
            $d = DateTime::createFromFormat($format, $this->data[$field]);
            if (!$d || $d->format($format) !== $this->data[$field]) {
                $this->errors[$field] = $message ?: "Ungültiges Datum (Format: $format).";
            }
        }
        return $this;
    }
    
    /**
     * File upload validation
     */
    public function file($field, $options = [], $message = null) {
        if (!empty($_FILES[$field]['name'])) {
            $file = $_FILES[$field];
            
            // Check upload errors
            if ($file['error'] !== UPLOAD_ERR_OK) {
                $this->errors[$field] = "Datei-Upload fehlgeschlagen.";
                return $this;
            }
            
            // Check max size (in bytes)
            if (isset($options['max_size']) && $file['size'] > $options['max_size']) {
                $maxMB = round($options['max_size'] / 1024 / 1024, 1);
                $this->errors[$field] = "Datei zu groß (max. {$maxMB}MB).";
                return $this;
            }
            
            // Check allowed extensions
            if (isset($options['extensions'])) {
                $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
                if (!in_array($ext, $options['extensions'])) {
                    $allowed = implode(', ', $options['extensions']);
                    $this->errors[$field] = "Nur folgende Dateitypen erlaubt: $allowed";
                    return $this;
                }
            }
            
            // Check MIME type
            if (isset($options['mime_types'])) {
                $finfo = finfo_open(FILEINFO_MIME_TYPE);
                $mime = finfo_file($finfo, $file['tmp_name']);
                finfo_close($finfo);
                
                if (!in_array($mime, $options['mime_types'])) {
                    $this->errors[$field] = "Ungültiger Dateityp.";
                    return $this;
                }
            }
        }
        return $this;
    }
    
    /**
     * Custom validation function
     */
    public function custom($field, callable $callback, $message = null) {
        if (!empty($this->data[$field])) {
            if (!$callback($this->data[$field], $this->data)) {
                $this->errors[$field] = $message ?: "Validierung fehlgeschlagen.";
            }
        }
        return $this;
    }
    
    /**
     * Check if validation passed
     */
    public function isValid() {
        return empty($this->errors);
    }
    
    /**
     * Get all errors
     */
    public function getErrors() {
        return $this->errors;
    }
    
    /**
     * Get first error
     */
    public function getFirstError() {
        return !empty($this->errors) ? reset($this->errors) : null;
    }
    
    /**
     * Add custom error
     */
    public function addError($field, $message) {
        $this->errors[$field] = $message;
        return $this;
    }
    
    /**
     * Static helper for quick validation
     */
    public static function validate($data, $rules) {
        $validator = new self($data);
        
        foreach ($rules as $field => $ruleset) {
            foreach ($ruleset as $rule => $params) {
                if (is_numeric($rule)) {
                    $rule = $params;
                    $params = [];
                } elseif (!is_array($params)) {
                    $params = [$params];
                }
                
                if (method_exists($validator, $rule)) {
                    call_user_func_array([$validator, $rule], array_merge([$field], $params));
                }
            }
        }
        
        return $validator;
    }
}

/**
 * Helper functions for common validations
 */

function validate_email($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

function validate_password_strength($password) {
    return strlen($password) >= 8 &&
           preg_match('/[A-Z]/', $password) &&
           preg_match('/[a-z]/', $password) &&
           preg_match('/[0-9]/', $password);
}

function sanitize_string($string) {
    return trim(strip_tags($string));
}

function sanitize_int($value) {
    return (int) filter_var($value, FILTER_SANITIZE_NUMBER_INT);
}

function sanitize_email($email) {
    return filter_var($email, FILTER_SANITIZE_EMAIL);
}

function sanitize_url($url) {
    return filter_var($url, FILTER_SANITIZE_URL);
}