<?php
/**
 * cms-core/backend/settings/farb_font.php
 * COMPLETE: Global Color & Font Customization with Header & Footer
 */

class FarbFont {
    
    const FONT_FAMILIES = [
        'Arial' => 'Arial, sans-serif',
        'Helvetica' => 'Helvetica, sans-serif',
        'Georgia' => 'Georgia, serif',
        'Times New Roman' => '"Times New Roman", serif',
        'Courier New' => '"Courier New", monospace',
        'Verdana' => 'Verdana, sans-serif',
        'Trebuchet MS' => '"Trebuchet MS", sans-serif',
        'Impact' => 'Impact, sans-serif',
        'Comic Sans MS' => '"Comic Sans MS", cursive',
        'Palatino' => 'Palatino, serif',
        'Garamond' => 'Garamond, serif',
        'Bookman' => 'Bookman, serif',
        'Tahoma' => 'Tahoma, sans-serif',
        'Lucida Console' => '"Lucida Console", monospace'
    ];
    
    const FONT_WEIGHTS = [
        '300' => 'Light',
        '400' => 'Normal',
        '500' => 'Medium',
        '600' => 'Semi-Bold',
        '700' => 'Bold',
        '800' => 'Extra-Bold',
        '900' => 'Black'
    ];
    
    public static function getAllStyles() {
        $styles = db()->fetchAll(
            "SELECT setting_key, setting_value, setting_type FROM custom_global_styles"
        );
        
        $result = [];
        foreach ($styles as $style) {
            $result[$style['setting_key']] = $style['setting_value'];
        }
        
        return $result;
    }
    
    public static function getStyle($key, $default = null) {
        $style = db()->fetchOne(
            "SELECT setting_value FROM custom_global_styles WHERE setting_key = ?",
            [$key]
        );
        
        return $style ? $style['setting_value'] : $default;
    }
    
    public static function saveStyle($key, $value, $type = 'text') {
        if ($type === 'color' && !self::validateHexColor($value)) {
            throw new Exception('Ungültiger Hex-Farbcode: ' . $value);
        }
        
        $existing = db()->fetchOne(
            "SELECT id FROM custom_global_styles WHERE setting_key = ?",
            [$key]
        );
        
        if ($existing) {
            db()->execute(
                "UPDATE custom_global_styles SET setting_value = ?, setting_type = ? WHERE setting_key = ?",
                [$value, $type, $key]
            );
        } else {
            db()->insert(
                "INSERT INTO custom_global_styles (setting_key, setting_value, setting_type) VALUES (?, ?, ?)",
                [$key, $value, $type]
            );
        }
        
        return true;
    }
    
    public static function saveMultipleStyles($styles) {
        foreach ($styles as $key => $value) {
            $type = 'text';
            if (strpos($key, '_color') !== false) {
                $type = 'color';
            } elseif (strpos($key, '_font') !== false && !strpos($key, '_weight')) {
                $type = 'font';
            }
            
            self::saveStyle($key, $value, $type);
        }
        
        return true;
    }
    
    private static function validateHexColor($color) {
        return preg_match('/^#[0-9A-F]{6}$/i', $color);
    }
    
    /**
     * COMPLETE: Get CSS Variables with Header & Footer
     */
    public static function getCSSVariables() {
        $styles = self::getAllStyles();
        
        // Helper function to get font family CSS value
        $getFont = function($key, $default) use ($styles) {
            $fontName = $styles[$key] ?? $default;
            return self::FONT_FAMILIES[$fontName] ?? $fontName;
        };
        
        $css = "/* ===========================\n";
        $css .= "   GLOBAL FARB & FONT SETTINGS\n";
        $css .= "   Generated: " . date('Y-m-d H:i:s') . "\n";
        $css .= "   Priority: Inline Styles > FarbFont Global > Default\n";
        $css .= "   =========================== */\n\n";
        
        // Define CSS Variables
        $css .= ":root {\n";
        
        // H1 Styles
        $css .= "    /* H1 Styles */\n";
        $css .= "    --global-h1-color: " . ($styles['h1_color'] ?? '#667eea') . ";\n";
        $css .= "    --global-h1-font: " . $getFont('h1_font', 'Arial') . ";\n";
        $css .= "    --global-h1-weight: " . ($styles['h1_weight'] ?? '700') . ";\n\n";
        
        // H2 Styles
        $css .= "    /* H2 Styles */\n";
        $css .= "    --global-h2-color: " . ($styles['h2_color'] ?? '#4a5568') . ";\n";
        $css .= "    --global-h2-font: " . $getFont('h2_font', 'Arial') . ";\n";
        $css .= "    --global-h2-weight: " . ($styles['h2_weight'] ?? '600') . ";\n\n";
        
        // H3 Styles
        $css .= "    /* H3 Styles */\n";
        $css .= "    --global-h3-color: " . ($styles['h3_color'] ?? '#4a5568') . ";\n";
        $css .= "    --global-h3-font: " . $getFont('h3_font', 'Arial') . ";\n";
        $css .= "    --global-h3-weight: " . ($styles['h3_weight'] ?? '600') . ";\n\n";
        
        // Text/Paragraph Styles
        $css .= "    /* Text/Paragraph Styles */\n";
        $css .= "    --global-text-color: " . ($styles['text_color'] ?? '#2d3748') . ";\n";
        $css .= "    --global-text-font: " . $getFont('text_font', 'Arial') . ";\n";
        $css .= "    --global-text-weight: " . ($styles['text_weight'] ?? '400') . ";\n\n";
        
        // Content Link Styles
        $css .= "    /* Content Link Styles (in text) */\n";
        $css .= "    --global-link-color: " . ($styles['link_color'] ?? '#667eea') . ";\n";
        $css .= "    --global-link-hover-color: " . ($styles['link_hover_color'] ?? '#5a67d8') . ";\n\n";
        
        // Button Styles
        $css .= "    /* Button Styles */\n";
        $css .= "    --global-button-bg-color: " . ($styles['button_bg_color'] ?? '#667eea') . ";\n";
        $css .= "    --global-button-text-color: " . ($styles['button_text_color'] ?? '#ffffff') . ";\n";
        $css .= "    --global-button-font: " . $getFont('button_font', 'Arial') . ";\n";
        $css .= "    --global-button-weight: " . ($styles['button_weight'] ?? '600') . ";\n\n";
        
        // Body/Section Backgrounds
        $css .= "    /* Body & Section Backgrounds */\n";
        $css .= "    --global-body-bg-color: " . ($styles['body_bg_color'] ?? '#ffffff') . ";\n";
        $css .= "    --global-section-bg-color: " . ($styles['section_bg_color'] ?? '#f9fafb') . ";\n\n";
        
        // Header Styles
        $css .= "    /* Header Styles */\n";
        $css .= "    --global-header-bg-color: " . ($styles['header_bg_color'] ?? '#ffffff') . ";\n";
        $css .= "    --global-header-text-color: " . ($styles['header_text_color'] ?? '#2d3748') . ";\n";
        $css .= "    --global-header-heading-color: " . ($styles['header_heading_color'] ?? '#1f2937') . ";\n";
        $css .= "    --global-header-link-color: " . ($styles['header_link_color'] ?? '#4b5563') . ";\n";
        $css .= "    --global-header-link-hover-color: " . ($styles['header_link_hover_color'] ?? '#667eea') . ";\n\n";
        
        // Footer Styles
        $css .= "    /* Footer Styles */\n";
        $css .= "    --global-footer-bg-color: " . ($styles['footer_bg_color'] ?? '#1a202c') . ";\n";
        $css .= "    --global-footer-text-color: " . ($styles['footer_text_color'] ?? '#ffffff') . ";\n";
        $css .= "    --global-footer-heading-color: " . ($styles['footer_heading_color'] ?? '#ffffff') . ";\n";
        $css .= "    --global-footer-link-color: " . ($styles['footer_link_color'] ?? '#d1d5db') . ";\n";
        $css .= "    --global-footer-link-hover-color: " . ($styles['footer_link_hover_color'] ?? '#ffffff') . ";\n";
        
        $css .= "}\n\n";
        
        // ============================================
        // APPLY STYLES TO ACTUAL ELEMENTS
        // Priority: Inline styles > FarbFont > Default
        // ============================================
        
        $css .= "/* Apply H1 Styles - RESPECTS INLINE STYLES */\n";
        $css .= "h1:not([style*=\"color\"]), .h1:not([style*=\"color\"]), .page-header h1:not([style*=\"color\"]) {\n";
        $css .= "    color: var(--global-h1-color) !important;\n";
        $css .= "}\n";
        $css .= "h1:not([style*=\"font-family\"]), .h1:not([style*=\"font-family\"]), .page-header h1:not([style*=\"font-family\"]) {\n";
        $css .= "    font-family: var(--global-h1-font) !important;\n";
        $css .= "}\n";
        $css .= "h1:not([style*=\"font-weight\"]), .h1:not([style*=\"font-weight\"]), .page-header h1:not([style*=\"font-weight\"]) {\n";
        $css .= "    font-weight: var(--global-h1-weight) !important;\n";
        $css .= "}\n";
        $css .= "/* Remove gradient from .page-header h1 when FarbFont is active */\n";
        $css .= ".page-header h1:not([style*=\"color\"]) {\n";
        $css .= "    background: none !important;\n";
        $css .= "    -webkit-background-clip: unset !important;\n";
        $css .= "    -webkit-text-fill-color: unset !important;\n";
        $css .= "    background-clip: unset !important;\n";
        $css .= "}\n\n";
        
        $css .= "/* Apply H2 Styles - RESPECTS INLINE STYLES */\n";
        $css .= "h2:not([style*=\"color\"]), .h2:not([style*=\"color\"]) {\n";
        $css .= "    color: var(--global-h2-color) !important;\n";
        $css .= "}\n";
        $css .= "h2:not([style*=\"font-family\"]), .h2:not([style*=\"font-family\"]) {\n";
        $css .= "    font-family: var(--global-h2-font) !important;\n";
        $css .= "}\n";
        $css .= "h2:not([style*=\"font-weight\"]), .h2:not([style*=\"font-weight\"]) {\n";
        $css .= "    font-weight: var(--global-h2-weight) !important;\n";
        $css .= "}\n\n";
        
        $css .= "/* Apply H3 Styles - RESPECTS INLINE STYLES */\n";
        $css .= "h3:not([style*=\"color\"]), .h3:not([style*=\"color\"]) {\n";
        $css .= "    color: var(--global-h3-color) !important;\n";
        $css .= "}\n";
        $css .= "h3:not([style*=\"font-family\"]), .h3:not([style*=\"font-family\"]) {\n";
        $css .= "    font-family: var(--global-h3-font) !important;\n";
        $css .= "}\n";
        $css .= "h3:not([style*=\"font-weight\"]), .h3:not([style*=\"font-weight\"]) {\n";
        $css .= "    font-weight: var(--global-h3-weight) !important;\n";
        $css .= "}\n\n";
        
        $css .= "/* Apply Text/Paragraph Styles - RESPECTS INLINE STYLES */\n";
        $css .= "body:not([style*=\"color\"]), p:not([style*=\"color\"]), .text:not([style*=\"color\"]), div:not([style*=\"color\"]), span:not([style*=\"color\"]), li:not([style*=\"color\"]) {\n";
        $css .= "    color: var(--global-text-color);\n";
        $css .= "}\n";
        $css .= "body:not([style*=\"font-family\"]), p:not([style*=\"font-family\"]), .text:not([style*=\"font-family\"]), div:not([style*=\"font-family\"]), span:not([style*=\"font-family\"]), li:not([style*=\"font-family\"]) {\n";
        $css .= "    font-family: var(--global-text-font);\n";
        $css .= "}\n";
        $css .= "body:not([style*=\"font-weight\"]), p:not([style*=\"font-weight\"]), .text:not([style*=\"font-weight\"]), div:not([style*=\"font-weight\"]), span:not([style*=\"font-weight\"]), li:not([style*=\"font-weight\"]) {\n";
        $css .= "    font-weight: var(--global-text-weight);\n";
        $css .= "}\n\n";
        
        $css .= "/* Apply Link Styles - EXCLUDING BUTTONS & RESPECTS INLINE STYLES */\n";
        $css .= "a:not([style*=\"color\"]):not(.nav-link):not(.menu-link):not(.navbar a):not(.site-header a):not(.site-footer a):not(.btn):not([class*=\"btn-\"]) {\n";
        $css .= "    color: var(--global-link-color, #667eea) !important;\n";
        $css .= "}\n";
        $css .= "a:not([style*=\"color\"]):not(.nav-link):not(.menu-link):not(.navbar a):not(.site-header a):not(.site-footer a):not(.btn):not([class*=\"btn-\"]):hover {\n";
        $css .= "    color: var(--global-link-hover-color, #5a67d8);\n";
        $css .= "}\n\n";
        
        $css .= "/* Apply Button Styles - USE BUTTON TEXT COLOR, NOT LINK COLOR */\n";
        $css .= ".btn, .button, button[type='submit'], .btn-primary, .btn-secondary, .btn-white, [class*=\"btn-\"] {\n";
        $css .= "    background-color: var(--global-button-bg-color) !important;\n";
        $css .= "    color: var(--global-button-text-color) !important;\n";
        $css .= "    font-family: var(--global-button-font) !important;\n";
        $css .= "    font-weight: var(--global-button-weight) !important;\n";
        $css .= "}\n\n";
        
        $css .= "/* Ensure button links don't inherit link colors */\n";
        $css .= "a.btn, a.btn-primary, a.btn-secondary, a.btn-white, a[class*=\"btn-\"] {\n";
        $css .= "    color: var(--global-button-text-color) !important;\n";
        $css .= "    text-decoration: none;\n";
        $css .= "}\n\n";
        
$css .= "/* Apply Header Styles */\n";
$css .= ".site-header, header {\n";
$css .= "    background-color: var(--global-header-bg-color) !important;\n";
$css .= "}\n";
$css .= ".site-header:not([style*=\"color\"]), header:not([style*=\"color\"]) {\n";
$css .= "    color: var(--global-header-text-color) !important;\n";
$css .= "}\n";
$css .= "/* Logo/Site Title - uses header_heading_color from farb_font */\n";
$css .= ".site-header .logo-text:not([style*=\"color\"]), .site-header .site-title:not([style*=\"color\"]), .header-logo .logo-text:not([style*=\"color\"]) {\n";
$css .= "    color: var(--global-header-heading-color) !important;\n";
$css .= "    background: none !important;\n";
$css .= "    -webkit-background-clip: unset !important;\n";
$css .= "    -webkit-text-fill-color: unset !important;\n";
$css .= "    background-clip: unset !important;\n";
$css .= "}\n";
$css .= "/* Navigation Links */\n";
$css .= ".site-header a:not([style*=\"color\"]):not(.logo-text), .navbar a:not([style*=\"color\"]), .menu-link:not([style*=\"color\"]), .nav-link:not([style*=\"color\"]), header a:not([style*=\"color\"]):not(.logo-text) {\n";
$css .= "    color: var(--global-header-link-color) !important;\n";
$css .= "}\n";
$css .= ".site-header a:not([style*=\"color\"]):not(.logo-text):hover, .navbar a:not([style*=\"color\"]):hover, .menu-link:not([style*=\"color\"]):hover, .nav-link:not([style*=\"color\"]):hover, .nav-link.active, header a:not([style*=\"color\"]):not(.logo-text):hover {\n";
$css .= "    color: var(--global-header-link-hover-color) !important;\n";
$css .= "}\n\n";
        
        $css .= "/* Apply Footer Styles */\n";
        $css .= ".site-footer, footer {\n";
        $css .= "    background-color: var(--global-footer-bg-color) !important;\n";
        $css .= "}\n";
        $css .= ".site-footer:not([style*=\"color\"]), footer:not([style*=\"color\"]), .site-footer *:not([style*=\"color\"]), footer *:not([style*=\"color\"]), .site-footer p:not([style*=\"color\"]), .site-footer div:not([style*=\"color\"]), .site-footer span:not([style*=\"color\"]), .site-footer li:not([style*=\"color\"]), .footer-section p:not([style*=\"color\"]), .footer-contact li:not([style*=\"color\"]) {\n";
        $css .= "    color: var(--global-footer-text-color) !important;\n";
        $css .= "}\n";
        $css .= ".site-footer h3:not([style*=\"color\"]), .site-footer h4:not([style*=\"color\"]), .footer-section h3:not([style*=\"color\"]), .footer-section h4:not([style*=\"color\"]) {\n";
        $css .= "    color: var(--global-footer-heading-color) !important;\n";
        $css .= "}\n";
        $css .= ".site-footer a:not([style*=\"color\"]), .footer-links a:not([style*=\"color\"]) {\n";
        $css .= "    color: var(--global-footer-link-color) !important;\n";
        $css .= "}\n";
        $css .= ".site-footer a:not([style*=\"color\"]):hover, .footer-links a:not([style*=\"color\"]):hover {\n";
        $css .= "    color: var(--global-footer-link-hover-color) !important;\n";
        $css .= "}\n\n";
        
        $css .= "/* Apply Body Background */\n";
        $css .= "body {\n";
        $css .= "    background-color: var(--global-body-bg-color) !important;\n";
        $css .= "}\n\n";
        
        $css .= "/* Apply Section/Block Background */\n";
        $css .= ".content-block, .block, .section-block, .flexible-block, section {\n";
        $css .= "    background-color: var(--global-section-bg-color);\n";
        $css .= "}\n\n";
        
        return $css;
    }
    
    /**
     * Reset to defaults
     */
    public static function resetToDefaults() {
        return db()->execute("DELETE FROM custom_global_styles");
    }
}
?>