<?php
/** cms-core/backend/settings/settings_manager.php */

/**
 * Settings Manager
 * Verwaltet Systemeinstellungen
 */

class SettingsManager {
    
    /**
     * Holt alle Einstellungen
     */
    public static function getAll() {
        $settings = db()->fetchAll("SELECT * FROM settings ORDER BY setting_key");
        
        $result = [];
        foreach ($settings as $setting) {
            $value = $setting['setting_value'];
            
            // Type-Casting
            if ($setting['setting_type'] === 'boolean') {
                $value = filter_var($value, FILTER_VALIDATE_BOOLEAN);
            } elseif ($setting['setting_type'] === 'json') {
                $value = json_decode($value, true);
            }
            
            $result[$setting['setting_key']] = $value;
        }
        
        return $result;
    }
    
    /**
     * Speichert mehrere Einstellungen
     */
    public static function saveMultiple($settings) {
        foreach ($settings as $key => $value) {
            // Bestimme Typ
            $type = 'text';
            if (is_bool($value)) {
                $type = 'boolean';
                $value = $value ? '1' : '0';
            } elseif (is_array($value)) {
                $type = 'json';
                $value = json_encode($value);
            }
            
            set_setting($key, $value, $type);
        }
        
        return true;
    }
    
    /**
     * Holt Einstellungsgruppen
     */
    public static function getGroups() {
        return [
            'general' => [
                'title' => 'Allgemein',
                'icon' => '⚙️',
                'settings' => [
                    'site_name' => [
                        'label' => 'Website-Name',
                        'type' => 'text',
                        'description' => 'Name Ihrer Website'
                    ],
                    'site_email' => [
                        'label' => 'Kontakt-E-Mail',
                        'type' => 'email',
                        'description' => 'Haupt-E-Mail-Adresse'
                    ],
                    'maintenance_mode' => [
                        'label' => 'Wartungsmodus',
                        'type' => 'boolean',
                        'description' => 'Website für Besucher deaktivieren'
                    ]
                ]
            ],
            'tracking' => [
                'title' => 'Tracking & Analytics',
                'icon' => '📊',
                'settings' => [
                    'google_analytics_id' => [
                        'label' => 'Google Analytics ID',
                        'type' => 'text',
                        'placeholder' => 'G-XXXXXXXXXX',
                        'description' => 'Tracking-ID aus Google Analytics'
                    ],
                    'google_analytics_enabled' => [
                        'label' => 'Google Analytics aktivieren',
                        'type' => 'boolean'
                    ]
                ]
            ],
            'social' => [
                'title' => 'Social Media Pixel',
                'icon' => '📱',
                'settings' => [
                    'tiktok_pixel_id' => [
                        'label' => 'TikTok Pixel ID',
                        'type' => 'text',
                        'description' => 'Pixel-ID aus TikTok Ads Manager'
                    ],
                    'tiktok_pixel_enabled' => [
                        'label' => 'TikTok Pixel aktivieren',
                        'type' => 'boolean'
                    ],
                    'instagram_pixel_id' => [
                        'label' => 'Meta/Instagram Pixel ID',
                        'type' => 'text',
                        'description' => 'Pixel-ID aus Meta Business Manager'
                    ],
                    'instagram_pixel_enabled' => [
                        'label' => 'Meta Pixel aktivieren',
                        'type' => 'boolean'
                    ]
                ]
            ],
            'privacy' => [
                'title' => 'Datenschutz',
                'icon' => '🔒',
                'settings' => [
                    'cookie_consent_enabled' => [
                        'label' => 'Cookie-Banner aktivieren',
                        'type' => 'boolean',
                        'description' => 'DSGVO-konformes Cookie-Consent-Banner'
                    ]
                ]
            ],
            'api' => [
                'title' => 'API-Schlüssel',
                'icon' => '🔑',
                'settings' => [
                    'deepl_api_key' => [
                        'label' => 'DeepL API Key',
                        'type' => 'password',
                        'description' => 'Für automatische Übersetzungen'
                    ]
                ]
            ]
        ];
    }
    
    /**
     * Validiert Einstellungen
     */
    public static function validate($settings) {
        $errors = [];
        
        // E-Mail-Validierung
        if (isset($settings['site_email']) && !empty($settings['site_email'])) {
            if (!filter_var($settings['site_email'], FILTER_VALIDATE_EMAIL)) {
                $errors['site_email'] = 'Ungültige E-Mail-Adresse';
            }
        }
        
        // Google Analytics ID Format
        if (isset($settings['google_analytics_id']) && !empty($settings['google_analytics_id'])) {
            if (!preg_match('/^(G-|UA-|GT-)/', $settings['google_analytics_id'])) {
                $errors['google_analytics_id'] = 'Ungültiges Format (sollte mit G-, UA- oder GT- beginnen)';
            }
        }
        
        return [
            'valid' => empty($errors),
            'errors' => $errors
        ];
    }
}