<?php
/** cms-core/backend/settings/site_info_manager.php - COMPLETE */

/**
 * Site Info Manager
 * Manages extended website information, contact data, and footer configuration
 */

class SiteInfoManager {
    
    /**
     * Get all site info settings grouped
     */
    public static function getAll() {
        return [
            // Website Basic Information
            'site_name' => get_setting('site_name', 'KynoCanis'),
            'site_description' => get_setting('site_description', ''),
            'use_logo' => (bool)get_setting('use_logo', false),
            'site_logo' => get_setting('site_logo', ''),
            
            // Contact Information
            'contact_email' => get_setting('contact_email', 'info@kynocanis.de'),
            'contact_phone' => get_setting('contact_phone', ''),
            'contact_address_line1' => get_setting('contact_address_line1', ''),
            'contact_address_line2' => get_setting('contact_address_line2', ''),
            'contact_opening_hours' => get_setting('contact_opening_hours', ''),
            
            // Footer Settings
            'footer_show_address' => (bool)get_setting('footer_show_address', false),
            'footer_company_description' => get_setting('footer_company_description', ''),
            
            // Social Media
            'social_facebook' => get_setting('social_facebook', ''),
            'social_instagram' => get_setting('social_instagram', ''),
            'social_linkedin' => get_setting('social_linkedin', ''),
            'social_youtube' => get_setting('social_youtube', ''),
            'social_twitter' => get_setting('social_twitter', ''),
        ];
    }
    
    /**
     * Save site info settings
     */
    public static function save($data) {
        $allowed = [
            'site_name', 'site_description', 'use_logo', 'site_logo',
            'contact_email', 'contact_phone', 
            'contact_address_line1', 'contact_address_line2', 
            'contact_opening_hours',
            'footer_show_address', 'footer_company_description',
            'social_facebook', 'social_instagram', 'social_linkedin',
            'social_youtube', 'social_twitter'
        ];
        
        foreach ($data as $key => $value) {
            if (in_array($key, $allowed)) {
                // Handle boolean values
                if ($key === 'use_logo' || $key === 'footer_show_address') {
                    set_setting($key, $value ? '1' : '0', 'boolean');
                } else {
                    set_setting($key, trim($value), 'text');
                }
            }
        }
        
        return true;
    }
    
    /**
     * Validate inputs
     */
    public static function validate($data) {
        $errors = [];
        
        // Website name is required
        if (empty(trim($data['site_name'] ?? ''))) {
            $errors['site_name'] = 'Website-Name ist erforderlich.';
        }
        
        // Email validation
        if (!empty($data['contact_email']) && !filter_var($data['contact_email'], FILTER_VALIDATE_EMAIL)) {
            $errors['contact_email'] = 'Ungültige E-Mail-Adresse.';
        }
        
        // Phone validation (optional, only check format if filled)
        if (!empty($data['contact_phone'])) {
            $phone = preg_replace('/[^0-9+\s\-\/()]/', '', $data['contact_phone']);
            if (strlen($phone) < 5) {
                $errors['contact_phone'] = 'Ungültige Telefonnummer.';
            }
        }
        
        // URL validation for social media
        $socialFields = ['social_facebook', 'social_instagram', 'social_linkedin', 'social_youtube', 'social_twitter'];
        foreach ($socialFields as $field) {
            if (!empty($data[$field]) && !filter_var($data[$field], FILTER_VALIDATE_URL)) {
                $errors[$field] = 'Ungültige URL.';
            }
        }
        
        return [
            'valid' => empty($errors),
            'errors' => $errors
        ];
    }
    
    /**
     * Get contact information (for contact page)
     */
    public static function getContactInfo() {
        return [
            'email' => get_setting('contact_email', 'info@kynocanis.de'),
            'phone' => get_setting('contact_phone', ''),
            'address_line1' => get_setting('contact_address_line1', ''),
            'address_line2' => get_setting('contact_address_line2', ''),
            'opening_hours' => get_setting('contact_opening_hours', ''),
        ];
    }
    
    /**
     * Get footer configuration
     */
    public static function getFooterConfig() {
        return [
            'show_address' => (bool)get_setting('footer_show_address', false),
            'description' => get_setting('footer_company_description', ''),
            'address_line1' => get_setting('contact_address_line1', ''),
            'address_line2' => get_setting('contact_address_line2', ''),
            'email' => get_setting('contact_email', ''),
            'phone' => get_setting('contact_phone', ''),
        ];
    }
    
    /**
     * Get social media links (only filled ones)
     */
    public static function getSocialLinks() {
        $links = [
            'facebook' => [
                'url' => get_setting('social_facebook', ''),
                'icon' => '📘',
                'label' => 'Facebook'
            ],
            'instagram' => [
                'url' => get_setting('social_instagram', ''),
                'icon' => '📷',
                'label' => 'Instagram'
            ],
            'linkedin' => [
                'url' => get_setting('social_linkedin', ''),
                'icon' => '💼',
                'label' => 'LinkedIn'
            ],
            'youtube' => [
                'url' => get_setting('social_youtube', ''),
                'icon' => '🎥',
                'label' => 'YouTube'
            ],
            'twitter' => [
                'url' => get_setting('social_twitter', ''),
                'icon' => '🐦',
                'label' => 'Twitter'
            ],
        ];
        
        // Only return filled links
        return array_filter($links, function($link) {
            return !empty($link['url']);
        });
    }
    
    /**
     * Check if complete address is filled
     */
    public static function hasCompleteAddress() {
        $line1 = get_setting('contact_address_line1', '');
        $line2 = get_setting('contact_address_line2', '');
        
        return !empty(trim($line1)) && !empty(trim($line2));
    }
    
    /**
     * Get logo info
     */
    public static function getLogoInfo() {
        return [
            'use_logo' => (bool)get_setting('use_logo', false),
            'logo_path' => get_setting('site_logo', ''),
            'site_name' => get_setting('site_name', 'KynoCanis')
        ];
    }
}