<?php
/**
 * cms-core/backend/updates/UpdateChecker.php
 * Update System with License Verification
 * 
 * Features:
 * - Works WITHOUT license (no updates)
 * - WITH license = Updates available
 * - Checks license at manidesign.de API
 * - Lifetime licenses
 * - No forced licensing
 */

class UpdateChecker {
    
    const API_URL = 'https://manidesign.de/wp-json/manicms/v1/';
    const VERSION_FILE = APP_ROOT . '/cms-core/VERSION.txt';
    
    private static $instance = null;
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Get current installed version
     * NEW METHOD - Required by dashboard
     */
    public static function getCurrentVersion() {
        if (file_exists(self::VERSION_FILE)) {
            $version = trim(file_get_contents(self::VERSION_FILE));
            if (!empty($version)) {
                return $version;
            }
        }
        
        // Fallback: Try from settings
        $version = get_setting('cms_version', '1.0');
        return $version;
    }
    
    /**
     * Check for updates
     * Returns update info if license is valid, otherwise info about licensing
     */
    public static function checkForUpdates() {
        $currentVersion = self::getCurrentVersion();
        
        // Check if we have a valid license
        if (!defined('LICENSE_VALID') || !LICENSE_VALID || empty(LICENSE_KEY)) {
            return [
                'success' => false,
                'has_license' => false,
                'current_version' => $currentVersion,
                'message' => 'Keine Lizenz - Updates deaktiviert',
                'license_required' => true
            ];
        }
        
        try {
            // Call manidesign.de API
            $ch = curl_init(self::API_URL . 'check-update');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
                'license_key' => LICENSE_KEY,
                'current_version' => $currentVersion,
                'domain' => $_SERVER['HTTP_HOST'] ?? 'unknown',
                'php_version' => PHP_VERSION
            ]));
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'User-Agent: ManiCMS/' . $currentVersion
            ]);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode !== 200 || !$response) {
                error_log('Update check failed: API request failed (HTTP ' . $httpCode . ')');
                return [
                    'success' => false,
                    'current_version' => $currentVersion,
                    'message' => 'Update-Server nicht erreichbar',
                    'offline_mode' => true
                ];
            }
            
            $data = json_decode($response, true);
            
            if (!$data) {
                error_log('Update check failed: Invalid JSON response');
                return [
                    'success' => false,
                    'current_version' => $currentVersion,
                    'message' => 'Ungültige Antwort vom Server'
                ];
            }
            
            // WordPress REST API might wrap response in 'data' key
            if (isset($data['data']) && is_array($data['data'])) {
                $data = $data['data'];
            }
            
            // Check if update is available by comparing versions
            $latestVersion = $data['latest_version'] ?? $data['version'] ?? $currentVersion;
            
            // CRITICAL: Remove any extra version segments (1.0.0 -> 1.0)
            // Convert both to same format for comparison
            $latestParts = explode('.', $latestVersion);
            $currentParts = explode('.', $currentVersion);
            
            // If we use 2-digit versions (1.0), normalize both
            if (count($currentParts) == 2) {
                $latestVersion = implode('.', array_slice($latestParts, 0, 2));
            }
            
            $updateAvailable = version_compare($latestVersion, $currentVersion, '>');
            
            return [
                'success' => true,
                'has_license' => true,
                'update_available' => $updateAvailable,
                'latest_version' => $latestVersion,
                'current_version' => $currentVersion,
                'download_url' => $data['download_url'] ?? null,
                'changelog' => $data['changelog'] ?? '',
                'release_date' => $data['release_date'] ?? '',
                'message' => $updateAvailable ? 'Update verfügbar' : 'Keine Updates verfügbar'
            ];
            
        } catch (Exception $e) {
            error_log('Update check failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'current_version' => $currentVersion,
                'message' => 'Fehler beim Update-Check',
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Verify license with ManiDesign API
     */
    public static function verifyLicense($licenseKey) {
        if (empty($licenseKey)) {
            return [
                'valid' => false,
                'message' => 'Kein Lizenz-Schlüssel angegeben'
            ];
        }
        
        try {
            $ch = curl_init(self::API_URL . 'verify-license');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
                'license_key' => $licenseKey,
                'domain' => $_SERVER['HTTP_HOST'] ?? 'unknown',
                'email' => get_setting('site_email', '')
            ]));
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode !== 200 || !$response) {
                return [
                    'valid' => false,
                    'message' => 'Lizenz-Server nicht erreichbar'
                ];
            }
            
            $data = json_decode($response, true);
            
            if ($data && isset($data['valid']) && $data['valid'] === true) {
                return [
                    'valid' => true,
                    'message' => 'Lizenz erfolgreich aktiviert!',
                    'customer_name' => $data['customer_name'] ?? '',
                    'license_type' => $data['license_type'] ?? 'lifetime',
                    'issued_date' => $data['issued_date'] ?? ''
                ];
            }
            
            return [
                'valid' => false,
                'message' => $data['message'] ?? 'Ungültiger Lizenz-Schlüssel'
            ];
            
        } catch (Exception $e) {
            error_log('License verification failed: ' . $e->getMessage());
            
            return [
                'valid' => false,
                'message' => 'Fehler bei der Lizenz-Verifizierung'
            ];
        }
    }
    
    /**
     * Download update (only with valid license)
     */
    public static function downloadUpdate($downloadUrl, $licenseKey) {
        if (empty($licenseKey)) {
            return [
                'success' => false,
                'message' => 'Lizenz-Schlüssel erforderlich'
            ];
        }
        
        try {
            // Create temp directory
            $tempDir = APP_ROOT . '/temp/updates/';
            if (!is_dir($tempDir)) {
                mkdir($tempDir, 0755, true);
            }
            
            $zipFile = $tempDir . 'update_' . time() . '.zip';
            
            // Download with license authentication
            $ch = curl_init($downloadUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'X-License-Key: ' . $licenseKey,
                'X-Domain: ' . ($_SERVER['HTTP_HOST'] ?? ''),
            ]);
            
            $data = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode !== 200 || !$data) {
                return [
                    'success' => false,
                    'message' => 'Download fehlgeschlagen (HTTP ' . $httpCode . ')'
                ];
            }
            
            // Save zip file
            file_put_contents($zipFile, $data);
            
            return [
                'success' => true,
                'zip_file' => $zipFile,
                'message' => 'Update erfolgreich heruntergeladen'
            ];
            
        } catch (Exception $e) {
            error_log('Update download failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Download fehlgeschlagen: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Install downloaded update
     */
    public static function installUpdate($zipFile) {
        // Create backup first
        $backupResult = self::createBackup();
        if (!$backupResult['success']) {
            return [
                'success' => false,
                'message' => 'Backup fehlgeschlagen - Update abgebrochen'
            ];
        }
        
        try {
            $zip = new ZipArchive();
            
            if ($zip->open($zipFile) === true) {
                $extractPath = APP_ROOT . '/cms-core/';
                
                // Extract
                $zip->extractTo($extractPath);
                $zip->close();
                
                // Delete zip
                unlink($zipFile);
                
                // Update version
                $newVersion = self::getCurrentVersion(); // Re-read after extraction
                set_setting('cms_version', $newVersion, 'text');
                
                // Log update
                self::logUpdate($newVersion, 'success');
                
                return [
                    'success' => true,
                    'message' => 'Update erfolgreich installiert!',
                    'new_version' => $newVersion,
                    'backup_file' => $backupResult['backup_file']
                ];
            }
            
            return [
                'success' => false,
                'message' => 'ZIP-Datei konnte nicht entpackt werden'
            ];
            
        } catch (Exception $e) {
            error_log('Update installation failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Installation fehlgeschlagen: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Create backup before update
     */
    private static function createBackup() {
        try {
            $backupDir = APP_ROOT . '/backups/';
            if (!is_dir($backupDir)) {
                mkdir($backupDir, 0755, true);
            }
            
            $backupFile = $backupDir . 'backup_' . date('Y-m-d_H-i-s') . '.zip';
            
            $zip = new ZipArchive();
            if ($zip->open($backupFile, ZipArchive::CREATE) === true) {
                self::addDirectoryToZip($zip, APP_ROOT . '/cms-core', 'cms-core');
                $zip->close();
                
                return [
                    'success' => true,
                    'backup_file' => $backupFile
                ];
            }
            
            return ['success' => false];
            
        } catch (Exception $e) {
            error_log('Backup failed: ' . $e->getMessage());
            return ['success' => false];
        }
    }
    
    /**
     * Add directory to zip recursively
     */
    private static function addDirectoryToZip($zip, $dir, $zipPath) {
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir),
            RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ($files as $file) {
            if (!$file->isDir()) {
                $filePath = $file->getRealPath();
                $relativePath = $zipPath . '/' . substr($filePath, strlen($dir) + 1);
                $zip->addFile($filePath, $relativePath);
            }
        }
    }
    
    /**
     * Log update activity
     */
    private static function logUpdate($version, $status) {
        $logFile = LOGS_ROOT . '/updates.log';
        $logEntry = date('Y-m-d H:i:s') . ' - Update zu Version ' . $version . ': ' . $status . PHP_EOL;
        file_put_contents($logFile, $logEntry, FILE_APPEND);
    }
    
    /**
     * Get license status for display
     */
    public static function getLicenseStatus() {
        if (!defined('LICENSE_KEY') || empty(LICENSE_KEY)) {
            return [
                'has_license' => false,
                'status' => 'unlicensed',
                'message' => 'Keine Lizenz - Updates deaktiviert',
                'can_update' => false
            ];
        }
        
        $status = LICENSE_VALID ? 'active' : 'inactive';
        
        return [
            'has_license' => true,
            'status' => $status,
            'license_key' => LICENSE_KEY,
            'license_key_masked' => substr(LICENSE_KEY, 0, 9) . '...',
            'activated_at' => LICENSE_ACTIVATED ?? null,
            'can_update' => LICENSE_VALID,
            'message' => $status === 'active' ? 'Lizenziert - Updates verfügbar' : 'Lizenz inaktiv'
        ];
    }
}