<?php
/** cms-core/config/config.php - COMPLETE WITH LICENSE SUPPORT */

// ============================================
// STEP 1: Load .env file
// ============================================
// From cms-core/config/ we need to go UP 2 levels to reach ROOT
// cms-core/config/ -> cms-core/ -> ROOT (htdocs/ManiDesign-KynoCanis/)
$envFile = dirname(dirname(__DIR__)) . '/.env';

if (!file_exists($envFile)) {
    die("❌ CRITICAL ERROR: .env file not found at: $envFile<br>Looking in: " . dirname(dirname(__DIR__)));
}

function parseEnvFile($file) {
    $env = [];
    $lines = file($file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    
    if ($lines === false) {
        return false;
    }
    
    foreach ($lines as $line) {
        $line = trim($line);
        
        // Skip comments and empty lines
        if (empty($line) || $line[0] === '#' || $line[0] === ';') {
            continue;
        }
        
        // Parse KEY=VALUE
        if (strpos($line, '=') !== false) {
            list($key, $value) = explode('=', $line, 2);
            $key = trim($key);
            $value = trim($value, " \t\n\r\0\x0B\"'");
            $env[$key] = $value;
        }
    }
    
    return $env;
}

$env = parseEnvFile($envFile);

if ($env === false || empty($env)) {
    die("❌ ERROR: Could not parse .env file");
}

// ============================================
// STEP 2: Database Settings
// ============================================
define('DB_HOST', $env['DB_HOST'] ?? '');
define('DB_NAME', $env['DB_NAME'] ?? '');
define('DB_USER', $env['DB_USER'] ?? '');
define('DB_PASS', $env['DB_PASS'] ?? '');
define('DB_CHARSET', $env['DB_CHARSET'] ?? 'utf8mb4');

// ============================================
// STEP 3: Application Settings
// ============================================
define('APP_URL', $env['APP_URL'] ?? '');
define('APP_ENV', $env['APP_ENV'] ?? 'production');
define('APP_DEBUG', filter_var($env['APP_DEBUG'] ?? 'false', FILTER_VALIDATE_BOOLEAN));
define('APP_LANGUAGE', $env['APP_LANGUAGE'] ?? 'de'); // 🌍 Language setting
define('FORCE_SSL', filter_var($env['FORCE_SSL'] ?? 'true', FILTER_VALIDATE_BOOLEAN));

// ============================================
// STEP 4: Security Settings
// ============================================
define('SESSION_NAME', $env['SESSION_NAME'] ?? 'CMS_SESSION');
define('SESSION_LIFETIME', (int)($env['SESSION_LIFETIME'] ?? 7200));
define('CSRF_TOKEN_NAME', $env['CSRF_TOKEN_NAME'] ?? 'csrf_token');

// CRITICAL: Encryption key for sensitive data (API keys, etc.)
define('ENCRYPTION_KEY', $env['ENCRYPTION_KEY'] ?? 'CHANGE_ME_TO_RANDOM_32_CHAR_STRING');

// Validate encryption key in production
if (ENCRYPTION_KEY === 'CHANGE_ME_TO_RANDOM_32_CHAR_STRING' && APP_ENV === 'production') {
    error_log('⚠️ SECURITY WARNING: ENCRYPTION_KEY not set in .env file!');
}

// ============================================
// STEP 4.5: LICENSE CONFIGURATION
// ============================================
define('LICENSE_KEY', $env['LICENSE_KEY'] ?? null);
define('LICENSE_VALID', filter_var($env['LICENSE_VALID'] ?? 'false', FILTER_VALIDATE_BOOLEAN));
define('LICENSE_ACTIVATED', $env['LICENSE_ACTIVATED'] ?? null);

// License API URL
define('LICENSE_API_URL', 'https://manidesign.de/wp-json/manicms/v1');

// ============================================
// STEP 5: PATH DEFINITIONS
// ============================================
// CRITICAL: APP_ROOT = Project Root Directory
// htdocs/ManiDesign-KynoCanis/
if (!defined('APP_ROOT')) {
    define('APP_ROOT', dirname(dirname(__DIR__)));
}

// Verify APP_ROOT is correct
if (!file_exists(APP_ROOT . '/.env')) {
    die("❌ ERROR: APP_ROOT is incorrect!<br>APP_ROOT: " . APP_ROOT . "<br>.env not found at: " . APP_ROOT . '/.env');
}

// ============================================
// FILESYSTEM PATHS (Server Perspective)
// ============================================

// Main CMS Directories
define('CMS_ROOT', APP_ROOT . '/cms-core');
define('BACKEND_ROOT', CMS_ROOT . '/backend');
define('PUBLIC_ROOT', CMS_ROOT . '/public');
define('WEBVIEW_ROOT', PUBLIC_ROOT . '/webview');

// Root Level Folders (parallel to cms-core)
define('MEDIA_ROOT', APP_ROOT . '/media');
define('CUSTOM_ROOT', APP_ROOT . '/customizations');
define('LOGS_ROOT', APP_ROOT . '/logs');

// Customizations Subfolders
define('CUSTOM_FEATURES_PATH', CUSTOM_ROOT . '/features');
define('CUSTOM_ADMIN_PATH', CUSTOM_ROOT . '/admin-pages');
define('CUSTOM_FRONTEND_PATH', CUSTOM_ROOT . '/frontend');
define('CUSTOM_BLOG_PATH', CUSTOM_ROOT . '/blog'); // Blog posts storage

// Assets Path (inside webview)
define('ASSETS_ROOT', WEBVIEW_ROOT . '/assets');

// ============================================
// URL PATHS (Browser Perspective)
// ============================================
// IONOS Shared Hosting: DocumentRoot = htdocs/ManiDesign-KynoCanis/ (project root)
// .htaccess handles routing to webview automatically
// Browser sees: / = project root, but .htaccess redirects internally

if (!empty(APP_URL)) {
    define('BASE_URL', rtrim(APP_URL, '/'));
} else {
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    define('BASE_URL', $protocol . $host);
}

// Browser-facing URLs
define('MEDIA_URL', BASE_URL . '/media');        // -> htdocs/ManiDesign-KynoCanis/media/
define('ASSETS_URL', BASE_URL . '/assets');      // -> cms-core/public/webview/assets/
define('CUSTOM_URL', BASE_URL . '/customizations'); // -> htdocs/ManiDesign-KynoCanis/customizations/
define('ADMIN_URL', BASE_URL . '/admin');        // -> cms-core/public/webview/admin/
define('BLOG_URL', BASE_URL . '/blog');          // -> Blog overview and posts

// ============================================
// STEP 6: Error Reporting
// ============================================
// Create logs directory if it doesn't exist
if (!is_dir(LOGS_ROOT)) {
    @mkdir(LOGS_ROOT, 0755, true);
}

if (APP_DEBUG) {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
    ini_set('log_errors', 1);
    ini_set('error_log', LOGS_ROOT . '/error.log');
} else {
    error_reporting(0);
    ini_set('display_errors', 0);
    ini_set('log_errors', 1);
    ini_set('error_log', LOGS_ROOT . '/error.log');
}

// ============================================
// STEP 7: Timezone
// ============================================
date_default_timezone_set($env['TIMEZONE'] ?? 'Europe/Berlin');

// ============================================
// STEP 8: Email Settings
// ============================================
define('MAIL_FROM', $env['MAIL_FROM'] ?? 'noreply@' . ($_SERVER['HTTP_HOST'] ?? 'localhost'));
define('MAIL_FROM_NAME', $env['MAIL_FROM_NAME'] ?? 'Website');

// ============================================
// STEP 9: Rate Limiting Settings
// ============================================
define('MAX_LOGIN_ATTEMPTS', (int)($env['MAX_LOGIN_ATTEMPTS'] ?? 5));
define('LOGIN_LOCKOUT_TIME', (int)($env['LOGIN_LOCKOUT_TIME'] ?? 900));

// ============================================
// STEP 10: FILE UPLOAD SECURITY
// ============================================

// Maximum upload size (10MB)
if (!defined('MAX_UPLOAD_SIZE')) {
    define('MAX_UPLOAD_SIZE', 10 * 1024 * 1024);
}

// Allowed image types (ARRAY - CRITICAL FOR SECURITY TEST)
if (!defined('ALLOWED_IMAGE_TYPES')) {
    define('ALLOWED_IMAGE_TYPES', ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg']);
}

// Allowed document types (ARRAY)
if (!defined('ALLOWED_DOC_TYPES')) {
    define('ALLOWED_DOC_TYPES', ['pdf', 'doc', 'docx', 'txt']);
}

// Dangerous file extensions to block (ARRAY - CRITICAL FOR SECURITY TEST)
if (!defined('BLOCKED_EXTENSIONS')) {
    define('BLOCKED_EXTENSIONS', [
        'php', 'phtml', 'php3', 'php4', 'php5', 'php7', 'phps',
        'exe', 'bat', 'cmd', 'sh', 'bash',
        'js', 'jsp', 'asp', 'aspx',
        'cgi', 'pl', 'py',
        'htaccess', 'htpasswd'
    ]);
}

// Validate uploaded file extension
if (!function_exists('validate_upload_extension')) {
    function validate_upload_extension($filename) {
        $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        
        // Check if blocked
        if (in_array($ext, BLOCKED_EXTENSIONS)) {
            return false;
        }
        
        // Check if in allowed list
        $allowed = array_merge(ALLOWED_IMAGE_TYPES, ALLOWED_DOC_TYPES);
        return in_array($ext, $allowed);
    }
}

// Validate MIME type
if (!function_exists('validate_upload_mime')) {
    function validate_upload_mime($filepath) {
        if (!function_exists('mime_content_type')) {
            return true; // Skip if function not available
        }
        
        $mime = mime_content_type($filepath);
        
        $allowedMimes = [
            // Images
            'image/jpeg', 'image/png', 'image/gif', 'image/webp', 'image/svg+xml',
            // Documents
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'text/plain'
        ];
        
        return in_array($mime, $allowedMimes);
    }
}

// ============================================
// STEP 11: Load Database Connection
// ============================================
// Database.php is in the same directory as config.php
require_once __DIR__ . '/database.php';