<?php
/** cms-core/config/database.php - SECURE WITH TRY-CATCH */

// Config.php should already be loaded and constants defined
if (!defined('DB_HOST')) {
    require_once __DIR__ . '/config.php';
}

class Database {
    private static $instance = null;
    private $connection;
    
    private function __construct() {
        $this->connect();
    }
    
    private function connect() {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                // OPTIMIZATION: Use persistent connections (reuse connections)
                PDO::ATTR_PERSISTENT => true,
            ];
            
            $this->connection = new PDO($dsn, DB_USER, DB_PASS, $options);
            
            // OPTIMIZATION: Set additional performance attributes AFTER connection
            $this->connection->setAttribute(PDO::ATTR_STRINGIFY_FETCHES, false);
            $this->connection->setAttribute(PDO::MYSQL_ATTR_USE_BUFFERED_QUERY, true);
            $this->connection->setAttribute(PDO::MYSQL_ATTR_COMPRESS, true);
            
        } catch (PDOException $e) {
            error_log('❌ Database connection failed: ' . $e->getMessage());
            if (defined('APP_DEBUG') && APP_DEBUG) {
                die('❌ Database Error: ' . $e->getMessage());
            } else {
                die('❌ Database connection failed. Please contact administrator.');
            }
        }
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function getConnection() {
        return $this->connection;
    }
    
    /**
     * Execute query with automatic try-catch
     */
    public function query($sql, $params = []) {
        try {
            $stmt = $this->connection->prepare($sql);
            $stmt->execute($params);
            return $stmt;
        } catch (PDOException $e) {
            error_log('❌ Query failed: ' . $e->getMessage() . ' | SQL: ' . $sql);
            
            // Log to Logger if available
            if (class_exists('Logger')) {
                Logger::error('Database query failed', [
                    'sql' => $sql,
                    'params' => $params,
                    'error' => $e->getMessage()
                ]);
            }
            
            if (defined('APP_DEBUG') && APP_DEBUG) {
                throw $e;
            }
            
            return false;
        }
    }
    
    /**
     * Fetch one row with try-catch
     */
    public function fetchOne($sql, $params = []) {
        try {
            $stmt = $this->query($sql, $params);
            if ($stmt === false) {
                return null;
            }
            return $stmt->fetch();
        } catch (Exception $e) {
            error_log('❌ fetchOne failed: ' . $e->getMessage());
            if (class_exists('Logger')) {
                Logger::error('Database fetchOne failed', [
                    'sql' => $sql,
                    'error' => $e->getMessage()
                ]);
            }
            return null;
        }
    }
    
    /**
     * Fetch all rows with try-catch
     */
    public function fetchAll($sql, $params = []) {
        try {
            $stmt = $this->query($sql, $params);
            if ($stmt === false) {
                return [];
            }
            return $stmt->fetchAll();
        } catch (Exception $e) {
            error_log('❌ fetchAll failed: ' . $e->getMessage());
            if (class_exists('Logger')) {
                Logger::error('Database fetchAll failed', [
                    'sql' => $sql,
                    'error' => $e->getMessage()
                ]);
            }
            return [];
        }
    }
    
    /**
     * Insert with try-catch and return last insert ID
     */
    public function insert($sql, $params = []) {
        try {
            $stmt = $this->query($sql, $params);
            if ($stmt === false) {
                return false;
            }
            return $this->connection->lastInsertId();
        } catch (Exception $e) {
            error_log('❌ Insert failed: ' . $e->getMessage());
            if (class_exists('Logger')) {
                Logger::error('Database insert failed', [
                    'sql' => $sql,
                    'error' => $e->getMessage()
                ]);
            }
            return false;
        }
    }
    
    /**
     * Execute (UPDATE/DELETE) with try-catch
     */
    public function execute($sql, $params = []) {
        try {
            $stmt = $this->query($sql, $params);
            if ($stmt === false) {
                return 0;
            }
            return $stmt->rowCount();
        } catch (Exception $e) {
            error_log('❌ Execute failed: ' . $e->getMessage());
            if (class_exists('Logger')) {
                Logger::error('Database execute failed', [
                    'sql' => $sql,
                    'error' => $e->getMessage()
                ]);
            }
            return 0;
        }
    }
    
    /**
     * Batch insert for better performance
     * NEW METHOD - Inside the class!
     */
    public function batchInsert($table, array $data) {
        if (empty($data)) {
            return 0;
        }
        
        try {
            $keys = array_keys($data[0]);
            $placeholders = '(' . implode(',', array_fill(0, count($keys), '?')) . ')';
            $values = [];
            
            foreach ($data as $row) {
                $values = array_merge($values, array_values($row));
            }
            
            $sql = "INSERT INTO $table (" . implode(',', $keys) . ") VALUES " . 
                   implode(',', array_fill(0, count($data), $placeholders));
            
            return $this->execute($sql, $values);
        } catch (Exception $e) {
            error_log('❌ Batch insert failed: ' . $e->getMessage());
            if (class_exists('Logger')) {
                Logger::error('Database batch insert failed', [
                    'table' => $table,
                    'error' => $e->getMessage()
                ]);
            }
            return 0;
        }
    }
    
    /**
     * Begin transaction with try-catch
     */
    public function beginTransaction() {
        try {
            return $this->connection->beginTransaction();
        } catch (PDOException $e) {
            error_log('❌ Begin transaction failed: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Commit transaction with try-catch
     */
    public function commit() {
        try {
            return $this->connection->commit();
        } catch (PDOException $e) {
            error_log('❌ Commit failed: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Rollback transaction with try-catch
     */
    public function rollback() {
        try {
            return $this->connection->rollBack();
        } catch (PDOException $e) {
            error_log('❌ Rollback failed: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Check if table exists
     */
    public function tableExists($tableName) {
        try {
            $result = $this->fetchOne("SHOW TABLES LIKE ?", [$tableName]);
            return $result !== null;
        } catch (Exception $e) {
            error_log('❌ Table exists check failed: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get table column info
     */
    public function getColumns($tableName) {
        try {
            return $this->fetchAll("SHOW COLUMNS FROM $tableName");
        } catch (Exception $e) {
            error_log('❌ Get columns failed: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Sanitize table/column names (for dynamic queries)
     */
    public function sanitizeIdentifier($identifier) {
        return preg_replace('/[^a-zA-Z0-9_]/', '', $identifier);
    }
    
    private function __clone() {}
    
    public function __wakeup() {
        throw new Exception("Cannot unserialize singleton");
    }
}

// Global helper function
if (!function_exists('db')) {
    function db() {
        return Database::getInstance();
    }
}