<?php
/** cms-core/public/webview/admin/blog_create_edit.php */

require_once __DIR__ . '/../../../backend/authorization/check_admin.php';

$postId = get_param('id') ? (int)get_param('id') : null;
$isEdit = $postId !== null;

// Get post data if edit mode
$post = null;
if ($isEdit) {
    $post = db()->fetchOne("SELECT * FROM blog_posts WHERE id = ?", [$postId]);
    
    if (!$post) {
        set_flash('error', 'Blog-Post nicht gefunden.');
        redirect('/cms-core/public/webview/admin/blog_manage.php');
		exit;
    }
}

$errors = [];

// Handle Form Submit
if (is_post()) {
    CSRF::require();
    
    $title = trim(post('title'));
    $slug = trim(post('slug'));
    $excerpt = trim(post('excerpt'));
    $content = post('content'); // Don't trim HTML
    $status = post('status', 'draft');
    
    // Validation
    if (empty($title)) {
        $errors['title'] = 'Titel ist erforderlich.';
    } elseif (strlen($title) < 3) {
        $errors['title'] = 'Titel muss mindestens 3 Zeichen lang sein.';
    }
    
    if (empty($slug)) {
        $errors['slug'] = 'Slug ist erforderlich.';
    } elseif (!preg_match('/^[a-z0-9-]+$/', $slug)) {
        $errors['slug'] = 'Slug darf nur Kleinbuchstaben, Zahlen und Bindestriche enthalten.';
    }
    
    if (empty($content)) {
        $errors['content'] = 'Inhalt ist erforderlich.';
    } elseif (strlen($content) < 10) {
        $errors['content'] = 'Inhalt muss mindestens 10 Zeichen lang sein.';
    }
    
    // Check slug uniqueness
    if (empty($errors['slug'])) {
        $slugCheck = db()->fetchOne(
            "SELECT id FROM blog_posts WHERE slug = ? AND id != ?",
            [$slug, $postId ?? 0]
        );
        
        if ($slugCheck) {
            $errors['slug'] = 'Dieser Slug wird bereits verwendet.';
        }
    }
    
    // Image upload - FIXED: Use APP_ROOT for /media/blog (outside cms-core)
    $featuredImage = $post['featured_image'] ?? null; // Keep existing image by default
    
    if (isset($_FILES['featured_image']) && $_FILES['featured_image']['error'] !== UPLOAD_ERR_NO_FILE) {
        // Debug logging
        error_log("Blog image upload attempt - File info: " . print_r($_FILES['featured_image'], true));
        error_log("Upload directory: " . APP_ROOT . '/media/blog');
        
        // Only proceed if a file was actually selected
        if ($_FILES['featured_image']['error'] === UPLOAD_ERR_OK) {
            // FIXED: Use APP_ROOT to reach media folder in root
            $uploadResult = handle_file_upload(
                'featured_image',
                APP_ROOT . '/media/blog'
            );
            
            error_log("Upload result: " . print_r($uploadResult, true));
            
            if ($uploadResult['success']) {
                // Delete old image ONLY if a new one was successfully uploaded
                if ($featuredImage && $featuredImage !== $uploadResult['filename']) {
                    delete_file(APP_ROOT . '/media/blog/' . $featuredImage);
                }
                $featuredImage = $uploadResult['filename'];
                error_log("Image uploaded successfully: " . $featuredImage);
            } else {
                $errors['featured_image'] = $uploadResult['error'];
                error_log("Image upload failed: " . $uploadResult['error']);
            }
        }
    }
    // If no new file uploaded, $featuredImage keeps the existing value from database
    
    // Save if no errors
    if (empty($errors)) {
        $user = current_user();
        
        if ($isEdit) {
            // Update
            db()->execute(
                "UPDATE blog_posts SET 
                    title = ?, 
                    slug = ?, 
                    excerpt = ?, 
                    content = ?, 
                    featured_image = ?, 
                    status = ?,
                    updated_at = NOW(),
                    published_at = IF(status = 'draft' AND ? = 'published', NOW(), published_at)
                WHERE id = ?",
                [$title, $slug, $excerpt, $content, $featuredImage, $status, $status, $postId]
            );
            
            set_flash('success', 'Blog-Post erfolgreich aktualisiert.');
                redirect('/cms-core/public/webview/admin/blog_create_edit.php?id=' . $postId); // FIXED - already correct
    exit;
        } else {
            // Insert
            $newId = db()->insert(
                "INSERT INTO blog_posts (title, slug, excerpt, content, featured_image, author_id, status, published_at) 
                 VALUES (?, ?, ?, ?, ?, ?, ?, IF(? = 'published', NOW(), NULL))",
                [$title, $slug, $excerpt, $content, $featuredImage, $user['id'], $status, $status]
            );
            
            set_flash('success', 'Blog-Post erfolgreich erstellt.');
                redirect('/cms-core/public/webview/admin/blog_create_edit.php?id=' . $postId); // FIXED - already correct
    exit;
        }
        exit; // Ensure no code runs after redirect
    }
}

$flashMessages = get_flash();
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $isEdit ? 'Blog-Post bearbeiten' : 'Neuer Blog-Post' ?></title>
    <?php
$adminCssPath = APP_ROOT . '/assets/css/admin.css';
$adminCssVersion = file_exists($adminCssPath) ? filemtime($adminCssPath) : time();
?>
<link rel="stylesheet" href="/assets/css/admin.css?v=<?= $adminCssVersion ?>"> 
    <!-- Quill Rich Text Editor -->
    <link href="https://cdn.quilljs.com/1.3.6/quill.snow.css" rel="stylesheet">
    <script src="https://cdn.quilljs.com/1.3.6/quill.js"></script>
    
    <style>
    #editor-container {
        height: 400px;
        background: white;
        border: 1px solid #e5e7eb;
        border-radius: 8px;
    }
    .ql-editor {
        min-height: 350px;
        font-size: 15px;
        line-height: 1.6;
    }
    .ql-toolbar {
        border-radius: 8px 8px 0 0;
        background: #f9fafb;
    }
    </style>
</head>
<body>
    <?php include 'includes/admin_header.php'; ?>
    
    <div class="admin-container">
        <?php include 'includes/admin_sidebar.php'; ?>
        
        <main class="admin-content">
            <?php foreach ($flashMessages as $flash): ?>
                <div class="alert alert-<?= e($flash['type']) ?>">
                    <?= e($flash['message']) ?>
                </div>
            <?php endforeach; ?>
            
            <div class="page-header">
                <h1><?= $isEdit ? '✏️ Blog-Post bearbeiten' : '➕ Neuer Blog-Post' ?></h1>
                <p><?= $isEdit ? 'Bearbeite deinen Blog-Beitrag' : 'Erstelle einen neuen Blog-Beitrag' ?></p>
            </div>
            
            <?php if (!empty($errors)): ?>
                <div class="alert alert-error" style="margin-bottom: 20px;">
                    <strong>Fehler beim Speichern:</strong>
                    <ul style="margin: 10px 0 0 20px;">
                        <?php foreach ($errors as $field => $error): ?>
                            <li><?= e($error) ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>
            
            <form method="POST" enctype="multipart/form-data" id="blog-form">
                <?= CSRF::field() ?>
                
                <!-- Hidden field for content -->
                <input type="hidden" name="content" id="content-input">
                
                <div style="display: grid; grid-template-columns: 2fr 1fr; gap: 20px;">
                    <!-- Main Content -->
                    <div>
                        <div class="content-card">
                            <div class="form-group">
                                <label class="form-label" for="title">Titel *</label>
                                <input 
                                    type="text" 
                                    id="title" 
                                    name="title" 
                                    class="form-input" 
                                    value="<?= e($post['title'] ?? post('title')) ?>"
                                    required
                                    onkeyup="generateSlug()"
                                >
                                <?php if (isset($errors['title'])): ?>
                                    <span style="color: #ef4444; font-size: 13px;"><?= e($errors['title']) ?></span>
                                <?php endif; ?>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label" for="slug">Slug (URL) *</label>
                                <input 
                                    type="text" 
                                    id="slug" 
                                    name="slug" 
                                    class="form-input" 
                                    value="<?= e($post['slug'] ?? post('slug')) ?>"
                                    required
                                >
                                <span style="font-size: 12px; color: #6b7280;">Nur Kleinbuchstaben, Zahlen und Bindestriche</span>
                                <?php if (isset($errors['slug'])): ?>
                                    <span style="color: #ef4444; font-size: 13px; display: block;"><?= e($errors['slug']) ?></span>
                                <?php endif; ?>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label" for="excerpt">Kurzbeschreibung (Excerpt)</label>
                                <textarea 
                                    id="excerpt" 
                                    name="excerpt" 
                                    class="form-textarea"
                                    rows="3"
                                ><?= e($post['excerpt'] ?? post('excerpt')) ?></textarea>
                                <span style="font-size: 12px; color: #6b7280;">Kurze Zusammenfassung für Vorschau</span>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">Inhalt *</label>
                                <div id="editor-container"></div>
                                <?php if (isset($errors['content'])): ?>
                                    <span style="color: #ef4444; font-size: 13px; display: block; margin-top: 5px;"><?= e($errors['content']) ?></span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Sidebar -->
                    <div>
                        <!-- Status & Publish -->
                        <div class="content-card" style="margin-bottom: 20px;">
                            <h3 style="margin: 0 0 15px 0;">📊 Status</h3>
                            
                            <div class="form-group">
                                <label class="form-label" for="status">Status</label>
                                <select id="status" name="status" class="form-input">
                                    <option value="draft" <?= ($post['status'] ?? 'draft') === 'draft' ? 'selected' : '' ?>>
                                        🔸 Entwurf
                                    </option>
                                    <option value="published" <?= ($post['status'] ?? '') === 'published' ? 'selected' : '' ?>>
                                        ✅ Veröffentlicht
                                    </option>
                                </select>
                            </div>
                            
                            <?php if ($isEdit && $post['published_at']): ?>
                                <div style="font-size: 13px; color: #6b7280; margin-top: 10px;">
                                    Veröffentlicht am: <?= date('d.m.Y H:i', strtotime($post['published_at'])) ?>
                                </div>
                            <?php endif; ?>
                            
                            <button type="submit" class="btn btn-primary" style="width: 100%; margin-top: 15px;">
                                💾 <?= $isEdit ? 'Aktualisieren' : 'Erstellen' ?>
                            </button>
                        </div>
                        
                        <!-- Featured Image -->
                        <div class="content-card">
                            <h3 style="margin: 0 0 15px 0;">🖼️ Beitragsbild</h3>
                            
                            <?php if ($post && $post['featured_image']): ?>
                                <div style="margin-bottom: 15px;">
                                    <img src="/media.php?file=blog/<?= e($post['featured_image']) ?>" 
                                         alt="Featured Image" 
                                         style="width: 100%; height: auto; border-radius: 8px; border: 1px solid #e5e7eb;"
                                         onerror="this.src='/assets/images/placeholder.jpg'">
                                </div>
                            <?php endif; ?>
                            
                            <div class="form-group">
                                <label class="form-label" for="featured_image">
                                    <?= ($post && $post['featured_image']) ? 'Bild ersetzen' : 'Bild hochladen' ?>
                                </label>
                                <input 
                                    type="file" 
                                    id="featured_image" 
                                    name="featured_image" 
                                    class="form-input"
                                    accept="image/*"
                                    onchange="previewImage(this)"
                                >
                                <span style="font-size: 12px; color: #6b7280;">Max. 5MB, JPG/PNG/GIF</span>
                                <?php if (isset($errors['featured_image'])): ?>
                                    <span style="color: #ef4444; font-size: 13px; display: block;"><?= e($errors['featured_image']) ?></span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </form>
        </main>
    </div>
    
    <script>
    // Initialize Quill editor
    var quill = new Quill('#editor-container', {
        theme: 'snow',
        modules: {
            toolbar: [
                [{ 'header': [1, 2, 3, false] }],
                ['bold', 'italic', 'underline', 'strike'],
                [{ 'align': [] }],
                [{ 'list': 'ordered'}, { 'list': 'bullet' }],
                ['link', 'image'],
                ['clean']
            ]
        },
        placeholder: 'Schreibe hier deinen Blog-Inhalt...'
    });
    
    // Load existing content if editing
    <?php if ($post && $post['content']): ?>
    quill.root.innerHTML = <?= json_encode($post['content']) ?>;
    <?php endif; ?>
    
    // Sync Quill content to hidden input before submit
    document.getElementById('blog-form').addEventListener('submit', function(e) {
        // Get content from Quill
        var content = quill.root.innerHTML;
        document.getElementById('content-input').value = content;
        
        // Validate content
        if (!content || content === '<p><br></p>' || content.trim() === '') {
            e.preventDefault();
            alert('Bitte füge Inhalt hinzu.');
            return false;
        }
        
        // Validate slug format
        var slug = document.getElementById('slug').value;
        if (!/^[a-z0-9-]+$/.test(slug)) {
            e.preventDefault();
            alert('Slug darf nur Kleinbuchstaben, Zahlen und Bindestriche enthalten.');
            return false;
        }
        
        return true;
    });
    
    function generateSlug() {
        const title = document.getElementById('title').value;
        const slugInput = document.getElementById('slug');
        
        // Only auto-generate if slug is empty (new post)
        if (!slugInput.value || <?= $isEdit ? 'false' : 'true' ?>) {
            const slug = title
                .toLowerCase()
                .replace(/ä/g, 'ae')
                .replace(/ö/g, 'oe')
                .replace(/ü/g, 'ue')
                .replace(/ß/g, 'ss')
                .replace(/[^a-z0-9]+/g, '-')
                .replace(/^-+|-+$/g, '');
            
            slugInput.value = slug;
        }
    }
    
    function previewImage(input) {
        if (input.files && input.files[0]) {
            var reader = new FileReader();
            reader.onload = function(e) {
                // Find or create preview container
                var container = document.getElementById('image-preview-container');
                if (!container) {
                    container = document.createElement('div');
                    container.id = 'image-preview-container';
                    container.style.cssText = 'margin-bottom: 15px;';
                    input.closest('.form-group').insertAdjacentElement('beforebegin', container);
                }
                
                // Update or create image
                var img = document.getElementById('current-featured-image');
                if (!img) {
                    img = document.createElement('img');
                    img.id = 'current-featured-image';
                    img.style.cssText = 'width: 100%; height: auto; border-radius: 8px; border: 1px solid #e5e7eb;';
                    container.appendChild(img);
                }
                
                img.src = e.target.result;
                img.style.display = 'block';
                
                // Hide error message if exists
                var noImageMsg = document.getElementById('no-image-msg');
                if (noImageMsg) {
                    noImageMsg.style.display = 'none';
                }
            }
            reader.readAsDataURL(input.files[0]);
        }
    }
    </script>
    <!-- CRITICAL: Add main.js for mobile menu functionality -->
    <script src="/assets/js/main.js"></script>
</body>
</html>