<?php
/** cms-core/public/webview/admin/blog_manage.php */

// NO OUTPUT BEFORE THIS LINE!
require_once __DIR__ . '/../../../backend/authorization/check_admin.php';

// Pagination & Search Settings
$perPage = 20;
$page = max(1, (int)get_param('page', 1));
$search = trim(get_param('search', ''));
$statusFilter = get_param('status', 'all'); // all, published, draft

// Handle Delete BEFORE any HTML
if (is_post() && isset($_POST['action']) && $_POST['action'] === 'delete') {
    try {
        CSRF::require();
    } catch (Exception $e) {
        set_flash('error', 'Ungültiger CSRF-Token.');
        error_log("CSRF validation failed: " . $e->getMessage());
        redirect('/admin/blog_manage.php' . buildQueryString());
        exit;
    }
    
    $postId = (int)post('post_id');
    
    try {
        // Hole Post-Daten
        $post = db()->fetchOne("SELECT featured_image FROM blog_posts WHERE id = ?", [$postId]);
        
        if ($post && !empty($post['featured_image'])) {
            $imagePath = APP_ROOT . '/media/blog/' . $post['featured_image'];
            if (file_exists($imagePath)) {
                unlink($imagePath);
            }
        }
        
        // Lösche Post
        db()->execute("DELETE FROM blog_posts WHERE id = ?", [$postId]);
        
        set_flash('success', '✅ Blog-Post gelöscht.');
        Logger::info("Blog post deleted: ID {$postId} by " . current_user()['username']);
    } catch (Exception $e) {
        error_log("Error deleting blog post: " . $e->getMessage());
        set_flash('error', 'Fehler beim Löschen: ' . $e->getMessage());
    }
    
    redirect('/admin/blog_manage.php' . buildQueryString());
    exit;
}

// Build query for pagination with search/filter
function buildSearchQuery() {
    global $search, $statusFilter;
    
    $where = "1=1";
    $params = [];
    
    if ($search) {
        $where .= " AND (bp.title LIKE ? OR bp.slug LIKE ? OR bp.excerpt LIKE ? OR bp.content LIKE ?)";
        $searchTerm = "%{$search}%";
        $params = [$searchTerm, $searchTerm, $searchTerm, $searchTerm];
    }
    
    if ($statusFilter === 'published') {
        $where .= " AND bp.status = 'published'";
    } elseif ($statusFilter === 'draft') {
        $where .= " AND bp.status = 'draft'";
    }
    
    return ['where' => $where, 'params' => $params];
}

function buildQueryString() {
    global $search, $statusFilter, $page;
    $params = [];
    if ($search) $params[] = 'search=' . urlencode($search);
    if ($statusFilter && $statusFilter !== 'all') $params[] = 'status=' . urlencode($statusFilter);
    if ($page > 1) $params[] = 'page=' . $page;
    return $params ? '?' . implode('&', $params) : '';
}

try {
    $query = buildSearchQuery();
    
    // Get total count for pagination
    $countSql = "SELECT COUNT(*) as count FROM blog_posts bp WHERE {$query['where']}";
    $totalCount = db()->fetchOne($countSql, $query['params'])['count'];
    $totalPages = max(1, ceil($totalCount / $perPage));
    $offset = ($page - 1) * $perPage;
    
    // Fetch paginated posts
    $sql = "SELECT bp.*, u.username as author_name 
            FROM blog_posts bp 
            LEFT JOIN users u ON bp.author_id = u.id 
            WHERE {$query['where']}
            ORDER BY bp.created_at DESC
            LIMIT {$perPage} OFFSET {$offset}";
    
    $posts = db()->fetchAll($sql, $query['params']);
    
    // Get all posts for statistics (not filtered)
    $allPostsCount = db()->fetchOne("SELECT COUNT(*) as count FROM blog_posts")['count'];
    $publishedCount = db()->fetchOne("SELECT COUNT(*) as count FROM blog_posts WHERE status = 'published'")['count'];
    $draftCount = db()->fetchOne("SELECT COUNT(*) as count FROM blog_posts WHERE status = 'draft'")['count'];
    
} catch (Exception $e) {
    error_log("Error fetching blog posts: " . $e->getMessage());
    set_flash('error', 'Fehler beim Laden der Blog-Posts: ' . $e->getMessage());
    $posts = [];
    $totalCount = 0;
    $totalPages = 1;
    $allPostsCount = 0;
    $publishedCount = 0;
    $draftCount = 0;
}

$flashMessages = get_flash();
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Blog-Posts verwalten</title>
<?php
$adminCssPath = APP_ROOT . '/assets/css/admin.css';
$adminCssVersion = file_exists($adminCssPath) ? filemtime($adminCssPath) : time();
?>
<link rel="stylesheet" href="/assets/css/admin.css?v=<?= $adminCssVersion ?>">
<style>
.search-filter-bar {
    display: flex;
    gap: 15px;
    margin-bottom: 25px;
    flex-wrap: wrap;
    align-items: center;
}

.search-box {
    position: relative;
    flex: 1;
    min-width: 300px;
}

.search-box input {
    width: 100%;
    padding: 12px 45px 12px 15px;
    border: 2px solid #e5e7eb;
    border-radius: 8px;
    font-size: 14px;
    transition: all 0.2s;
}

.search-box input:focus {
    border-color: #667eea;
    outline: none;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
}

.search-box button {
    position: absolute;
    right: 5px;
    top: 50%;
    transform: translateY(-50%);
    background: none;
    border: none;
    cursor: pointer;
    padding: 8px 12px;
    font-size: 18px;
    color: #667eea;
    transition: color 0.2s;
}

.search-box button:hover {
    color: #5568d3;
}

.filter-select {
    padding: 12px 15px;
    border: 2px solid #e5e7eb;
    border-radius: 8px;
    font-size: 14px;
    background: white;
    cursor: pointer;
    transition: all 0.2s;
}

.filter-select:hover {
    border-color: #667eea;
}

.pagination {
    display: flex;
    justify-content: center;
    align-items: center;
    gap: 8px;
    margin-top: 30px;
    padding: 20px 0;
}

.pagination a, .pagination span {
    padding: 10px 15px;
    background: white;
    border: 2px solid #e5e7eb;
    border-radius: 8px;
    text-decoration: none;
    color: #374151;
    transition: all 0.2s;
    min-width: 45px;
    text-align: center;
}

.pagination a:hover {
    border-color: #667eea;
    color: #667eea;
    transform: translateY(-1px);
}

.pagination .active {
    background: #667eea;
    color: white;
    border-color: #667eea;
    font-weight: 600;
}

.pagination .disabled {
    opacity: 0.5;
    cursor: not-allowed;
    pointer-events: none;
}

.results-info {
    text-align: center;
    color: #6b7280;
    font-size: 14px;
    margin-bottom: 20px;
}

.no-results {
    text-align: center;
    padding: 60px 20px;
}

.no-results-icon {
    font-size: 64px;
    margin-bottom: 20px;
}

.no-results h3 {
    color: #6b7280;
    margin-bottom: 10px;
}

.no-results p {
    color: #9ca3af;
    margin-bottom: 30px;
}

.post-thumbnail {
    width: 60px;
    height: 60px;
    object-fit: cover;
    border-radius: 8px;
    border: 2px solid #e5e7eb;
}

.post-title-cell {
    display: flex;
    align-items: center;
    gap: 15px;
}

.post-title-info {
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.post-slug {
    font-size: 12px;
    color: #9ca3af;
}
</style>
</head>
<body>
    <?php include 'includes/admin_header.php'; ?>
    <?php 
    if (file_exists(__DIR__ . '/includes/maintenance_banner.php')) {
        include 'includes/maintenance_banner.php';
    }
    ?>
    
    <div class="admin-container">
        <?php include 'includes/admin_sidebar.php'; ?>
        
        <main class="admin-content">
            <?php foreach ($flashMessages as $flash): ?>
                <div class="alert alert-<?= e($flash['type']) ?>">
                    <?= e($flash['message']) ?>
                </div>
            <?php endforeach; ?>
            
            <div class="page-header">
                <h1>📝 Blog-Posts verwalten</h1>
                <p>Alle Blog-Posts bearbeiten und verwalten</p>
            </div>
            
            <div class="content-card">
                <div class="card-header">
                    <h2 class="card-title">Alle Blog-Posts (<?= $totalCount ?> Ergebnis<?= $totalCount !== 1 ? 'se' : '' ?>)</h2>
                    <a href="blog_create_edit.php" class="btn btn-primary">➕ Neuer Post</a>
                </div>
                
                <!-- Search & Filter Bar -->
                <div class="search-filter-bar">
                    <form method="get" class="search-box">
                        <input type="hidden" name="status" value="<?= e($statusFilter) ?>">
                        <input type="text" 
                               name="search" 
                               placeholder="🔍 Blog-Posts durchsuchen (Titel, Slug, Inhalt)..." 
                               value="<?= e($search) ?>"
                               autocomplete="off">
                        <button type="submit">🔍</button>
                    </form>
                    
                    <select class="filter-select" 
                            onchange="window.location.href='?status=' + this.value + '<?= $search ? '&search=' . urlencode($search) : '' ?>'">
                        <option value="all" <?= $statusFilter === 'all' ? 'selected' : '' ?>>Alle Status</option>
                        <option value="published" <?= $statusFilter === 'published' ? 'selected' : '' ?>>✓ Veröffentlicht</option>
                        <option value="draft" <?= $statusFilter === 'draft' ? 'selected' : '' ?>>○ Entwürfe</option>
                    </select>
                    
                    <?php if ($search || $statusFilter !== 'all'): ?>
                        <a href="blog_manage.php" class="btn btn-secondary">Filter zurücksetzen</a>
                    <?php endif; ?>
                </div>
                
                <!-- Results Info -->
                <?php if ($search || $statusFilter !== 'all'): ?>
                    <div class="results-info">
                        Zeige <?= $totalCount ?> von <?= $allPostsCount ?> Blog-Posts
                        <?php if ($search): ?>
                            für "<strong><?= e($search) ?></strong>"
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
                
                <?php if (empty($posts)): ?>
                    <div class="no-results">
                        <div class="no-results-icon">
                            <?= $search ? '🔍' : '📝' ?>
                        </div>
                        <h3><?= $search ? 'Keine Blog-Posts gefunden' : 'Noch keine Blog-Posts' ?></h3>
                        <p>
                            <?= $search 
                                ? 'Versuchen Sie einen anderen Suchbegriff oder passen Sie die Filter an.' 
                                : 'Erstellen Sie Ihren ersten Blog-Post!' 
                            ?>
                        </p>
                        <?php if (!$search): ?>
                            <a href="blog_create_edit.php" class="btn btn-primary">Ersten Post erstellen</a>
                        <?php endif; ?>
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th style="width: 40%;">Titel</th>
                                    <th style="width: 15%;">Autor</th>
                                    <th style="width: 12%;">Status</th>
                                    <th style="width: 12%;">Erstellt</th>
                                    <th style="width: 12%;">Veröffentlicht</th>
                                    <th style="width: 9%;">Aktionen</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($posts as $post): ?>
                                    <tr>
                                        <td>
                                            <div class="post-title-cell">
                                                <?php if (!empty($post['featured_image'])): ?>
                                                    <img src="/media/blog/<?= e($post['featured_image']) ?>" 
                                                         alt="Thumbnail" 
                                                         class="post-thumbnail">
                                                <?php else: ?>
                                                    <div class="post-thumbnail" style="display: flex; align-items: center; justify-content: center; background: #f3f4f6; color: #9ca3af; font-size: 24px;">
                                                        📝
                                                    </div>
                                                <?php endif; ?>
                                                <div class="post-title-info">
                                                    <strong><?= e($post['title']) ?></strong>
                                                    <span class="post-slug">/blog/<?= e($post['slug']) ?></span>
                                                </div>
                                            </div>
                                        </td>
                                        <td><?= e($post['author_name'] ?? 'Unbekannt') ?></td>
                                        <td>
                                            <?php if ($post['status'] === 'published'): ?>
                                                <span class="badge badge-success">✓ Veröffentlicht</span>
                                            <?php elseif ($post['status'] === 'draft'): ?>
                                                <span class="badge badge-warning">○ Entwurf</span>
                                            <?php else: ?>
                                                <span class="badge badge-secondary"><?= e($post['status']) ?></span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <span style="font-size: 13px;">
                                                <?= date('d.m.Y', strtotime($post['created_at'])) ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if ($post['published_at']): ?>
                                                <span style="font-size: 13px;">
                                                    <?= date('d.m.Y', strtotime($post['published_at'])) ?>
                                                </span>
                                            <?php else: ?>
                                                <span style="color: #9ca3af;">—</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div class="actions">
                                                <a href="blog_edit.php?id=<?= $post['id'] ?>" 
                                                   class="btn btn-sm btn-primary" 
                                                   title="Bearbeiten">
                                                    ✏️
                                                </a>
                                                <a href="/blog/<?= e($post['slug']) ?>" 
                                                   class="btn btn-sm btn-secondary" 
                                                   title="Ansehen" 
                                                   target="_blank">
                                                    👁️
                                                </a>
                                                <form method="POST" 
                                                      style="display: inline;" 
                                                      onsubmit="return confirm('Blog-Post &quot;<?= e($post['title']) ?>&quot; wirklich löschen?');">
                                                    <?= CSRF::field() ?>
                                                    <input type="hidden" name="action" value="delete">
                                                    <input type="hidden" name="post_id" value="<?= $post['id'] ?>">
                                                    <button type="submit" 
                                                            class="btn btn-sm btn-danger" 
                                                            title="Löschen">
                                                        🗑️
                                                    </button>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Pagination -->
                    <?php if ($totalPages > 1): ?>
                        <div class="pagination">
                            <?php if ($page > 1): ?>
                                <a href="?page=1<?= $search ? '&search=' . urlencode($search) : '' ?><?= $statusFilter !== 'all' ? '&status=' . $statusFilter : '' ?>" title="Erste Seite">
                                    ⏮️
                                </a>
                                <a href="?page=<?= $page - 1 ?><?= $search ? '&search=' . urlencode($search) : '' ?><?= $statusFilter !== 'all' ? '&status=' . $statusFilter : '' ?>">
                                    ← Zurück
                                </a>
                            <?php else: ?>
                                <span class="disabled">⏮️</span>
                                <span class="disabled">← Zurück</span>
                            <?php endif; ?>
                            
                            <?php
                            $range = 2;
                            $start = max(1, $page - $range);
                            $end = min($totalPages, $page + $range);
                            
                            if ($start > 1): ?>
                                <a href="?page=1<?= $search ? '&search=' . urlencode($search) : '' ?><?= $statusFilter !== 'all' ? '&status=' . $statusFilter : '' ?>">1</a>
                                <?php if ($start > 2): ?>
                                    <span>...</span>
                                <?php endif; ?>
                            <?php endif; ?>
                            
                            <?php for ($i = $start; $i <= $end; $i++): ?>
                                <a href="?page=<?= $i ?><?= $search ? '&search=' . urlencode($search) : '' ?><?= $statusFilter !== 'all' ? '&status=' . $statusFilter : '' ?>" 
                                   class="<?= $i === $page ? 'active' : '' ?>">
                                    <?= $i ?>
                                </a>
                            <?php endfor; ?>
                            
                            <?php if ($end < $totalPages): ?>
                                <?php if ($end < $totalPages - 1): ?>
                                    <span>...</span>
                                <?php endif; ?>
                                <a href="?page=<?= $totalPages ?><?= $search ? '&search=' . urlencode($search) : '' ?><?= $statusFilter !== 'all' ? '&status=' . $statusFilter : '' ?>"><?= $totalPages ?></a>
                            <?php endif; ?>
                            
                            <?php if ($page < $totalPages): ?>
                                <a href="?page=<?= $page + 1 ?><?= $search ? '&search=' . urlencode($search) : '' ?><?= $statusFilter !== 'all' ? '&status=' . $statusFilter : '' ?>">
                                    Weiter →
                                </a>
                                <a href="?page=<?= $totalPages ?><?= $search ? '&search=' . urlencode($search) : '' ?><?= $statusFilter !== 'all' ? '&status=' . $statusFilter : '' ?>" title="Letzte Seite">
                                    ⏭️
                                </a>
                            <?php else: ?>
                                <span class="disabled">Weiter →</span>
                                <span class="disabled">⏭️</span>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </main>
    </div>
    
    <script src="/assets/js/main.js"></script>
</body>
</html>