<?php
/** cms-core/public/webview/admin/cdn_setup_wizard.php */

require_once __DIR__ . '/../../../backend/authorization/check_admin.php';

// Get current step from URL or default to 1
$currentStep = isset($_GET['step']) ? (int)$_GET['step'] : 1;

// Check if Cloudflare is already active
function is_cloudflare_active() {
    $domain = $_SERVER['HTTP_HOST'];
    $nameservers = @dns_get_record($domain, DNS_NS);
    
    if ($nameservers) {
        foreach ($nameservers as $ns) {
            if (isset($ns['target']) && strpos($ns['target'], 'cloudflare') !== false) {
                return true;
            }
        }
    }
    
    return false;
}

// Handle form submissions
if (is_post()) {
    CSRF::require();
    
    $action = post('action');
    
    // Save Cloudflare email (optional tracking)
    if ($action === 'save_email') {
        $email = trim(post('cloudflare_email'));
        if (!empty($email)) {
            set_setting('cloudflare_email', $email, 'text');
        }
        redirect('/admin/cdn_setup_wizard.php?step=2');
        exit;
    }
    
    // Skip to next step
    if ($action === 'next_step') {
        $nextStep = (int)post('next_step');
        redirect('/admin/cdn_setup_wizard.php?step=' . $nextStep);
        exit;
    }
    
    // Check status and complete
    if ($action === 'check_status') {
        if (is_cloudflare_active()) {
            // Activate CDN in CMS
            IntegrationHelper::enable('cloudflare_cdn');
            set_setting('cloudflare_cdn_media_only', true, 'boolean');
            
            set_flash('success', '🎉 Cloudflare CDN erfolgreich aktiviert!');
            redirect('/admin/integrations.php');
            exit;
        } else {
            set_flash('warning', 'Cloudflare noch nicht erkannt. DNS-Änderung kann bis zu 48h dauern.');
            redirect('/admin/cdn_setup_wizard.php?step=4');
            exit;
        }
    }
}

// Check if already active and skip to end
if (is_cloudflare_active() && $currentStep < 5) {
    $currentStep = 5;
}

$domain = $_SERVER['HTTP_HOST'];
$flashMessages = get_flash();
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cloudflare CDN Setup</title>
    <?php
    $adminCssPath = APP_ROOT . '/assets/css/admin.css';
    $adminCssVersion = file_exists($adminCssPath) ? filemtime($adminCssPath) : time();
    ?>
    <link rel="stylesheet" href="/assets/css/admin.css?v=<?= $adminCssVersion ?>">
    <style>
        .wizard-container {
            max-width: 900px;
            margin: 0 auto;
        }
        
        .wizard-progress {
            display: flex;
            justify-content: space-between;
            margin-bottom: 40px;
            position: relative;
        }
        
        .wizard-progress::before {
            content: '';
            position: absolute;
            top: 20px;
            left: 0;
            right: 0;
            height: 2px;
            background: #e5e7eb;
            z-index: 0;
        }
        
        .progress-line {
            position: absolute;
            top: 20px;
            left: 0;
            height: 2px;
            background: #10b981;
            transition: width 0.3s;
            z-index: 0;
        }
        
        .wizard-step {
            flex: 1;
            text-align: center;
            position: relative;
            z-index: 1;
        }
        
        .step-circle {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: white;
            border: 2px solid #e5e7eb;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 10px;
            font-weight: bold;
            transition: all 0.3s;
        }
        
        .wizard-step.completed .step-circle {
            background: #10b981;
            border-color: #10b981;
            color: white;
        }
        
        .wizard-step.active .step-circle {
            border-color: #667eea;
            color: #667eea;
            border-width: 3px;
        }
        
        .step-label {
            font-size: 13px;
            color: #6b7280;
        }
        
        .wizard-step.active .step-label {
            color: #667eea;
            font-weight: 600;
        }
        
        .wizard-content {
            background: white;
            padding: 40px;
            border-radius: 12px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        
        .wizard-content h2 {
            margin-top: 0;
            color: #1f2937;
        }
        
        .instruction-box {
            background: #f9fafb;
            border: 2px solid #e5e7eb;
            border-radius: 8px;
            padding: 20px;
            margin: 20px 0;
        }
        
        .instruction-box ol {
            margin: 10px 0;
            padding-left: 20px;
        }
        
        .instruction-box li {
            margin: 10px 0;
            line-height: 1.6;
        }
        
        .copy-box {
            display: flex;
            gap: 10px;
            margin: 15px 0;
        }
        
        .copy-box input {
            flex: 1;
            padding: 12px;
            border: 2px solid #667eea;
            border-radius: 6px;
            font-family: monospace;
            font-size: 16px;
            background: #f0f4ff;
        }
        
        .copy-box button {
            padding: 12px 20px;
            background: #667eea;
            color: white;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-weight: 600;
        }
        
        .copy-box button:hover {
            background: #5568d3;
        }
        
        .wizard-actions {
            display: flex;
            justify-content: space-between;
            margin-top: 30px;
            padding-top: 30px;
            border-top: 1px solid #e5e7eb;
        }
        
        .btn-wizard {
            padding: 12px 30px;
            border-radius: 8px;
            font-weight: 600;
            text-decoration: none;
            display: inline-block;
            transition: all 0.2s;
            border: none;
            cursor: pointer;
        }
        
        .btn-primary {
            background: #667eea;
            color: white;
        }
        
        .btn-primary:hover {
            background: #5568d3;
        }
        
        .btn-secondary {
            background: #f3f4f6;
            color: #6b7280;
        }
        
        .btn-secondary:hover {
            background: #e5e7eb;
        }
        
        .btn-success {
            background: #10b981;
            color: white;
        }
        
        .btn-success:hover {
            background: #059669;
        }
        
        .alert-info {
            background: #dbeafe;
            border: 2px solid #3b82f6;
            color: #1e40af;
            padding: 15px;
            border-radius: 8px;
            margin: 20px 0;
        }
        
        .alert-success {
            background: #d1fae5;
            border: 2px solid #10b981;
            color: #065f46;
            padding: 15px;
            border-radius: 8px;
            margin: 20px 0;
        }
        
        .external-link {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            padding: 12px 24px;
            background: #10b981;
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-weight: 600;
            transition: all 0.2s;
        }
        
        .external-link:hover {
            background: #059669;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(16, 185, 129, 0.3);
        }
        
        .status-check {
            text-align: center;
            padding: 40px;
        }
        
        .spinner {
            border: 4px solid #f3f4f6;
            border-top: 4px solid #667eea;
            border-radius: 50%;
            width: 50px;
            height: 50px;
            animation: spin 1s linear infinite;
            margin: 20px auto;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <?php include 'includes/admin_header.php'; ?>
    
    <div class="admin-container">
        <?php include 'includes/admin_sidebar.php'; ?>
        
        <main class="admin-content">
            <?php foreach ($flashMessages as $flash): ?>
                <div class="alert alert-<?= e($flash['type']) ?>">
                    <?= e($flash['message']) ?>
                </div>
            <?php endforeach; ?>
            
            <div class="wizard-container">
                <!-- Progress Bar -->
                <div class="wizard-progress">
                    <div class="progress-line" style="width: <?= ($currentStep - 1) * 25 ?>%;"></div>
                    
                    <div class="wizard-step <?= $currentStep >= 1 ? 'completed' : '' ?> <?= $currentStep == 1 ? 'active' : '' ?>">
                        <div class="step-circle"><?= $currentStep > 1 ? '✓' : '1' ?></div>
                        <div class="step-label">Account</div>
                    </div>
                    
                    <div class="wizard-step <?= $currentStep >= 2 ? 'completed' : '' ?> <?= $currentStep == 2 ? 'active' : '' ?>">
                        <div class="step-circle"><?= $currentStep > 2 ? '✓' : '2' ?></div>
                        <div class="step-label">Domain</div>
                    </div>
                    
                    <div class="wizard-step <?= $currentStep >= 3 ? 'completed' : '' ?> <?= $currentStep == 3 ? 'active' : '' ?>">
                        <div class="step-circle"><?= $currentStep > 3 ? '✓' : '3' ?></div>
                        <div class="step-label">Nameserver</div>
                    </div>
                    
                    <div class="wizard-step <?= $currentStep >= 4 ? 'completed' : '' ?> <?= $currentStep == 4 ? 'active' : '' ?>">
                        <div class="step-circle"><?= $currentStep > 4 ? '✓' : '4' ?></div>
                        <div class="step-label">Warten</div>
                    </div>
                    
                    <div class="wizard-step <?= $currentStep >= 5 ? 'completed' : '' ?> <?= $currentStep == 5 ? 'active' : '' ?>">
                        <div class="step-circle"><?= $currentStep >= 5 ? '✓' : '5' ?></div>
                        <div class="step-label">Fertig</div>
                    </div>
                </div>
                
                <!-- Step Content -->
                <div class="wizard-content">
                    <?php if ($currentStep == 1): ?>
                        <!-- Step 1: Create Cloudflare Account -->
                        <h2>🌐 Schritt 1: Cloudflare Account erstellen</h2>
                        
                        <p>Cloudflare bietet einen <strong>kostenlosen Plan</strong> mit unbegrenzter Bandbreite und automatischer CDN-Optimierung.</p>
                        
                        <div class="instruction-box">
                            <h3 style="margin-top: 0;">Was du tun musst:</h3>
                            <ol>
                                <li>Klicke auf den Button unten um Cloudflare zu öffnen</li>
                                <li>Registriere dich mit deiner E-Mail-Adresse</li>
                                <li>Bestätige deine E-Mail</li>
                                <li>Wähle den <strong>Free Plan</strong></li>
                            </ol>
                        </div>
                        
                        <div style="text-align: center; margin: 30px 0;">
                            <a href="https://dash.cloudflare.com/sign-up" target="_blank" class="external-link">
                                🚀 Zu Cloudflare (neues Fenster)
                            </a>
                        </div>
                        
                        <div class="alert-info">
                            💡 <strong>Tipp:</strong> Behalte dieses Fenster offen während du dich bei Cloudflare registrierst.
                        </div>
                        
                        <div class="wizard-actions">
                            <a href="/admin/integrations.php" class="btn-wizard btn-secondary">← Abbrechen</a>
                            
                            <form method="POST" style="display: inline;">
                                <?= CSRF::field() ?>
                                <input type="hidden" name="action" value="next_step">
                                <input type="hidden" name="next_step" value="2">
                                <button type="submit" class="btn-wizard btn-primary">Weiter →</button>
                            </form>
                        </div>
                        
                    <?php elseif ($currentStep == 2): ?>
                        <!-- Step 2: Add Domain to Cloudflare -->
                        <h2>🌍 Schritt 2: Domain zu Cloudflare hinzufügen</h2>
                        
                        <p>Jetzt fügst du deine Domain zu Cloudflare hinzu.</p>
                        
                        <div class="instruction-box">
                            <h3 style="margin-top: 0;">So geht's:</h3>
                            <ol>
                                <li>Öffne das <a href="https://dash.cloudflare.com" target="_blank">Cloudflare Dashboard</a></li>
                                <li>Klicke auf <strong>"Add a Site"</strong> oder <strong>"Website hinzufügen"</strong></li>
                                <li>Gib diese Domain ein:</li>
                            </ol>
                        </div>
                        
                        <div class="copy-box">
                            <input type="text" value="<?= e($domain) ?>" readonly id="domain-input">
                            <button onclick="copyToClipboard('domain-input')">📋 Kopieren</button>
                        </div>
                        
                        <div class="instruction-box">
                            <ol start="4">
                                <li>Wähle den <strong>Free Plan</strong></li>
                                <li>Cloudflare scannt automatisch deine DNS-Einträge (dauert ~1 Minute)</li>
                                <li>Klicke auf <strong>"Continue"</strong></li>
                            </ol>
                        </div>
                        
                        <div class="alert-info">
                            ⏱️ <strong>Hinweis:</strong> Der DNS-Scan dauert ungefähr 1 Minute. Warte bis er fertig ist.
                        </div>
                        
                        <div class="wizard-actions">
                            <a href="/admin/cdn_setup_wizard.php?step=1" class="btn-wizard btn-secondary">← Zurück</a>
                            
                            <form method="POST" style="display: inline;">
                                <?= CSRF::field() ?>
                                <input type="hidden" name="action" value="next_step">
                                <input type="hidden" name="next_step" value="3">
                                <button type="submit" class="btn-wizard btn-primary">Weiter →</button>
                            </form>
                        </div>
                        
                    <?php elseif ($currentStep == 3): ?>
                        <!-- Step 3: Change Nameservers -->
                        <h2>🔧 Schritt 3: Nameserver bei IONOS ändern</h2>
                        
                        <p>Cloudflare zeigt dir jetzt <strong>2 Nameserver</strong>. Diese musst du bei IONOS eintragen.</p>
                        
                        <div class="instruction-box">
                            <h3 style="margin-top: 0;">1. Nameserver von Cloudflare kopieren</h3>
                            <p>Cloudflare zeigt dir etwas wie:</p>
                            <ul>
                                <li><code>alice.ns.cloudflare.com</code></li>
                                <li><code>bob.ns.cloudflare.com</code></li>
                            </ul>
                            <p><strong>Kopiere diese beiden Nameserver!</strong></p>
                        </div>
                        
                        <div class="instruction-box">
                            <h3 style="margin-top: 0;">2. Bei IONOS einloggen und ändern</h3>
                            <ol>
                                <li>Öffne <a href="https://www.ionos.de/login" target="_blank" class="external-link">IONOS Login</a></li>
                                <li>Gehe zu <strong>Domains & SSL</strong></li>
                                <li>Wähle deine Domain: <strong><?= e($domain) ?></strong></li>
                                <li>Klicke auf <strong>DNS-Einstellungen</strong></li>
                                <li>Suche nach <strong>"Nameserver"</strong></li>
                                <li>Ändere von IONOS Nameservern zu den Cloudflare Nameservern</li>
                                <li>Speichern!</li>
                            </ol>
                        </div>
                        
                        <div class="alert-info">
                            🔐 <strong>Wichtig:</strong> Diese Änderung machst du direkt bei IONOS. Dein CMS kann das nicht automatisch für dich machen (Sicherheitsgründe).
                        </div>
                        
                        <div class="wizard-actions">
                            <a href="/admin/cdn_setup_wizard.php?step=2" class="btn-wizard btn-secondary">← Zurück</a>
                            
                            <form method="POST" style="display: inline;">
                                <?= CSRF::field() ?>
                                <input type="hidden" name="action" value="next_step">
                                <input type="hidden" name="next_step" value="4">
                                <button type="submit" class="btn-wizard btn-primary">Weiter →</button>
                            </form>
                        </div>
                        
                    <?php elseif ($currentStep == 4): ?>
                        <!-- Step 4: Wait for DNS Propagation -->
                        <h2>⏳ Schritt 4: Warten auf DNS-Propagierung</h2>
                        
                        <p>Die Nameserver-Änderung ist jetzt aktiv, aber es dauert etwas bis sie weltweit verbreitet ist.</p>
                        
                        <div class="status-check">
                            <div class="spinner"></div>
                            <h3>Prüfe DNS-Status...</h3>
                            <p>Dies kann zwischen 2 und 48 Stunden dauern.</p>
                        </div>
                        
                        <div class="instruction-box">
                            <h3 style="margin-top: 0;">⏱️ Typische Wartezeiten:</h3>
                            <ul>
                                <li><strong>Schnell:</strong> 5-30 Minuten (selten)</li>
                                <li><strong>Normal:</strong> 2-6 Stunden (häufig)</li>
                                <li><strong>Langsam:</strong> 24-48 Stunden (bei manchen Providern)</li>
                            </ul>
                        </div>
                        
                        <div class="alert-info">
                            💡 <strong>Tipp:</strong> Du kannst diesen Tab schließen und später wiederkommen. Cloudflare sendet dir auch eine E-Mail wenn alles fertig ist!
                        </div>
                        
                        <div class="wizard-actions">
                            <a href="/admin/cdn_setup_wizard.php?step=3" class="btn-wizard btn-secondary">← Zurück</a>
                            
                            <form method="POST" style="display: inline;">
                                <?= CSRF::field() ?>
                                <input type="hidden" name="action" value="check_status">
                                <button type="submit" class="btn-wizard btn-success">🔍 Status jetzt prüfen</button>
                            </form>
                        </div>
                        
                    <?php elseif ($currentStep == 5): ?>
                        <!-- Step 5: Success! -->
                        <div class="alert-success">
                            <h2 style="margin-top: 0;">🎉 Cloudflare CDN ist aktiv!</h2>
                            <p>Deine Website läuft jetzt über das Cloudflare CDN.</p>
                        </div>
                        
                        <div class="instruction-box">
                            <h3 style="margin-top: 0;">✅ Was jetzt aktiv ist:</h3>
                            <ul>
                                <li>🌍 Globales CDN (200+ Standorte weltweit)</li>
                                <li>🚀 Automatisches Caching statischer Dateien</li>
                                <li>🔒 Kostenlo SSL/TLS Verschlüsselung</li>
                                <li>🛡️ DDoS-Schutz</li>
                                <li>⚡ Bildoptimierung</li>
                            </ul>
                        </div>
                        
                        <div class="instruction-box">
                            <h3 style="margin-top: 0;">🎯 Empfohlene nächste Schritte:</h3>
                            <ol>
                                <li>Gehe zu deinem Cloudflare Dashboard</li>
                                <li>Aktiviere <strong>"Auto Minify"</strong> für CSS, JS, HTML</li>
                                <li>Aktiviere <strong>"Brotli Compression"</strong></li>
                                <li>SSL/TLS Modus auf <strong>"Full (strict)"</strong> setzen</li>
                            </ol>
                        </div>
                        
                        <div style="text-align: center; margin: 30px 0;">
                            <a href="https://dash.cloudflare.com" target="_blank" class="external-link" style="margin-right: 10px;">
                                🌐 Zum Cloudflare Dashboard
                            </a>
                            <a href="/admin/integrations.php" class="btn-wizard btn-success">
                                ✓ Zurück zu Integrationen
                            </a>
                        </div>
                        
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>
    
    <script>
    function copyToClipboard(inputId) {
        const input = document.getElementById(inputId);
        input.select();
        input.setSelectionRange(0, 99999); // For mobile
        document.execCommand('copy');
        
        // Visual feedback
        const button = event.target;
        const originalText = button.textContent;
        button.textContent = '✓ Kopiert!';
        button.style.background = '#10b981';
        
        setTimeout(() => {
            button.textContent = originalText;
            button.style.background = '';
        }, 2000);
    }
    </script>
    
    <script src="/assets/js/main.js"></script>
</body>
</html>