<?php
/**
 * cms-core/public/webview/admin/cms_core_updater.php
 * Hidden CMS-Core Update System
 * 
 * Access: /admin/cms_core_updater.php
 * Purpose: Upload zip file with cms-core updates, auto-extracts and installs
 * 
 * SECURITY:
 * - Only accessible by logged-in admins
 * - CSRF protection
 * - File validation (only .zip)
 * - Backup creation before update
 */

require_once __DIR__ . '/../../../backend/authorization/check_admin.php';

$flashMessages = get_flash();
$updateLog = [];

// Handle Upload & Install
if (is_post() && isset($_FILES['cms_core_zip'])) {
    try {
        CSRF::require();
        
        $file = $_FILES['cms_core_zip'];
        
        // Validate file
        if ($file['error'] !== UPLOAD_ERR_OK) {
            throw new Exception('Upload Fehler: ' . $file['error']);
        }
        
        if ($file['size'] > 50 * 1024 * 1024) { // Max 50MB
            throw new Exception('Datei zu groß. Maximum: 50MB');
        }
        
        $fileExt = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if ($fileExt !== 'zip') {
            throw new Exception('Nur .zip Dateien erlaubt');
        }
        
        // Create backup of current cms-core
        $backupDir = APP_ROOT . '/backups';
        if (!file_exists($backupDir)) {
            mkdir($backupDir, 0755, true);
        }
        
        $backupName = 'cms-core-backup-' . date('Y-m-d_His') . '.zip';
        $backupPath = $backupDir . '/' . $backupName;
        
        $updateLog[] = '📦 Erstelle Backup...';
        
        // Backup current cms-core folder
        $zip = new ZipArchive();
        if ($zip->open($backupPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) === true) {
            $cmsCorePath = APP_ROOT . '/cms-core';
            $files = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($cmsCorePath),
                RecursiveIteratorIterator::LEAVES_ONLY
            );
            
            foreach ($files as $name => $file) {
                if (!$file->isDir()) {
                    $filePath = $file->getRealPath();
                    $relativePath = substr($filePath, strlen(APP_ROOT) + 1);
                    $zip->addFile($filePath, $relativePath);
                }
            }
            
            $zip->close();
            $updateLog[] = '✅ Backup erstellt: ' . $backupName;
        } else {
            throw new Exception('Backup konnte nicht erstellt werden');
        }
        
        // Extract uploaded zip to temp directory
        $tempDir = APP_ROOT . '/temp_cms_update_' . time();
        mkdir($tempDir, 0755, true);
        
        $updateLog[] = '📂 Entpacke Update...';
        
        $uploadedZip = new ZipArchive();
        if ($uploadedZip->open($file['tmp_name']) === true) {
            $uploadedZip->extractTo($tempDir);
            $uploadedZip->close();
            $updateLog[] = '✅ Update entpackt';
        } else {
            throw new Exception('Zip-Datei konnte nicht geöffnet werden');
        }
        
        // Find cms-core folder in extracted files
        $cmsCoreTempPath = null;
        
        // Check if cms-core is directly in temp folder
        if (file_exists($tempDir . '/cms-core')) {
            $cmsCoreTempPath = $tempDir . '/cms-core';
        } 
        // Check if zip contains folder with cms-core inside
        else {
            $dirs = glob($tempDir . '/*', GLOB_ONLYDIR);
            foreach ($dirs as $dir) {
                if (file_exists($dir . '/cms-core')) {
                    $cmsCoreTempPath = $dir . '/cms-core';
                    break;
                }
            }
        }
        
        if (!$cmsCoreTempPath) {
            throw new Exception('Kein cms-core Ordner in der Zip-Datei gefunden');
        }
        
        $updateLog[] = '🔄 Installiere Update...';
        
        // Copy new cms-core files over existing
        $targetPath = APP_ROOT . '/cms-core';
        
        function recursiveCopy($src, $dst) {
            $dir = opendir($src);
            @mkdir($dst, 0755, true);
            
            while (false !== ($file = readdir($dir))) {
                if (($file != '.') && ($file != '..')) {
                    if (is_dir($src . '/' . $file)) {
                        recursiveCopy($src . '/' . $file, $dst . '/' . $file);
                    } else {
                        copy($src . '/' . $file, $dst . '/' . $file);
                    }
                }
            }
            
            closedir($dir);
        }
        
        recursiveCopy($cmsCoreTempPath, $targetPath);
        
        $updateLog[] = '✅ Dateien kopiert';
        
        // Clean up temp directory
        function deleteDirectory($dir) {
            if (!file_exists($dir)) {
                return true;
            }
            
            if (!is_dir($dir)) {
                return unlink($dir);
            }
            
            foreach (scandir($dir) as $item) {
                if ($item == '.' || $item == '..') {
                    continue;
                }
                
                if (!deleteDirectory($dir . DIRECTORY_SEPARATOR . $item)) {
                    return false;
                }
            }
            
            return rmdir($dir);
        }
        
        deleteDirectory($tempDir);
        $updateLog[] = '🧹 Temp-Dateien gelöscht';
        
        $updateLog[] = '✅ <strong>Update erfolgreich installiert!</strong>';
        $updateLog[] = '📁 Backup gespeichert in: /backups/' . $backupName;
        
        set_flash('success', 'CMS-Core erfolgreich aktualisiert!');
        
    } catch (Exception $e) {
        $updateLog[] = '❌ FEHLER: ' . $e->getMessage();
        set_flash('error', 'Update fehlgeschlagen: ' . $e->getMessage());
    }
}

// Get list of backups
$backupDir = APP_ROOT . '/backups';
$backups = [];
if (file_exists($backupDir)) {
    $backupFiles = glob($backupDir . '/cms-core-backup-*.zip');
    foreach ($backupFiles as $backup) {
        $backups[] = [
            'name' => basename($backup),
            'size' => filesize($backup),
            'date' => filemtime($backup),
            'path' => $backup
        ];
    }
    // Sort by date (newest first)
    usort($backups, function($a, $b) {
        return $b['date'] - $a['date'];
    });
}

?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>CMS-Core Updater - Admin</title>
    <link rel="stylesheet" href="../assets/css/admin.css">
    <style>
        .updater-container {
            max-width: 900px;
            margin: 0 auto;
        }
        
        .upload-zone {
            border: 2px dashed #d1d5db;
            border-radius: 12px;
            padding: 40px;
            text-align: center;
            background: #f9fafb;
            transition: all 0.3s ease;
        }
        
        .upload-zone:hover {
            border-color: #667eea;
            background: #eff6ff;
        }
        
        .upload-zone.dragging {
            border-color: #667eea;
            background: #dbeafe;
        }
        
        .upload-icon {
            font-size: 64px;
            margin-bottom: 16px;
        }
        
        .file-input-wrapper {
            position: relative;
            overflow: hidden;
            display: inline-block;
            margin-top: 20px;
        }
        
        .file-input-wrapper input[type=file] {
            position: absolute;
            left: -9999px;
        }
        
        .file-input-label {
            padding: 12px 32px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            transition: transform 0.2s;
        }
        
        .file-input-label:hover {
            transform: translateY(-2px);
        }
        
        .update-log {
            background: #1f2937;
            color: #f9fafb;
            padding: 20px;
            border-radius: 8px;
            font-family: 'Courier New', monospace;
            font-size: 14px;
            line-height: 1.8;
            max-height: 400px;
            overflow-y: auto;
            margin-top: 20px;
        }
        
        .update-log div {
            margin-bottom: 4px;
        }
        
        .backup-list {
            background: white;
            border: 1px solid #e5e7eb;
            border-radius: 8px;
            overflow: hidden;
        }
        
        .backup-item {
            padding: 16px;
            border-bottom: 1px solid #e5e7eb;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .backup-item:last-child {
            border-bottom: none;
        }
        
        .backup-info {
            flex: 1;
        }
        
        .backup-name {
            font-weight: 600;
            color: #1f2937;
            margin-bottom: 4px;
        }
        
        .backup-meta {
            font-size: 13px;
            color: #6b7280;
        }
        
        .warning-box {
            background: #fef3c7;
            border: 2px solid #fbbf24;
            border-radius: 8px;
            padding: 16px;
            margin: 20px 0;
        }
        
        .warning-box h4 {
            color: #92400e;
            margin: 0 0 8px 0;
            font-size: 16px;
        }
        
        .warning-box p {
            color: #78350f;
            margin: 0;
            font-size: 14px;
            line-height: 1.6;
        }
    </style>
</head>
<body>
    <?php include 'includes/admin_header.php'; ?>
    
    <div class="admin-container">
        <?php include 'includes/admin_sidebar.php'; ?>
        
        <main class="admin-content">
            <div class="updater-container">
                <?php foreach ($flashMessages as $flash): ?>
                    <div class="alert alert-<?= e($flash['type']) ?>">
                        <?= e($flash['message']) ?>
                    </div>
                <?php endforeach; ?>
                
                <div class="page-header">
                    <h1>🔧 CMS-Core Updater</h1>
                    <p>Lade neue cms-core Dateien hoch - Automatische Installation mit Backup</p>
                </div>
                
                <!-- Warning -->
                <div class="warning-box">
                    <h4>⚠️ Wichtig vor dem Update:</h4>
                    <p>
                        • Dieser Updater erstellt automatisch ein Backup<br>
                        • Upload nur vertrauenswürdige .zip Dateien<br>
                        • Maximale Dateigröße: 50MB<br>
                        • Die Zip muss einen <code>cms-core</code> Ordner enthalten
                    </p>
                </div>
                
                <!-- Upload Form -->
                <div class="content-card">
                    <div class="card-header">
                        <h2 class="card-title">📤 CMS-Core Update hochladen</h2>
                    </div>
                    
                    <form method="POST" enctype="multipart/form-data" id="uploadForm">
                        <?= CSRF::field() ?>
                        
                        <div class="upload-zone" id="uploadZone">
                            <div class="upload-icon">📦</div>
                            <h3 style="margin: 0 0 8px 0; color: #1f2937;">Ziehe deine .zip Datei hierher</h3>
                            <p style="color: #6b7280; margin: 0 0 20px 0;">oder klicke um eine Datei auszuwählen</p>
                            
                            <div class="file-input-wrapper">
                                <input 
                                    type="file" 
                                    name="cms_core_zip" 
                                    id="cmsZipFile" 
                                    accept=".zip"
                                    required
                                    onchange="handleFileSelect(this)"
                                >
                                <label for="cmsZipFile" class="file-input-label">
                                    📁 Datei auswählen
                                </label>
                            </div>
                            
                            <div id="selectedFile" style="margin-top: 20px; display: none;">
                                <strong>Ausgewählt:</strong> <span id="fileName"></span>
                            </div>
                        </div>
                        
                        <div style="margin-top: 24px; text-align: center;">
                            <button type="submit" class="btn btn-primary btn-large" id="uploadBtn" style="display: none;">
                                🚀 Update installieren
                            </button>
                        </div>
                    </form>
                </div>
                
                <!-- Update Log -->
                <?php if (!empty($updateLog)): ?>
                <div class="content-card" style="margin-top: 24px;">
                    <div class="card-header">
                        <h2 class="card-title">📋 Update-Log</h2>
                    </div>
                    
                    <div class="update-log">
                        <?php foreach ($updateLog as $logEntry): ?>
                            <div><?= $logEntry ?></div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Backups -->
                <?php if (!empty($backups)): ?>
                <div class="content-card" style="margin-top: 24px;">
                    <div class="card-header">
                        <h2 class="card-title">💾 Backups (<?= count($backups) ?>)</h2>
                        <p style="color: #6b7280; font-size: 14px; margin-top: 8px;">
                            Automatisch erstellte Backups vor jedem Update
                        </p>
                    </div>
                    
                    <div class="backup-list">
                        <?php foreach ($backups as $backup): ?>
                        <div class="backup-item">
                            <div class="backup-info">
                                <div class="backup-name">📦 <?= e($backup['name']) ?></div>
                                <div class="backup-meta">
                                    📅 <?= date('d.m.Y H:i', $backup['date']) ?> Uhr • 
                                    💾 <?= round($backup['size'] / 1024 / 1024, 2) ?> MB
                                </div>
                            </div>
                            <a href="<?= e(str_replace(APP_ROOT, '', $backup['path'])) ?>" 
                               class="btn btn-sm btn-secondary" 
                               download>
                                ⬇️ Download
                            </a>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>
                
            </div>
        </main>
    </div>
    
    <script src="../assets/js/main.js"></script>
    <script>
        // File select handler
        function handleFileSelect(input) {
            const file = input.files[0];
            if (file) {
                document.getElementById('fileName').textContent = file.name;
                document.getElementById('selectedFile').style.display = 'block';
                document.getElementById('uploadBtn').style.display = 'inline-block';
            }
        }
        
        // Drag & Drop
        const uploadZone = document.getElementById('uploadZone');
        const fileInput = document.getElementById('cmsZipFile');
        
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
            uploadZone.addEventListener(eventName, preventDefaults, false);
        });
        
        function preventDefaults(e) {
            e.preventDefault();
            e.stopPropagation();
        }
        
        ['dragenter', 'dragover'].forEach(eventName => {
            uploadZone.addEventListener(eventName, () => {
                uploadZone.classList.add('dragging');
            }, false);
        });
        
        ['dragleave', 'drop'].forEach(eventName => {
            uploadZone.addEventListener(eventName, () => {
                uploadZone.classList.remove('dragging');
            }, false);
        });
        
        uploadZone.addEventListener('drop', (e) => {
            const dt = e.dataTransfer;
            const files = dt.files;
            
            if (files.length > 0) {
                fileInput.files = files;
                handleFileSelect(fileInput);
            }
        }, false);
        
        // Click to upload
        uploadZone.addEventListener('click', (e) => {
            if (e.target.tagName !== 'INPUT' && e.target.tagName !== 'LABEL') {
                fileInput.click();
            }
        });
        
        // Form submission confirmation
        document.getElementById('uploadForm').addEventListener('submit', (e) => {
            if (!confirm('CMS-Core Update jetzt installieren?\n\nEin Backup wird automatisch erstellt.')) {
                e.preventDefault();
            }
        });
    </script>
</body>
</html>