<?php
/** cms-core/public/webview/admin/dashboard.php - FIXED */

// START OUTPUT BUFFERING
ob_start();

require_once __DIR__ . '/../../../backend/authorization/check_admin.php';
require_once __DIR__ . '/../../../backend/analytics/AnalyticsTracker.php';

// Load Update Checker
require_once __DIR__ . '/../../../backend/updates/UpdateChecker.php';

// Force check if requested
if (isset($_GET['force_update_check'])) {
    UpdateChecker::checkForUpdates(true);
    header('Location: /admin/dashboard.php');
    exit;
}

// Check for updates (will be cached for 12 hours)
$updateInfo = UpdateChecker::checkForUpdates();

// AUTO-REDIRECT TO WELCOME ON FIRST LOGIN - FIXED WITH ABSOLUTE PATH
if (!get_setting('welcome_dismissed', false) && !isset($_SESSION['seen_welcome'])) {
    $_SESSION['seen_welcome'] = true;
    
    // FIXED: Use absolute redirect
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
    $host = $_SERVER['HTTP_HOST'];
    
    while (ob_get_level()) ob_end_clean();
    header("Location: " . $protocol . $host . "/admin/welcome.php");
    exit;
}

$flashMessages = get_flash();

// FIXED: Get user safely (check_admin.php is now loaded, so current_user() exists)
$user = function_exists('current_user') ? current_user() : [
    'username' => $_SESSION['username'] ?? 'Admin',
    'role' => $_SESSION['user_role'] ?? 'admin'
];

// Check analytics settings
$gaEnabled = get_setting('google_analytics_enabled', false);
$gaTrackingId = get_setting('google_analytics_id', '');
$internalAnalyticsEnabled = get_setting('internal_analytics_enabled', true);

// Get selected periods for each section
$periodStats = $_GET['period_stats'] ?? '24h';
$periodPages = $_GET['period_pages'] ?? '7d';
$periodCountries = $_GET['period_countries'] ?? '7d';

$validPeriods = ['24h', '7d', '30d'];
if (!in_array($periodStats, $validPeriods)) $periodStats = '24h';
if (!in_array($periodPages, $validPeriods)) $periodPages = '7d';
if (!in_array($periodCountries, $validPeriods)) $periodCountries = '7d';

// Get analytics stats for each section
$stats = $internalAnalyticsEnabled ? AnalyticsTracker::getStats($periodStats) : null;
$pagesData = $internalAnalyticsEnabled ? AnalyticsTracker::getStats($periodPages) : null;
$countriesData = $internalAnalyticsEnabled ? AnalyticsTracker::getStats($periodCountries) : null;

// Get recent pages
$recentPages = db()->fetchAll(
    "SELECT slug, title, last_modified FROM pages ORDER BY last_modified DESC LIMIT 5"
);

// Get recent blog posts  
$recentBlogPosts = db()->fetchAll(
    "SELECT id, title, status, created_at FROM blog_posts ORDER BY created_at DESC LIMIT 5"
);

// Period labels
$periodLabels = [
    '24h' => 'Letzte 24 Stunden',
    '7d' => 'Letzte 7 Tage',
    '30d' => 'Letzte 30 Tage'
];

// Flush output buffer
ob_end_flush();
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard</title>
<?php
$adminCssPath = APP_ROOT . '/assets/css/admin.css';
$adminCssVersion = file_exists($adminCssPath) ? filemtime($adminCssPath) : time();
?>
<link rel="stylesheet" href="/assets/css/admin.css?v=<?= $adminCssVersion ?>">
<!-- Keep all your existing CSS styles here - they're fine -->
    <style>
    /* Update Notification Banner */
    .update-banner {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        padding: 16px 20px;
        border-radius: 12px;
        margin-bottom: 24px;
        display: flex;
        align-items: center;
        gap: 16px;
        box-shadow: 0 4px 12px rgba(102, 126, 234, 0.3);
        animation: slideDown 0.5s ease-out;
        position: relative;
        overflow: hidden;
    }
    
    .update-banner::before {
        content: '';
        position: absolute;
        top: 0;
        left: -100%;
        width: 100%;
        height: 100%;
        background: linear-gradient(90deg, transparent, rgba(255,255,255,0.1), transparent);
        animation: shimmer 3s infinite;
    }
    
    @keyframes shimmer {
        100% { left: 100%; }
    }
    
    @keyframes slideDown {
        from {
            opacity: 0;
            transform: translateY(-20px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }
    
    .update-banner.critical {
        background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
        animation: slideDown 0.5s ease-out, pulse 2s infinite;
    }
    
    @keyframes pulse {
        0%, 100% { transform: scale(1); }
        50% { transform: scale(1.01); }
    }
    
    .update-icon {
        font-size: 32px;
        line-height: 1;
        flex-shrink: 0;
        animation: bounce 2s infinite;
    }
    
    @keyframes bounce {
        0%, 100% { transform: translateY(0); }
        50% { transform: translateY(-5px); }
    }
    
    .update-content {
        flex: 1;
        min-width: 0;
    }
    
    .update-content h3 {
        margin: 0 0 6px 0;
        font-size: 18px;
        font-weight: 700;
    }
    
    .update-content p {
        margin: 0;
        font-size: 14px;
        opacity: 0.95;
        line-height: 1.5;
    }
    
    .update-action {
        flex-shrink: 0;
    }
    
    .update-btn {
        background: white;
        color: #667eea;
        padding: 10px 20px;
        border-radius: 8px;
        text-decoration: none;
        font-weight: 600;
        font-size: 14px;
        display: inline-flex;
        align-items: center;
        gap: 6px;
        transition: all 0.2s;
        white-space: nowrap;
    }
    
    .update-banner.critical .update-btn {
        color: #ef4444;
    }
    
    .update-btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 8px rgba(0,0,0,0.2);
    }
    
    .update-dismiss {
        background: transparent;
        border: 2px solid rgba(255,255,255,0.3);
        color: white;
        padding: 8px 16px;
        border-radius: 6px;
        cursor: pointer;
        font-size: 13px;
        margin-left: 8px;
        transition: all 0.2s;
    }
    
    .update-dismiss:hover {
        background: rgba(255,255,255,0.1);
        border-color: rgba(255,255,255,0.5);
    }
    
    /* Update Badge for System Info */
    .system-info-with-badge {
        position: relative;
    }
    
    .update-badge {
        position: absolute;
        top: -8px;
        right: -8px;
        background: #ef4444;
        color: white;
        width: 20px;
        height: 20px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 11px;
        font-weight: 700;
        animation: pulse 2s infinite;
        box-shadow: 0 2px 8px rgba(239, 68, 68, 0.4);
    }
    
    /* Existing Styles */
    .period-selector {
        display: flex;
        gap: 8px;
        flex-wrap: wrap;
    }
    
    .period-btn {
        padding: 6px 14px;
        border: 1px solid #d1d5db;
        background: white;
        border-radius: 6px;
        cursor: pointer;
        transition: all 0.2s;
        text-decoration: none;
        color: #374151;
        font-weight: 500;
        font-size: 13px;
        white-space: nowrap;
    }
    
    .period-btn:hover {
        background: #f3f4f6;
        border-color: #9ca3af;
    }
    
    .period-btn.active {
        background: #3b82f6;
        color: white;
        border-color: #3b82f6;
    }
    
    .card-header {
        display: flex;
        justify-content: space-between;
        align-items: flex-start;
        gap: 20px;
        flex-wrap: wrap;
        padding: 20px;
        border-bottom: 1px solid #e5e7eb;
    }
    
    .card-header-left {
        flex: 1;
        min-width: 200px;
    }
    
    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 20px;
        margin: 20px 0;
    }
    
    .stat-card {
        background: white;
        padding: 20px;
        border-radius: 8px;
        border: 1px solid #e5e7eb;
    }
    
    .stat-card.disabled {
        opacity: 0.5;
        background: #f9fafb;
    }
    
    .stat-card h3 {
        font-size: 32px;
        font-weight: 700;
        margin: 8px 0 4px;
        color: #111827;
    }
    
    .stat-card p {
        font-size: 14px;
        color: #6b7280;
        margin: 0;
    }
    
    .stat-card small {
        font-size: 12px;
        color: #9ca3af;
    }
    
    /* Mobile Responsive for Update Banner */
    @media (max-width: 768px) {
        .update-banner {
            flex-direction: column;
            align-items: flex-start;
            padding: 16px;
        }
        
        .update-icon {
            font-size: 28px;
            align-self: center;
        }
        
        .update-content h3 {
            font-size: 16px;
        }
        
        .update-content p {
            font-size: 13px;
        }
        
        .update-action {
            width: 100%;
            display: flex;
            flex-direction: column;
            gap: 8px;
        }
        
        .update-btn,
        .update-dismiss {
            width: 100%;
            justify-content: center;
            margin-left: 0;
        }
    }
    </style>
</head>
<body>
    <?php include 'includes/admin_header.php'; ?>
    
    <div class="admin-container">
        <?php include 'includes/admin_sidebar.php'; ?>
        
        <main class="admin-content">
            
            <?php 
            // ===================================
            // UPDATE NOTIFICATION BANNER
            // ===================================
            if ($updateInfo['success'] && $updateInfo['update_available'] && 
                (!isset($_SESSION['update_dismissed_' . $updateInfo['latest_version']]))): 
                $isCritical = $updateInfo['is_critical'] ?? false;
            ?>
                <div class="update-banner <?= $isCritical ? 'critical' : '' ?>" id="updateBanner">
                    <div class="update-icon">
                        <?= $isCritical ? '⚠️' : '🎉' ?>
                    </div>
                    <div class="update-content">
                        <h3>
                            <?= $isCritical ? 'Kritisches Update verfügbar!' : 'Neues Update verfügbar!' ?>
                        </h3>
                        <p>
                            <strong>Version <?= htmlspecialchars($updateInfo['latest_version']) ?></strong> 
                            ist jetzt verfügbar · 
                            Ihre Version: <?= htmlspecialchars($updateInfo['current_version']) ?>
                        </p>
                    </div>
                    <div class="update-action">
                        <a href="https://manidesign.de/mani-cms/" 
                           target="_blank" 
                           class="update-btn">
                            📥 Jetzt updaten
                        </a>
                        <button onclick="dismissUpdate('<?= htmlspecialchars($updateInfo['latest_version']) ?>')" 
                                class="update-dismiss">
                            Später
                        </button>
                    </div>
                </div>
            <?php endif; ?>
            
            <?php foreach ($flashMessages as $flash): ?>
                <div class="alert alert-<?= e($flash['type']) ?>">
                    <?= e($flash['message']) ?>
                </div>
            <?php endforeach; ?>
            
            <div class="page-header">
                <h1>📊 Dashboard</h1>
                <p>Willkommen zurück, <strong><?= e($user['username']) ?></strong>! 👋</p>
            </div>
            
            <!-- Internal Analytics Section -->
            <?php if ($internalAnalyticsEnabled && $stats): ?>
                <div class="content-card">
                    <div class="card-header" style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 15px;">
                        <div>
                            <h2 class="card-title">📈 Analytics (DSGVO-konform)</h2>
                            <p style="font-size: 13px; color: #6b7280; margin-top: 4px;">
                                Anonymisierte Daten ohne Cookies
                            </p>
                        </div>
                        
                        <!-- Period Selector for Stats -->
                        <div class="period-selector">
                            <?php foreach ($validPeriods as $p): ?>
                                <a href="?period_stats=<?= $p ?>&period_pages=<?= $periodPages ?>&period_countries=<?= $periodCountries ?>" 
                                   class="period-btn <?= $periodStats === $p ? 'active' : '' ?>">
                                    <?= $periodLabels[$p] ?>
                                </a>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    
                    <!-- Stats Grid -->
                    <div class="stats-grid" style="padding: 20px;">
                        <div class="stat-card">
                            <p>👥 Besucher Gesamt</p>
                            <h3><?= number_format($stats['total_visitors']) ?></h3>
                        </div>
                        
                        <div class="stat-card">
                            <p>🔄 Sessions Gesamt</p>
                            <h3><?= number_format($stats['total_sessions']) ?></h3>
                        </div>
                        
                        <div class="stat-card">
                            <p>📊 Sessions/Besucher</p>
                            <h3><?= $stats['sessions_per_visitor'] ?></h3>
                            <small>Ø pro Besucher</small>
                        </div>
                        
                        <div class="stat-card">
                            <p>🆕 Neue Besucher</p>
                            <h3><?= number_format($stats['new_visitors']) ?></h3>
                            <small><?= $stats['total_visitors'] > 0 ? round(($stats['new_visitors'] / $stats['total_visitors']) * 100) : 0 ?>%</small>
                        </div>
                        
                        <div class="stat-card">
                            <p>🔁 Wiederkehrend</p>
                            <h3><?= number_format($stats['returning_visitors']) ?></h3>
                            <small><?= $stats['total_visitors'] > 0 ? round(($stats['returning_visitors'] / $stats['total_visitors']) * 100) : 0 ?>%</small>
                        </div>
                        
                        <div class="stat-card">
                            <p>⏱️ Ø Session-Dauer</p>
                            <h3><?= gmdate("i:s", $stats['avg_session_duration']) ?></h3>
                            <small>Min:Sek</small>
                        </div>
                        
                        <div class="stat-card">
                            <p>↩️ Absprungrate</p>
                            <h3><?= $stats['bounce_rate'] ?>%</h3>
                            <small>Single-Page Sessions</small>
                        </div>
                    </div>
                </div>
                
                <!-- Top Pages & Countries -->
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-top: 20px;">
                    <!-- Top 5 Pages -->
                    <div class="content-card">
                        <div class="card-header" style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 10px;">
                            <h2 class="card-title">🔥 Top 5 Seiten</h2>
                            <div class="period-selector" style="margin: 0;">
                                <?php foreach ($validPeriods as $p): ?>
                                    <a href="?period_stats=<?= $periodStats ?>&period_pages=<?= $p ?>&period_countries=<?= $periodCountries ?>#pages" 
                                       class="period-btn <?= $periodPages === $p ? 'active' : '' ?>"
                                       style="padding: 4px 12px; font-size: 13px;">
                                        <?= str_replace(['Letzte ', ' Stunden', ' Tage'], ['', 'h', 'd'], $periodLabels[$p]) ?>
                                    </a>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        <table class="data-table" id="pages">
                            <thead>
                                <tr>
                                    <th>Seite</th>
                                    <th>Aufrufe</th>
                                    <th>Besucher</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($pagesData['top_pages'])): ?>
                                    <tr>
                                        <td colspan="3" style="text-align: center; color: #9ca3af; padding: 40px;">
                                            Noch keine Daten
                                        </td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($pagesData['top_pages'] as $page): ?>
                                        <tr>
                                            <td><strong><?= e($page['page_url']) ?></strong></td>
                                            <td><?= number_format($page['views']) ?></td>
                                            <td><?= number_format($page['visitors']) ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Top Countries -->
                    <div class="content-card">
                        <div class="card-header" style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 10px;">
                            <h2 class="card-title">🌍 Top Standorte</h2>
                            <div class="period-selector" style="margin: 0;">
                                <?php foreach ($validPeriods as $p): ?>
                                    <a href="?period_stats=<?= $periodStats ?>&period_pages=<?= $periodPages ?>&period_countries=<?= $p ?>#countries" 
                                       class="period-btn <?= $periodCountries === $p ? 'active' : '' ?>"
                                       style="padding: 4px 12px; font-size: 13px;">
                                        <?= str_replace(['Letzte ', ' Stunden', ' Tage'], ['', 'h', 'd'], $periodLabels[$p]) ?>
                                    </a>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        <table class="data-table" id="countries">
                            <thead>
                                <tr>
                                    <th>Land / Stadt</th>
                                    <th>Besucher</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($countriesData['top_countries'])): ?>
                                    <tr>
                                        <td colspan="2" style="text-align: center; color: #9ca3af; padding: 40px;">
                                            Noch keine Daten
                                        </td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($countriesData['top_countries'] as $country): ?>
                                        <tr>
                                            <td>
                                                <strong><?= e($country['country_name'] ?? 'Unbekannt') ?></strong>
                                                <?php if ($country['city']): ?>
                                                    <br><small style="color: #9ca3af;">📍 <?= e($country['city']) ?></small>
                                                <?php endif; ?>
                                            </td>
                                            <td><?= number_format($country['visitors']) ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                
                
                <!-- User Flow (Full Width) -->
                <div class="content-card" style="margin-top: 20px;">
                    <div class="card-header">
                        <h2 class="card-title">🔀 User Flow</h2>
                        <small style="color: #9ca3af;">Basiert auf <?= $periodLabels[$periodStats] ?></small>
                    </div>
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Einstieg → Ausstieg</th>
                                <th>Sessions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($stats['user_flow'])): ?>
                                <tr>
                                    <td colspan="2" style="text-align: center; color: #9ca3af; padding: 40px;">
                                        Noch keine Daten
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($stats['user_flow'] as $flow): ?>
                                    <tr>
                                        <td>
                                            <strong><?= e($flow['entry_page']) ?></strong> 
                                            → 
                                            <?= e($flow['exit_page']) ?>
                                        </td>
                                        <td><?= number_format($flow['sessions']) ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <!-- Analytics Disabled -->
                <div class="content-card">
                    <div class="card-header">
                        <h2 class="card-title">📈 Analytics</h2>
                    </div>
                    <div style="padding: 40px; text-align: center; color: #9ca3af;">
                        <span style="font-size: 64px; opacity: 0.3;">📊</span>
                        <h3 style="margin: 16px 0 8px;">Analytics deaktiviert</h3>
                        <p>Aktiviere interne Analytics in den Einstellungen.</p>
                    </div>
                </div>
            <?php endif; ?>
            
            <!-- Google Analytics -->
            <?php if ($gaEnabled && !empty($gaTrackingId)): ?>
                <div class="content-card" style="margin-top: 20px;">
                    <div class="card-header">
                        <h2 class="card-title">📊 Google Analytics</h2>
                    </div>
                    <div style="padding: 20px; text-align: center;">
                        <p style="color: #6b7280; margin-bottom: 16px;">
                            Tracking-ID: <code><?= e($gaTrackingId) ?></code>
                        </p>
                        <a href="https://analytics.google.com" target="_blank" class="btn btn-primary">
                            Analytics öffnen →
                        </a>
                    </div>
                </div>
            <?php endif; ?>
            
            <!-- Recent Activity -->
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-top: 20px;">
                <!-- Recent Pages -->
                <div class="content-card">
                    <div class="card-header">
                        <h2 class="card-title">📄 Letzte Änderungen</h2>
                    </div>
                    <table class="data-table">
                        <tbody>
                            <?php if (empty($recentPages)): ?>
                                <tr>
                                    <td style="text-align: center; color: #9ca3af; padding: 40px;">
                                        Noch keine Seiten
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($recentPages as $page): ?>
                                    <tr>
                                        <td>
                                            <strong><?= e($page['title']) ?></strong><br>
                                            <small style="color: #9ca3af;"><?= format_date($page['last_modified'], 'd.m.Y H:i') ?></small>
                                        </td>
                                        <td>
                                            <a href="page_flexible_edit.php?slug=<?= e($page['slug']) ?>" 
                                               class="btn btn-sm btn-secondary">
                                                Bearbeiten
                                            </a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
                
                <!-- Recent Blog Posts -->
                <div class="content-card">
                    <div class="card-header">
                        <h2 class="card-title">📝 Neueste Posts</h2>
                    </div>
                    <table class="data-table">
                        <tbody>
                            <?php if (empty($recentBlogPosts)): ?>
                                <tr>
                                    <td style="text-align: center; color: #9ca3af; padding: 40px;">
                                        Noch keine Posts
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($recentBlogPosts as $post): ?>
                                    <tr>
                                        <td>
                                            <strong><?= e($post['title']) ?></strong><br>
                                            <small style="color: #9ca3af;">
                                                <?= $post['status'] === 'published' ? '✅' : '📝' ?> 
                                                <?= format_date($post['created_at'], 'd.m.Y H:i') ?>
                                            </small>
                                        </td>
                                        <td>
                                            <a href="blog_create_edit.php?id=<?= $post['id'] ?>" 
                                               class="btn btn-sm btn-secondary">
                                                Bearbeiten
                                            </a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            
            <!-- System Info -->
            <div class="content-card" style="margin-top: 30px;">
                <div class="card-header">
                    <h2 class="card-title">ℹ️ System-Info</h2>
                </div>
                <div style="padding: 20px; display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px;">
                    <div>
                        <p style="font-size: 12px; color: #9ca3af; margin-bottom: 4px;">PHP Version</p>
                        <p style="font-size: 16px; font-weight: 600;"><?= PHP_VERSION ?></p>
                    </div>
                    <div>
                        <p style="font-size: 12px; color: #9ca3af; margin-bottom: 4px;">Umgebung</p>
                        <p style="font-size: 16px; font-weight: 600;">
                            <?= APP_ENV === 'production' ? '🟢 Production' : '🟡 Development' ?>
                        </p>
                    </div>
                    <div>
                        <p style="font-size: 12px; color: #9ca3af; margin-bottom: 4px;">Interne Analytics</p>
                        <p style="font-size: 16px; font-weight: 600;">
                            <?= $internalAnalyticsEnabled ? '✅ Aktiv' : '❌ Inaktiv' ?>
                        </p>
                    </div>
                    <div>
                        <p style="font-size: 12px; color: #9ca3af; margin-bottom: 4px;">Google Analytics</p>
                        <p style="font-size: 16px; font-weight: 600;">
                            <?= $gaEnabled ? '✅ Aktiv' : '❌ Inaktiv' ?>
                        </p>
                    </div>
                    
                    <!-- CMS Version with Update Badge -->
                    <div class="system-info-with-badge">
                        <?php if ($updateInfo['success'] && $updateInfo['update_available']): ?>
                            <span class="update-badge">1</span>
                        <?php endif; ?>
                        <p style="font-size: 12px; color: #9ca3af; margin-bottom: 4px;">CMS Version</p>
                        <p style="font-size: 16px; font-weight: 600;">
                            v<?= htmlspecialchars(UpdateChecker::getCurrentVersion()) ?>
                            <?php if ($updateInfo['success'] && $updateInfo['update_available']): ?>
                                <a href="https://manidesign.de/mani-cms/" 
                                   target="_blank"
                                   style="font-size: 12px; color: #ef4444; text-decoration: none; margin-left: 8px;">
                                    (Update: v<?= htmlspecialchars($updateInfo['latest_version']) ?>)
                                </a>
                            <?php endif; ?>
                        </p>
                    </div>
                </div>
            </div>
        </main>
    </div>
    
    <script>
    // Dismiss update notification
    function dismissUpdate(version) {
        const banner = document.getElementById('updateBanner');
        if (banner) {
            banner.style.animation = 'slideUp 0.3s ease-out';
            setTimeout(() => {
                banner.style.display = 'none';
            }, 300);
        }
        
        // Store dismissal in session (via AJAX)
        fetch('/admin/dashboard.php?dismiss_update=' + version)
            .catch(err => console.error('Failed to dismiss update:', err));
        
        // Also store in sessionStorage for immediate effect
        sessionStorage.setItem('update_dismissed_' + version, 'true');
    }
    
    // Check if update was dismissed this session
    window.addEventListener('DOMContentLoaded', function() {
        const banner = document.getElementById('updateBanner');
        if (banner) {
            const version = banner.querySelector('.update-btn')?.href?.match(/v=(\d+\.\d+)/)?.[1];
            if (version && sessionStorage.getItem('update_dismissed_' + version)) {
                banner.style.display = 'none';
            }
        }
    });
    
    // Slide up animation
    const style = document.createElement('style');
    style.textContent = `
        @keyframes slideUp {
            from {
                opacity: 1;
                transform: translateY(0);
            }
            to {
                opacity: 0;
                transform: translateY(-20px);
            }
        }
    `;
    document.head.appendChild(style);
    </script>
    
    <!-- CRITICAL: Add main.js for mobile menu functionality -->
    <script src="/assets/js/main.js"></script>
</body>
</html>
<?php
// Handle AJAX dismissal
if (isset($_GET['dismiss_update'])) {
    $version = $_GET['dismiss_update'];
    $_SESSION['update_dismissed_' . $version] = true;
    exit;
}
?>