<?php
/**
 * cms-core/public/webview/admin/error_logs.php
 * FIXED: Now uses LOGS_ROOT constant from config
 */

require_once __DIR__ . '/../../../backend/authorization/check_admin.php';

// Use LOGS_ROOT from config.php
$logFile = LOGS_ROOT . '/error.log';
$logs = [];
$logExists = file_exists($logFile);

// Hole Log-Einträge
if ($logExists) {
    $content = file_get_contents($logFile);
    $lines = explode("\n", $content);
    
    // Parse Log-Einträge
    foreach ($lines as $line) {
        if (empty(trim($line))) continue;
        
        // Parse Format: [2024-01-01 12:00:00] [LEVEL] Message
        if (preg_match('/^\[(.*?)\]\s*\[(.*?)\]\s*(.*)$/', $line, $matches)) {
            $logs[] = [
                'timestamp' => $matches[1],
                'level' => $matches[2],
                'message' => $matches[3],
                'raw' => $line
            ];
        } else {
            // Falls Format nicht passt, als "raw" speichern
            $logs[] = [
                'timestamp' => '',
                'level' => 'UNKNOWN',
                'message' => $line,
                'raw' => $line
            ];
        }
    }
    
    // Neueste zuerst
    $logs = array_reverse($logs);
}

// Handle Actions
if (is_post()) {
    try {
        CSRF::require();
    } catch (Exception $e) {
        set_flash('error', 'Ungültiger CSRF-Token oder Serverfehler.');
        error_log("CSRF validation failed: " . $e->getMessage());
        redirect('/cms-core/public/webview/admin/error_logs.php');
        exit;
    }
    
    $action = post('action');
    
    if ($action === 'clear') {
        file_put_contents($logFile, '');
        set_flash('success', 'Error-Log erfolgreich geleert.');
        redirect('/cms-core/public/webview/admin/error_logs.php');
        exit;
    }
    
    if ($action === 'download') {
        header('Content-Type: text/plain');
        header('Content-Disposition: attachment; filename="error_log_' . date('Y-m-d_H-i-s') . '.txt"');
        readfile($logFile);
        exit;
    }
}

$flashMessages = get_flash();

// Statistiken
$stats = [
    'total' => count($logs),
    'critical' => count(array_filter($logs, fn($l) => $l['level'] === 'CRITICAL')),
    'error' => count(array_filter($logs, fn($l) => $l['level'] === 'ERROR')),
    'warning' => count(array_filter($logs, fn($l) => $l['level'] === 'WARNING')),
];

// Filter
$filterLevel = get_param('level', 'all');
if ($filterLevel !== 'all') {
    $logs = array_filter($logs, fn($l) => strtolower($l['level']) === strtolower($filterLevel));
}

// Pagination
$perPage = 50;
$pageParam = get_param('page', 1);
$page = max(1, (int)(is_array($pageParam) ? $pageParam[0] : $pageParam));
$totalLogs = count($logs);
$totalPages = max(1, ceil($totalLogs / $perPage));
$offset = ($page - 1) * $perPage;
$logsPage = array_slice($logs, $offset, $perPage);
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Error Logs</title>
<?php
$adminCssPath = APP_ROOT . '/assets/css/admin.css';
$adminCssVersion = file_exists($adminCssPath) ? filemtime($adminCssPath) : time();
?>
<link rel="stylesheet" href="/assets/css/admin.css?v=<?= $adminCssVersion ?>"></head>
<body>
    <?php include 'includes/admin_header.php'; ?>
    <?php include 'includes/maintenance_banner.php'; ?>
    
    <div class="admin-container">
        <?php include 'includes/admin_sidebar.php'; ?>
        
        <main class="admin-content">
            <?php foreach ($flashMessages as $flash): ?>
                <div class="alert alert-<?= e($flash['type']) ?>">
                    <?= e($flash['message']) ?>
                </div>
            <?php endforeach; ?>
            
            <div class="page-header">
                <h1>🔍 Error Logs</h1>
                <p>System-Fehler und Warnungen überwachen</p>
            </div>
            
            <?php if ($logExists): ?>
                <!-- Statistiken -->
                <div class="log-stats-grid">
                    <div class="log-stat-card">
                        <div class="log-stat-number"><?= $stats['total'] ?></div>
                        <div class="log-stat-label">Gesamt</div>
                    </div>
                    
                    <div class="log-stat-card critical">
                        <div class="log-stat-number"><?= $stats['critical'] ?></div>
                        <div class="log-stat-label">Critical</div>
                    </div>
                    
                    <div class="log-stat-card error">
                        <div class="log-stat-number"><?= $stats['error'] ?></div>
                        <div class="log-stat-label">Errors</div>
                    </div>
                    
                    <div class="log-stat-card warning">
                        <div class="log-stat-number"><?= $stats['warning'] ?></div>
                        <div class="log-stat-label">Warnings</div>
                    </div>
                </div>
                
                <!-- Filter -->
                <div class="filter-tabs">
                    <a href="error_logs.php?level=all" class="filter-tab <?= $filterLevel === 'all' ? 'active' : '' ?>">
                        Alle <span class="badge" style="background: rgba(0,0,0,0.2);"><?= $stats['total'] ?></span>
                    </a>
                    <a href="error_logs.php?level=critical" class="filter-tab <?= $filterLevel === 'critical' ? 'active' : '' ?>">
                        Critical <span class="badge" style="background: rgba(220,38,38,0.2); color: #dc2626;"><?= $stats['critical'] ?></span>
                    </a>
                    <a href="error_logs.php?level=error" class="filter-tab <?= $filterLevel === 'error' ? 'active' : '' ?>">
                        Error <span class="badge" style="background: rgba(239,68,68,0.2); color: #ef4444;"><?= $stats['error'] ?></span>
                    </a>
                    <a href="error_logs.php?level=warning" class="filter-tab <?= $filterLevel === 'warning' ? 'active' : '' ?>">
                        Warning <span class="badge" style="background: rgba(245,158,11,0.2); color: #f59e0b;"><?= $stats['warning'] ?></span>
                    </a>
                </div>
                
                <!-- Log Viewer -->
                <div class="log-viewer">
                    <div class="log-header">
                        <div class="log-title">
                            📄 error.log (<?= $totalLogs ?> Einträge)
                        </div>
                        <div class="log-actions">
                            <form method="POST" style="display: inline;">
                                <?= CSRF::field() ?>
                                <input type="hidden" name="action" value="download">
                                <button type="submit" class="btn btn-sm btn-secondary">
                                    💾 Download
                                </button>
                            </form>
                            
                            <form method="POST" style="display: inline;" onsubmit="return confirm('Wirklich alle Logs löschen?');">
                                <?= CSRF::field() ?>
                                <input type="hidden" name="action" value="clear">
                                <button type="submit" class="btn btn-sm btn-danger">
                                    🗑️ Logs leeren
                                </button>
                            </form>
                        </div>
                    </div>
                    
                    <div class="log-content">
                        <?php if (empty($logsPage)): ?>
                            <div class="empty-logs">
                                <div style="font-size: 48px; margin-bottom: 15px;">✨</div>
                                <p>Keine <?= $filterLevel !== 'all' ? strtoupper($filterLevel) : '' ?> Log-Einträge gefunden.</p>
                            </div>
                        <?php else: ?>
                            <?php foreach ($logsPage as $log): ?>
                                <div class="log-entry">
                                    <div class="log-timestamp"><?= e($log['timestamp']) ?></div>
                                    <div>
                                        <span class="log-level <?= e($log['level']) ?>">
                                            <?= e($log['level']) ?>
                                        </span>
                                    </div>
                                    <div class="log-message"><?= e($log['message']) ?></div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Pagination -->
                <?php if ($totalPages > 1): ?>
                    <div class="pagination">
                        <?php if ($page > 1): ?>
                            <a href="?level=<?= e($filterLevel) ?>&page=<?= $page - 1 ?>">← Zurück</a>
                        <?php endif; ?>
                        
                        <span>Seite <?= $page ?> von <?= $totalPages ?></span>
                        
                        <?php if ($page < $totalPages): ?>
                            <a href="?level=<?= e($filterLevel) ?>&page=<?= $page + 1 ?>">Weiter →</a>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
                
            <?php else: ?>
                <div class="content-card">
                    <div style="text-align: center; padding: 60px 20px;">
                        <div style="font-size: 64px; margin-bottom: 20px;">📝</div>
                        <h3 style="color: #6b7280; margin-bottom: 10px;">Keine Log-Datei gefunden</h3>
                        <p style="color: #9ca3af;">Die Error-Log-Datei wurde noch nicht erstellt.</p>
                        <p style="color: #9ca3af; font-size: 13px; margin-top: 10px;">
                            Pfad: <code style="background: #f3f4f6; padding: 4px 8px; border-radius: 4px;"><?= e($logFile) ?></code>
                        </p>
                        <p style="color: #9ca3af; font-size: 13px; margin-top: 5px;">
                            LOGS_ROOT: <code style="background: #f3f4f6; padding: 4px 8px; border-radius: 4px;"><?= e(LOGS_ROOT) ?></code>
                        </p>
                    </div>
                </div>
            <?php endif; ?>
        </main>
    </div>
    
    <script src="/assets/js/main.js"></script>
    <script>
    // Auto-scroll Log-Content auf neuesten Eintrag
    const logContent = document.querySelector('.log-content');
    if (logContent && <?= json_encode(empty(get_param('page'))) ?>) {
        // Nur beim ersten Laden, nicht bei Pagination
        // logContent.scrollTop = 0; // Neueste oben
    }
    
    // Auto-refresh alle 30 Sekunden (optional)
    // setTimeout(() => location.reload(), 30000);
    </script>
</body>
</html>