<?php
/**
 * cms-core/public/webview/admin/includes/admin_header.php
 * Clean dropdown with site name from settings + Language Switcher
 */

// CRITICAL: Fallback if Translation system not loaded yet
if (!function_exists('t')) {
    function t($text, $vars = []) {
        return $text; // Just return German text if Translation not available
    }
}

if (!function_exists('current_lang')) {
    function current_lang() {
        // Check session first, then constant, then default
        if (isset($_SESSION['language'])) {
            return $_SESSION['language'];
        }
        return defined('APP_LANGUAGE') ? APP_LANGUAGE : 'de';
    }
}

// CRITICAL: Handle language switch IMMEDIATELY (before any rendering)
if (isset($_GET['switch_lang'])) {
    $newLang = $_GET['switch_lang'];
    if (in_array($newLang, ['de', 'en'])) {
        // Set in session IMMEDIATELY
        $_SESSION['language'] = $newLang;
        
        // If Translation class is loaded, use it
        if (class_exists('Translation')) {
            Translation::setLanguage($newLang);
        }
        
        // Only redirect if headers not sent yet
        if (!headers_sent()) {
            // Redirect to clean URL (remove query string)
            $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
            $host = $_SERVER['HTTP_HOST'];
            $path = strtok($_SERVER['REQUEST_URI'], '?');
            
            header('Location: ' . $protocol . $host . $path);
            exit;
        }
    }
}

$user = current_user();
$siteName = get_setting('site_name', 'CMS Admin');
$currentLang = current_lang();
$languages = [
    'de' => '🇩🇪 Deutsch',
    'en' => '🇬🇧 English'
];
?>
<header class="admin-header">
    <div class="header-left">
        <!-- Sidebar Toggle (Mobile) -->
        <button class="admin-sidebar-toggle" id="sidebar-toggle" onclick="toggleSidebar()">
            <span></span>
            <span></span>
            <span></span>
        </button>
        
        <div class="logo">
            <h2><?= e($siteName) ?></h2>
            <span class="admin-badge">ADMIN</span>
        </div>

        <!-- UPDATE NOTIFICATION -->
        <?php
        $updateAvailable = get_setting('cms_update_available', false);
        $updateVersion = get_setting('cms_update_version', '');
        if ($updateAvailable): ?>
            <a href="/admin/cms_core_updater.php" class="notification-badge notification-badge-pulse" 
               data-count="1"
               style="margin-left: 16px; padding: 8px 16px; background: linear-gradient(135deg, #10b981, #059669); color: white; border-radius: 20px; text-decoration: none; font-size: 13px; font-weight: 600; display: inline-flex; align-items: center; gap: 8px;">
                🎉 <?= t('Update verfügbar') ?> (<?= e($updateVersion) ?>)
            </a>
        <?php endif; ?>
    </div>
    
    <div class="header-right">
        <!-- Language Switcher -->
        <div class="language-switcher">
            <button class="lang-btn" onclick="toggleLangMenu(event)" title="<?= t('Sprache wechseln') ?>">
                <?= $currentLang === 'de' ? '🇩🇪' : '🇬🇧' ?>
                <span class="lang-text"><?= strtoupper($currentLang) ?></span>
            </button>
            
            <div class="lang-menu" id="langMenu">
                <?php foreach ($languages as $code => $name): ?>
                    <?php if ($code !== $currentLang): ?>
                        <a href="?switch_lang=<?= $code ?>" class="lang-option">
                            <?= $name ?>
                        </a>
                    <?php endif; ?>
                <?php endforeach; ?>
            </div>
        </div>
        
        <!-- User Menu Dropdown -->
        <div class="user-dropdown">
            <button class="user-dropdown-trigger" onclick="toggleUserDropdown(event)">
                <div class="user-menu">
                    <span class="user-name"><?= e($user['username']) ?></span>
                    <span class="user-role"><?= e($user['role'] ?? 'admin') ?></span>
                </div>
                <span class="dropdown-arrow">▼</span>
            </button>
            
            <div class="user-dropdown-menu" id="user-dropdown-menu">
                <a href="forgot_password.php" class="dropdown-item">
                    <span class="dropdown-icon">🔐</span>
                    <span><?= t('Passwort ändern') ?></span>
                </a>
                <div class="dropdown-divider"></div>
                <a href="/" target="_blank" class="dropdown-item">
                    <span class="dropdown-icon">🌐</span>
                    <span><?= t('Website ansehen') ?></span>
                </a>
                <div class="dropdown-divider"></div>
                
                <!-- Logout as FORM with POST -->
                <form method="POST" action="logout.php" id="logout-form" style="margin: 0;">
                    <?= CSRF::field() ?>
                    <button type="submit" class="dropdown-item dropdown-item-danger" style="width: 100%; text-align: left; background: none; border: none; cursor: pointer; font: inherit; padding: 12px 16px;">
                        <span class="dropdown-icon">🚪</span>
                        <span><?= t('Abmelden') ?></span>
                    </button>
                </form>
            </div>
        </div>
    </div>
</header>

<style>
/* Language Switcher Styles */
.language-switcher {
    position: relative;
    display: inline-block;
    margin-right: 16px;
}

.lang-btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 8px 16px;
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    background: white;
    cursor: pointer;
    font-size: 16px;
    transition: all 0.2s;
}

.lang-btn:hover {
    border-color: #667eea;
    background: #f9fafb;
}

.lang-text {
    font-weight: 600;
    color: #374151;
    font-size: 14px;
}

.lang-menu {
    position: absolute;
    top: calc(100% + 8px);
    right: 0;
    background: white;
    border: 1px solid #e5e7eb;
    border-radius: 12px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.15);
    min-width: 150px;
    display: none;
    z-index: 1000;
    overflow: hidden;
}

.lang-menu.active {
    display: block;
    animation: dropdown-fade-in 0.2s ease;
}

.lang-option {
    display: block;
    padding: 12px 16px;
    color: #374151;
    text-decoration: none;
    transition: background 0.2s;
}

.lang-option:hover {
    background: #f0f4ff;
    color: #667eea;
}

/* User Dropdown Styles */
.user-dropdown {
    position: relative;
}

.user-dropdown-trigger {
    display: flex;
    align-items: center;
    gap: 12px;
    padding: 8px 16px;
    background: white;
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.2s;
}

.user-dropdown-trigger:hover {
    border-color: #667eea;
    background: #f9fafb;
}

.user-menu {
    display: flex;
    flex-direction: column;
    align-items: flex-start;
}

.user-name {
    font-weight: 600;
    font-size: 14px;
    color: #1f2937;
}

.user-role {
    font-size: 12px;
    color: #6b7280;
    text-transform: uppercase;
}

.dropdown-arrow {
    font-size: 10px;
    color: #6b7280;
    transition: transform 0.3s;
}

.user-dropdown-trigger.active .dropdown-arrow {
    transform: rotate(180deg);
}

.user-dropdown-menu {
    position: absolute;
    top: calc(100% + 8px);
    right: 0;
    background: white;
    border: 1px solid #e5e7eb;
    border-radius: 12px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.15);
    min-width: 220px;
    display: none;
    z-index: 1000;
    overflow: hidden;
}

.user-dropdown-menu.show {
    display: block;
    animation: dropdown-fade-in 0.2s ease;
}

@keyframes dropdown-fade-in {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.dropdown-item {
    display: flex;
    align-items: center;
    gap: 12px;
    padding: 12px 16px;
    color: #1f2937;
    text-decoration: none;
    transition: background 0.2s;
    border: none;
    width: 100%;
    text-align: left;
    background: none;
    cursor: pointer;
    font: inherit;
}

.dropdown-item:hover {
    background: #f9fafb;
}

.dropdown-item-danger:hover {
    background: #fef2f2;
    color: #dc2626;
}

.dropdown-icon {
    font-size: 16px;
    width: 20px;
    text-align: center;
}

.dropdown-divider {
    height: 1px;
    background: #e5e7eb;
    margin: 4px 0;
}

/* Mobile Adjustments */
@media (max-width: 768px) {
    .lang-btn {
        padding: 8px 12px;
    }
    
    .lang-text {
        display: none;
    }
    
    .language-switcher {
        margin-right: 12px;
    }
    
    .user-dropdown-trigger {
        padding: 6px 12px;
    }
    
    .user-name {
        font-size: 13px;
    }
    
    .user-role {
        font-size: 11px;
    }
}
</style>

<script>
// Toggle language menu
function toggleLangMenu(event) {
    event.stopPropagation();
    const menu = document.getElementById('langMenu');
    menu.classList.toggle('active');
}

// Toggle user dropdown
function toggleUserDropdown(event) {
    event.stopPropagation();
    const trigger = event.currentTarget;
    const menu = document.getElementById('user-dropdown-menu');
    
    // Close language menu if open
    document.getElementById('langMenu')?.classList.remove('active');
    
    // Toggle classes
    trigger.classList.toggle('active');
    menu.classList.toggle('show');
}

// Close dropdowns when clicking outside
document.addEventListener('click', function(event) {
    const langSwitcher = document.querySelector('.language-switcher');
    const userDropdown = document.querySelector('.user-dropdown');
    const langMenu = document.getElementById('langMenu');
    const userMenu = document.getElementById('user-dropdown-menu');
    const userTrigger = document.querySelector('.user-dropdown-trigger');
    
    // Close language menu
    if (langSwitcher && !langSwitcher.contains(event.target)) {
        langMenu?.classList.remove('active');
    }
    
    // Close user menu
    if (userDropdown && !userDropdown.contains(event.target)) {
        userMenu?.classList.remove('show');
        userTrigger?.classList.remove('active');
    }
});

// Prevent dropdown close when clicking inside menu
document.getElementById('user-dropdown-menu')?.addEventListener('click', function(event) {
    // Allow form submission for logout
    if (event.target.closest('form')) {
        return;
    }
    event.stopPropagation();
});

// Sidebar toggle function
function toggleSidebar() {
    const sidebar = document.querySelector('.admin-sidebar');
    const overlay = document.querySelector('.sidebar-overlay');
    
    if (sidebar) {
        sidebar.classList.toggle('mobile-active');
    }
    
    // Create overlay if it doesn't exist
    if (!overlay && sidebar?.classList.contains('mobile-active')) {
        const newOverlay = document.createElement('div');
        newOverlay.className = 'sidebar-overlay';
        newOverlay.onclick = toggleSidebar;
        document.body.appendChild(newOverlay);
        setTimeout(() => newOverlay.classList.add('active'), 10);
    } else if (overlay) {
        overlay.classList.remove('active');
        setTimeout(() => overlay.remove(), 300);
    }
}

// Close sidebar when clicking on links (mobile)
document.addEventListener('DOMContentLoaded', function() {
    const sidebarLinks = document.querySelectorAll('.admin-sidebar a');
    sidebarLinks.forEach(link => {
        link.addEventListener('click', function() {
            if (window.innerWidth <= 768) {
                toggleSidebar();
            }
        });
    });
});
</script>