<?php
/** cms-core/public/webview/admin/integrations.php - MIT LIZENZ & UPDATES */

require_once __DIR__ . '/../../../backend/authorization/check_admin.php';
require_once __DIR__ . '/../../../backend/integrations/integration_helper.php';

$errors = [];

// Handle form submissions
if (is_post() && isset($_POST['action'])) {
    try {
        CSRF::require();
    } catch (Exception $e) {
        set_flash('error', 'Ungültiger CSRF-Token oder Serverfehler.');
        error_log("CSRF validation failed: " . $e->getMessage());
        redirect('/admin/integrations.php');
        exit;
    }
    
    $action = post('action');
    
    // ============================================
    // LICENSE ACTIONS
    // ============================================
    if ($action === 'activate_license') {
        $licenseKey = trim(post('license_key', ''));
        
        if (empty($licenseKey)) {
            set_flash('error', 'Bitte gib einen Lizenz-Schlüssel ein.');
        } elseif (!preg_match('/^MANI-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$/', $licenseKey)) {
            set_flash('error', 'Ungültiges Lizenz-Format. Erwartet: MANI-XXXX-XXXX-XXXX-XXXX');
        } else {
            // Verify license with API
            $ch = curl_init(LICENSE_API_URL . '/verify-license');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
                'license_key' => $licenseKey,
                'domain' => parse_url(APP_URL, PHP_URL_HOST),
                'email' => get_setting('site_email')
            ]));
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode === 200 && $response) {
                $result = json_decode($response, true);
                if ($result && $result['valid'] === true) {
                    // Update .env file
                    $envPath = APP_ROOT . '/.env';
                    $envContent = file_get_contents($envPath);
                    
                    // Update or add LICENSE_KEY
                    if (strpos($envContent, 'LICENSE_KEY=') !== false) {
                        $envContent = preg_replace('/LICENSE_KEY=.*/', "LICENSE_KEY={$licenseKey}", $envContent);
                    } else {
                        $envContent .= "\nLICENSE_KEY={$licenseKey}\n";
                    }
                    
                    // Update LICENSE_VALID
                    if (strpos($envContent, 'LICENSE_VALID=') !== false) {
                        $envContent = preg_replace('/LICENSE_VALID=.*/', 'LICENSE_VALID=true', $envContent);
                    } else {
                        $envContent .= "LICENSE_VALID=true\n";
                    }
                    
                    // Update LICENSE_ACTIVATED
                    $activatedTime = date('Y-m-d H:i:s');
                    if (strpos($envContent, 'LICENSE_ACTIVATED=') !== false) {
                        $envContent = preg_replace('/LICENSE_ACTIVATED=.*/', "LICENSE_ACTIVATED={$activatedTime}", $envContent);
                    } else {
                        $envContent .= "LICENSE_ACTIVATED={$activatedTime}\n";
                    }
                    
                    file_put_contents($envPath, $envContent);
                    
                    set_flash('success', '✅ Lizenz erfolgreich aktiviert! Updates sind jetzt verfügbar.');
                } else {
                    set_flash('error', 'Lizenz-Validierung fehlgeschlagen: ' . ($result['message'] ?? 'Unbekannter Fehler'));
                }
            } else {
                set_flash('error', 'Lizenz-Server nicht erreichbar. Bitte versuche es später erneut.');
            }
        }
        
        redirect('/admin/integrations.php');
        exit;
    }
    
    if ($action === 'check_update') {
        if (!LICENSE_VALID || empty(LICENSE_KEY)) {
            set_flash('error', 'Keine gültige Lizenz vorhanden. Bitte aktiviere zuerst eine Lizenz.');
            redirect('/admin/integrations.php');
            exit;
        }
        
        // Check for updates
        $ch = curl_init(LICENSE_API_URL . '/check-update');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
            'license_key' => LICENSE_KEY,
            'current_version' => '1.0.0', // TODO: Make this dynamic from version file
            'domain' => parse_url(APP_URL, PHP_URL_HOST)
        ]));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode === 200 && $response) {
            $result = json_decode($response, true);
            if ($result && isset($result['update_available'])) {
                if ($result['update_available']) {
                    set_flash('info', '🎉 Update verfügbar: Version ' . $result['latest_version'] . ' - <a href="' . LICENSE_API_URL . '/download-update?license=' . LICENSE_KEY . '" style="color: white; text-decoration: underline;">Jetzt herunterladen</a>');
                } else {
                    set_flash('success', '✅ Du hast bereits die neueste Version!');
                }
            }
        } else {
            set_flash('error', 'Update-Check fehlgeschlagen. Server nicht erreichbar.');
        }
        
        redirect('/admin/integrations.php');
        exit;
    }
    
    // ============================================
    // INTEGRATION ACTIONS
    // ============================================
    
    // Handle connection test
    if ($action === 'test_connection') {
        $integration = post('integration');
        $result = IntegrationHelper::testConnection($integration);
        
        if ($result['success']) {
            set_flash('success', $result['message']);
        } else {
            set_flash('error', $result['message']);
        }
        
        redirect('/admin/integrations.php');
        exit;
    }
    
    // Handle cookie_consent toggle
    if ($action === 'toggle_cookie_consent') {
        $enabled = isset($_POST['cookie_consent_enabled']) && $_POST['cookie_consent_enabled'] === '1';
        
        set_setting('cookie_consent_enabled', $enabled, 'boolean');
        
        if ($enabled) {
            set_flash('success', 'Cookie-Banner aktiviert.');
        } else {
            set_flash('success', 'Cookie-Banner deaktiviert.');
        }
        
        redirect('/admin/integrations.php');
        exit;
    }
    
    // Handle WebP toggle
    if ($action === 'toggle_webp') {
        $currentStatus = get_setting('auto_convert_webp', true);
        set_setting('auto_convert_webp', !$currentStatus, 'boolean');
        
        if (!$currentStatus) {
            set_flash('success', 'WebP Auto-Konvertierung aktiviert. Neue Uploads werden automatisch optimiert.');
        } else {
            set_flash('success', 'WebP Auto-Konvertierung deaktiviert.');
        }
        
        redirect('/admin/integrations.php');
        exit;
    }
    
    // Handle WebP config save
    if ($action === 'save_webp_config') {
        $maxWidth = (int)post('max_image_width', 2000);
        $quality = (int)post('webp_quality', 85);
        $preferWebp = isset($_POST['prefer_webp']) && $_POST['prefer_webp'] === '1';
        
        // Validate
        if ($maxWidth < 800 || $maxWidth > 4000) {
            set_flash('error', 'Ungültige Bildbreite. Erlaubt: 800-4000px');
            redirect('/admin/integrations.php');
            exit;
        }
        
        if ($quality < 60 || $quality > 100) {
            set_flash('error', 'Ungültige Qualität. Erlaubt: 60-100%');
            redirect('/admin/integrations.php');
            exit;
        }
        
        set_setting('max_image_width', $maxWidth, 'text');
        set_setting('webp_quality', $quality, 'text');
        set_setting('prefer_webp', $preferWebp, 'boolean');
        
        set_flash('success', 'WebP-Einstellungen gespeichert.');
        redirect('/admin/integrations.php');
        exit;
    }
    
    $integration = post('integration');
    
    if ($action === 'toggle') {
        $currentStatus = IntegrationHelper::isActive($integration);
        
        if ($currentStatus) {
            IntegrationHelper::disable($integration);
            set_flash('success', 'Integration deaktiviert.');
        } else {
            IntegrationHelper::enable($integration);
            set_flash('success', 'Integration aktiviert.');
        }
        
        redirect('/admin/integrations.php');
        exit;
    }
    
    if ($action === 'save_config') {
        $config = $_POST['config'] ?? [];
        
        // Handle checkbox for cdn_media_only
        if ($integration === 'cloudflare_cdn') {
            $config['cdn_media_only'] = isset($_POST['config']['cdn_media_only']) && $_POST['config']['cdn_media_only'] === '1';
        }
        
        IntegrationHelper::saveConfig($integration, $config);
        set_flash('success', 'Konfiguration gespeichert.');
        redirect('/admin/integrations.php');
        exit;
    }
}

// Get license status
$hasLicense = !empty(LICENSE_KEY);
$licenseValid = LICENSE_VALID;

// Hole verfügbare Integrationen
$availableIntegrations = IntegrationHelper::getAvailable();

// Sort integrations by category priority
$categoryOrder = ['tracking', 'marketing', 'cdn', 'performance'];
$sortedIntegrations = [];

// First, group by category
$grouped = [];
foreach ($availableIntegrations as $key => $integration) {
    $category = $integration['category'] ?? 'other';
    $grouped[$category][$key] = $integration;
}

// Then sort by priority
foreach ($categoryOrder as $category) {
    if (isset($grouped[$category])) {
        $sortedIntegrations = array_merge($sortedIntegrations, $grouped[$category]);
        unset($grouped[$category]);
    }
}

// Add remaining categories
foreach ($grouped as $category => $integrations) {
    $sortedIntegrations = array_merge($sortedIntegrations, $integrations);
}

$flashMessages = get_flash();
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Integrationen</title>
<?php
$adminCssPath = APP_ROOT . '/assets/css/admin.css';
$adminCssVersion = file_exists($adminCssPath) ? filemtime($adminCssPath) : time();
?>
<link rel="stylesheet" href="/assets/css/admin.css?v=<?= $adminCssVersion ?>">
</head>
<body>
    <?php include 'includes/admin_header.php'; ?>
    <?php include 'includes/maintenance_banner.php'; ?>
    
    <div class="admin-container">
        <?php include 'includes/admin_sidebar.php'; ?>
        
        <main class="admin-content">
            <?php foreach ($flashMessages as $flash): ?>
                <div class="alert alert-<?= e($flash['type']) ?>">
                    <?= $flash['message'] ?> <!-- Allow HTML for update download link -->
                </div>
            <?php endforeach; ?>
            
            <div class="page-header" style="display: flex; justify-content: space-between; align-items: center;">
                <div>
                    <h1>🔌 Integrationen</h1>
                    <p>Externe Dienste und Tracking-Tools verwalten</p>
                </div>
                <a href="/admin/performance-test.php" class="btn btn-primary" style="white-space: nowrap;">
                    ⚡ Performance-Test & CMS-Vergleich
                </a>
            </div>

            <!-- Integrations Grid -->
            <div class="integrations-grid">
                <?php foreach ($sortedIntegrations as $key => $integration): ?>
                    <?php 
                    $isActive = IntegrationHelper::isActive($key);
                    $config = IntegrationHelper::getConfig($key);
                    $category = $integration['category'] ?? 'other';
                    
                    // Category colors
                    $categoryColors = [
                        'performance' => ['border' => '#8b5cf6', 'bg' => '#f5f3ff'],
                        'cdn' => ['border' => '#10b981', 'bg' => '#f0fdf4'],
                        'tracking' => ['border' => '#3b82f6', 'bg' => '#eff6ff'],
                        'marketing' => ['border' => '#f59e0b', 'bg' => '#fffbeb']
                    ];
                    
                    $colors = $categoryColors[$category] ?? ['border' => '#6b7280', 'bg' => '#f9fafb'];
                    ?>
                    
                    <div class="integration-card <?= $isActive ? 'active' : '' ?>" style="<?= $isActive ? "border-color: {$colors['border']}; background: {$colors['bg']};" : '' ?>">
                        <div class="integration-header">
                            <div class="integration-icon"><?= $integration['icon'] ?></div>
                            <div class="integration-info">
                                <h3><?= e($integration['name']) ?></h3>
                                <p><?= e($integration['description']) ?></p>
                            </div>
                            <div class="integration-toggle">
                                <form method="POST" style="display: inline;">
                                    <?= CSRF::field() ?>
                                    <input type="hidden" name="action" value="toggle">
                                    <input type="hidden" name="integration" value="<?= e($key) ?>">
                                    
                                    <label class="toggle-switch">
                                        <input 
                                            type="checkbox" 
                                            <?= $isActive ? 'checked' : '' ?>
                                            onchange="this.form.submit()"
                                        >
                                        <span class="toggle-slider"></span>
                                    </label>
                                </form>
                            </div>
                        </div>
                        
                        <?php if (!empty($integration['config_fields'])): ?>
                            <div class="integration-config">
                                <form method="POST">
                                    <?= CSRF::field() ?>
                                    <input type="hidden" name="action" value="save_config">
                                    <input type="hidden" name="integration" value="<?= e($key) ?>">
                                    
                                    <?php foreach ($integration['config_fields'] as $fieldKey => $fieldConfig): ?>
                                        <div class="form-group">
                                            <label class="form-label" style="font-size: 13px;">
                                                <?= e($fieldConfig['label'] ?? $fieldConfig) ?>
                                            </label>
                                            
                                            <?php if (isset($fieldConfig['type']) && $fieldConfig['type'] === 'checkbox'): ?>
                                                <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                                                    <input 
                                                        type="checkbox" 
                                                        name="config[<?= e($fieldKey) ?>]" 
                                                        value="1"
                                                        <?= !empty($config[$fieldKey]) ? 'checked' : '' ?>
                                                    >
                                                    <span style="font-size: 13px;"><?= e($fieldConfig['description'] ?? '') ?></span>
                                                </label>
                                            <?php elseif (isset($fieldConfig['type']) && $fieldConfig['type'] === 'number'): ?>
                                                <input 
                                                    type="number" 
                                                    name="config[<?= e($fieldKey) ?>]" 
                                                    class="form-input"
                                                    value="<?= e($config[$fieldKey] ?? $fieldConfig['default'] ?? '') ?>"
                                                    min="<?= e($fieldConfig['min'] ?? '') ?>"
                                                    max="<?= e($fieldConfig['max'] ?? '') ?>"
                                                    placeholder="<?= e($fieldConfig['placeholder'] ?? '') ?>"
                                                >
                                                <?php if (isset($fieldConfig['description'])): ?>
                                                    <small><?= e($fieldConfig['description']) ?></small>
                                                <?php endif; ?>
                                            <?php elseif (isset($fieldConfig['type']) && $fieldConfig['type'] === 'password'): ?>
                                                <input 
                                                    type="password" 
                                                    name="config[<?= e($fieldKey) ?>]" 
                                                    class="form-input"
                                                    value="<?= e($config[$fieldKey] ?? '') ?>"
                                                    placeholder="<?= e(is_array($fieldConfig) ? $fieldConfig['placeholder'] ?? '' : $fieldConfig) ?>"
                                                >
                                                <?php if (is_array($fieldConfig) && isset($fieldConfig['description'])): ?>
                                                    <small><?= e($fieldConfig['description']) ?></small>
                                                <?php endif; ?>
                                            <?php else: ?>
                                                <input 
                                                    type="text" 
                                                    name="config[<?= e($fieldKey) ?>]" 
                                                    class="form-input"
                                                    value="<?= e($config[$fieldKey] ?? '') ?>"
                                                    placeholder="<?= e(is_array($fieldConfig) ? $fieldConfig['placeholder'] ?? '' : $fieldConfig) ?>"
                                                >
                                                <?php if (is_array($fieldConfig) && isset($fieldConfig['description'])): ?>
                                                    <small><?= e($fieldConfig['description']) ?></small>
                                                <?php endif; ?>
                                            <?php endif; ?>
                                        </div>
                                    <?php endforeach; ?>
                                    
                                    <div style="display: flex; gap: 8px;">
                                        <button type="submit" class="btn btn-sm btn-primary" style="flex: 1;">
                                            💾 Speichern
                                        </button>
                                        
                                        <?php if ($category === 'performance'): ?>
                                        <button type="submit" 
                                                name="action" 
                                                value="test_connection" 
                                                class="btn btn-sm" 
                                                style="background: #6b7280; color: white;"
                                                onclick="this.form.querySelector('[name=action][value=save_config]').remove();">
                                            🔍 Test
                                        </button>
                                        <?php endif; ?>
                                    </div>
                                </form>
                            </div>
                        <?php endif; ?>
                        
                        <div class="integration-status">
                            <?php if ($isActive): ?>
                                <span class="badge badge-success">✓ Aktiv</span>
                            <?php else: ?>
                                <span class="badge badge-warning">⏸ Inaktiv</span>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
            
            <!-- LICENSE & UPDATES CARD -->
            <div style="margin-top: 40px; margin-bottom: 30px;">
                <h2 style="font-size: 20px; margin-bottom: 20px; color: #1f2937;">⚙️ System & Updates</h2>
                
                <div class="integration-card" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border: none; color: white;">
                    <div class="integration-header" style="border-bottom-color: rgba(255,255,255,0.2);">
                        <div class="integration-icon" style="background: rgba(255,255,255,0.2); font-size: 50px;">
                            🎫
                        </div>
                        <div class="integration-info">
                            <h3 style="color: white;">Lizenz & Updates</h3>
                            <p style="color: rgba(255,255,255,0.9);">
                                <?php if ($hasLicense && $licenseValid): ?>
                                    ✅ Lizenz aktiv - Updates verfügbar
                                <?php elseif ($hasLicense && !$licenseValid): ?>
                                    ⚠️ Lizenz eingegeben aber nicht validiert
                                <?php else: ?>
                                    Aktiviere eine Lizenz um automatische Updates zu erhalten
                                <?php endif; ?>
                            </p>
                        </div>
                        <?php if ($hasLicense && $licenseValid): ?>
                        <div style="text-align: right;">
                            <span style="background: rgba(255,255,255,0.3); padding: 8px 16px; border-radius: 20px; font-size: 13px; font-weight: 600;">
                                ✓ AKTIVIERT
                            </span>
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <?php if (!$hasLicense || !$licenseValid): ?>
                    <!-- License Activation Form -->
                    <div class="integration-config" style="background: rgba(255,255,255,0.1); padding: 20px; border-radius: 8px; margin-top: 20px;">
                        <form method="POST">
                            <?= CSRF::field() ?>
                            <input type="hidden" name="action" value="activate_license">
                            
                            <div class="form-group" style="margin-bottom: 15px;">
                                <label class="form-label" style="color: white; font-weight: 600;">
                                    🔑 Lizenz-Schlüssel
                                </label>
                                <input 
                                    type="text" 
                                    name="license_key" 
                                    class="form-input"
                                    value="<?= e(LICENSE_KEY ?? '') ?>"
                                    placeholder="MANI-XXXX-XXXX-XXXX-XXXX"
                                    style="font-family: monospace; letter-spacing: 1px;"
                                >
                                <small style="color: rgba(255,255,255,0.8); display: block; margin-top: 8px;">
                                    Hast du noch keine Lizenz? Kontaktiere uns auf <a href="https://manidesign.de" target="_blank" style="color: white; text-decoration: underline;">manidesign.de</a>
                                </small>
                            </div>
                            
                            <button type="submit" class="btn" style="width: 100%; background: white; color: #667eea; font-weight: 600;">
                                ✨ Lizenz aktivieren
                            </button>
                        </form>
                    </div>
                    <?php else: ?>
                    <!-- License Active - Show Update Check -->
                    <div style="background: rgba(255,255,255,0.1); padding: 20px; border-radius: 8px; margin-top: 20px;">
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                            <div>
                                <p style="color: rgba(255,255,255,0.8); font-size: 13px; margin-bottom: 4px;">Lizenz-Key</p>
                                <p style="color: white; font-weight: 600; font-family: monospace; font-size: 14px;"><?= e(LICENSE_KEY) ?></p>
                            </div>
                            <div>
                                <p style="color: rgba(255,255,255,0.8); font-size: 13px; margin-bottom: 4px;">Aktiviert am</p>
                                <p style="color: white; font-weight: 600; font-size: 14px;"><?= LICENSE_ACTIVATED ? date('d.m.Y', strtotime(LICENSE_ACTIVATED)) : 'Unbekannt' ?></p>
                            </div>
                        </div>
                        
                        <form method="POST">
                            <?= CSRF::field() ?>
                            <input type="hidden" name="action" value="check_update">
                            <button type="submit" class="btn" style="width: 100%; background: white; color: #667eea; font-weight: 600;">
                                🔄 Nach Updates suchen
                            </button>
                        </form>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Cookie Consent Settings -->
            <div class="content-card">
                <div class="card-header">
                    <h2 class="card-title">🍪 Cookie-Consent-Einstellungen</h2>
                </div>
                
                <form method="POST">
                    <?= CSRF::field() ?>
                    <input type="hidden" name="action" value="toggle_cookie_consent">
                    
                    <div class="form-group">
                        <label class="form-label">Cookie-Banner aktivieren</label>
                        <p style="font-size: 13px; color: #6b7280; margin-bottom: 10px;">
                            Wenn aktiviert, müssen Besucher ihre Zustimmung erteilen, bevor Tracking-Code geladen wird.
                        </p>
                        <label class="toggle-switch">
                            <input 
                                type="checkbox" 
                                name="cookie_consent_enabled" 
                                value="1"
                                <?= get_setting('cookie_consent_enabled', true) ? 'checked' : '' ?>
                                onchange="this.form.submit()"
                            >
                            <span class="toggle-slider"></span>
                        </label>
                    </div>
                </form>
                
                <div class="alert alert-warning" style="margin-top: 20px;">
                    ⚠️ <strong>Wichtig:</strong> Für DSGVO-Konformität sollte der Cookie-Banner immer aktiviert sein, wenn Sie Tracking-Tools verwenden.
                </div>
            </div>
            
            <!-- Help Section -->
            <div class="content-card">
                <div class="card-header">
                    <h2 class="card-title">❓ Hilfe zu Integrationen</h2>
                </div>
                
                <div style="display: grid; gap: 15px;">
                    <div style="padding: 15px; background: #eff6ff; border-radius: 8px; border-left: 4px solid #3b82f6;">
                        <h4 style="margin-bottom: 8px; color: #1e40af;">📊 Google Analytics</h4>
                        <p style="font-size: 13px; color: #1e40af; margin-bottom: 8px;">
                            <strong>Tracking-ID Format:</strong> <code style="background: white; padding: 2px 6px; border-radius: 4px;">G-XXXXXXXXXX</code>
                        </p>
                        <p style="font-size: 13px; color: #1e40af;">
                            Die ID findest du in deinem Google Analytics Account unter:<br>
                            <strong>Admin → Property → Data Streams</strong>
                        </p>
                    </div>
                    
                    <div style="padding: 15px; background: #fffbeb; border-radius: 8px; border-left: 4px solid #f59e0b;">
                        <h4 style="margin-bottom: 8px; color: #92400e;">🎵 TikTok Pixel</h4>
                        <p style="font-size: 13px; color: #92400e;">
                            Die Pixel-ID findest du im TikTok Ads Manager unter:<br>
                            <strong>Assets → Events → Web Events</strong>
                        </p>
                    </div>
                    
                    <div style="padding: 15px; background: #fffbeb; border-radius: 8px; border-left: 4px solid #f59e0b;">
                        <h4 style="margin-bottom: 8px; color: #92400e;">📷 Meta/Instagram Pixel</h4>
                        <p style="font-size: 13px; color: #92400e;">
                            Die Pixel-ID findest du im Meta Business Manager unter:<br>
                            <strong>Events Manager → Data Sources</strong>
                        </p>
                    </div>
                    
                    <div style="padding: 15px; background: #f0fdf4; border-radius: 8px; border: 2px solid #10b981;">
                        <h4 style="margin-bottom: 8px; color: #065f46;">📧 Mailchimp Newsletter</h4>
                        
                        <p style="font-size: 13px; color: #047857; margin-bottom: 12px;">
                            <strong>📝 API Key finden:</strong><br>
                            1. Gehe zu Mailchimp → Account<br>
                            2. Klicke auf <strong>Extras → API keys</strong><br>
                            3. Klicke auf <strong>"Create A Key"</strong><br>
                            4. Kopiere den API Key (Format: <code style="background: white; padding: 2px 6px; border-radius: 4px;">abc123def456-us19</code>)
                        </p>
                        
                        <p style="font-size: 13px; color: #047857; margin-bottom: 12px;">
                            <strong>📋 List ID (Audience ID) finden:</strong><br>
                            1. Gehe zu <strong>Audience → All contacts</strong><br>
                            2. Klicke auf <strong>Settings</strong> (oben rechts)<br>
                            3. Wähle <strong>Audience name and defaults</strong><br>
                            4. Kopiere die <strong>Audience ID</strong> (z.B. <code style="background: white; padding: 2px 6px; border-radius: 4px;">a1b2c3d4e5</code>)
                        </p>
                        
                        <div style="background: white; padding: 12px; border-radius: 6px; margin-top: 12px; border-left: 4px solid #10b981;">
                            <p style="font-size: 13px; color: #065f46; margin: 0;">
                                <strong>✅ Verwendung:</strong> Füge den <strong>"Newsletter Anmeldung"</strong> Block zu deiner Seite hinzu. Besucher werden automatisch zu deiner Mailchimp-Liste hinzugefügt (Double Opt-in für DSGVO-Konformität).
                            </p>
                        </div>
                    </div>
                    
                    <div style="padding: 15px; background: #f0fdf4; border-radius: 8px; border-left: 4px solid #10b981;">
                        <h4 style="margin-bottom: 8px; color: #065f46;">🌐 Cloudflare CDN</h4>
                        <p style="font-size: 13px; color: #047857; margin-bottom: 8px;">
                            <strong>Setup:</strong>
                        </p>
                        <ol style="margin: 0 0 0 20px; font-size: 13px; color: #065f46; line-height: 1.6;">
                            <li>Erstelle kostenlosen Account auf <a href="https://dash.cloudflare.com/sign-up" target="_blank">cloudflare.com</a></li>
                            <li>Füge deine Domain hinzu</li>
                            <li>Ändere Nameserver bei deinem Provider (z.B. IONOS)</li>
                            <li>Warte 24h auf DNS-Propagierung</li>
                            <li>Aktiviere CDN im CMS - fertig!</li>
                        </ol>
                    </div>
                    
                    <div style="padding: 15px; background: #f5f3ff; border-radius: 8px; border-left: 4px solid #8b5cf6;">
                        <h4 style="margin-bottom: 8px; color: #5b21b6;">⚡ Performance-Caching</h4>
                        <p style="font-size: 13px; color: #6b21a8; margin-bottom: 8px;">
                            <strong>Redis / Memcached:</strong> Deutlich schneller als Datenbank-Queries. Ideal für High-Traffic Websites.
                        </p>
                        <p style="font-size: 13px; color: #6b21a8;">
                            <strong>Setup:</strong> Muss auf dem Server installiert sein. Bei Shared Hosting meist nicht verfügbar. Bei VPS/Dedicated Server via SSH installierbar.
                        </p>
                    </div>
                    
                    <div style="padding: 15px; background: #fef3c7; border-radius: 8px; border-left: 4px solid #f59e0b;">
                        <h4 style="margin-bottom: 8px; color: #92400e;">🖼️ WebP Optimierung</h4>
                        <p style="font-size: 13px; color: #92400e;">
                            <strong>Funktionsweise:</strong> Bei jedem Bild-Upload wird automatisch eine WebP-Version erstellt. Das Original bleibt als Fallback für ältere Browser erhalten. Moderne Browser laden automatisch die kleinere WebP-Version.
                        </p>
                    </div>
                </div>
            </div>
        </main>
    </div>
    
    <style>
    .integrations-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
        gap: 20px;
        margin-bottom: 30px;
    }
    
    .integration-card {
        background: white;
        border-radius: 12px;
        padding: 24px;
        box-shadow: 0 1px 3px rgba(0,0,0,0.08);
        border: 2px solid #e5e7eb;
        transition: all 0.2s;
    }
    
    .integration-card.active {
        box-shadow: 0 4px 12px rgba(102, 126, 234, 0.15);
    }
    
    .integration-header {
        display: grid;
        grid-template-columns: auto 1fr auto;
        gap: 15px;
        align-items: start;
        margin-bottom: 20px;
        padding-bottom: 20px;
        border-bottom: 1px solid #e5e7eb;
    }
    
    .integration-icon {
        font-size: 40px;
        width: 60px;
        height: 60px;
        display: flex;
        align-items: center;
        justify-content: center;
        background: #f3f4f6;
        border-radius: 12px;
    }
    
    .integration-info h3 {
        font-size: 18px;
        font-weight: 600;
        margin-bottom: 4px;
        color: #1f2937;
    }
    
    .integration-info p {
        font-size: 13px;
        color: #6b7280;
        margin: 0;
    }
    
    .integration-config {
        margin-bottom: 15px;
    }
    
    .integration-status {
        display: flex;
        justify-content: flex-end;
    }
    
    .toggle-switch {
        position: relative;
        display: inline-block;
        width: 50px;
        height: 28px;
    }
    
    .toggle-switch input {
        opacity: 0;
        width: 0;
        height: 0;
    }
    
    .toggle-slider {
        position: absolute;
        cursor: pointer;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background-color: #d1d5db;
        transition: 0.3s;
        border-radius: 28px;
    }
    
    .toggle-slider:before {
        position: absolute;
        content: "";
        height: 20px;
        width: 20px;
        left: 4px;
        bottom: 4px;
        background-color: white;
        transition: 0.3s;
        border-radius: 50%;
    }
    
    input:checked + .toggle-slider {
        background-color: #667eea;
    }
    
    input:checked + .toggle-slider:before {
        transform: translateX(22px);
    }
    
    /* Responsive */
    @media (max-width: 768px) {
        .integrations-grid {
            grid-template-columns: 1fr;
        }
        
        .integration-header {
            grid-template-columns: auto 1fr;
            gap: 12px;
        }
        
        .integration-toggle {
            grid-column: 1 / -1;
            justify-self: end;
        }
        
        .page-header {
            flex-direction: column;
            align-items: flex-start !important;
            gap: 15px;
        }
        
        .page-header .btn {
            width: 100%;
        }
    }
    </style>
    
    <script src="/assets/js/main.js"></script>
</body>
</html>