<?php
/** cms-core/public/webview/admin/login.php */

// CRITICAL: ZERO output before requiring init.php
require_once __DIR__ . '/../../../backend/core/init.php';
require_once __DIR__ . '/../../../backend/auth/login.php';

// Generate CSRF token
CSRF::generateToken();

// Get site name
$siteName = get_setting('site_name', 'Ihre Website');

// ============================================
// Process Login FIRST (before any HTML)
// ============================================

// Check if already logged in
if (LoginHandler::isLoggedIn()) {
    header("Location: /admin/dashboard.php", true, 302);
    exit;
}

// Initialize variables
$error = '';
$success = '';

// Check for messages
if (isset($_GET['logged_out'])) {
    $success = 'Sie wurden erfolgreich abgemeldet.';
}

if (isset($_GET['reset_success'])) {
    $success = 'Ihr Passwort wurde erfolgreich zurückgesetzt!';
}

// Process login submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = $_POST['username'] ?? '';
    $password = $_POST['password'] ?? '';
    
    $result = LoginHandler::process($username, $password);
    
    if ($result['success']) {
        // Get redirect path
        $redirectPath = $_SESSION['redirect_after_login'] ?? '/admin/dashboard.php';
        unset($_SESSION['redirect_after_login']);
        
        // Regenerate CSRF
        CSRF::regenerate();
        
        // Redirect
        header("Location: " . $redirectPath, true, 302);
        exit;
    } else {
        $error = $result['error'];
        CSRF::regenerate();
    }
}

// NOW output HTML
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Login - <?= htmlspecialchars($siteName) ?></title>
    
    <!-- CRITICAL: Define CSS Variables inline from farb_font database -->
    <style>
    :root {
        /* Load colors directly from farb_font database */
        --global-header-bg-color: <?= get_setting('header_bg_color', '#000000') ?>;
        --global-header-text-color: <?= get_setting('header_text_color', '#ffffff') ?>;
        --global-header-heading-color: <?= get_setting('header_heading_color', '#000000') ?>;
        --global-button-bg-color: <?= get_setting('button_bg_color', '#000000') ?>;
        --global-button-text-color: <?= get_setting('button_text_color', '#ffffff') ?>;
        --global-link-color: <?= get_setting('link_color', '#667eea') ?>;
        --global-link-hover-color: <?= get_setting('link_hover_color', '#5568d3') ?>;
    }
    
    * {
        margin: 0;
        padding: 0;
        box-sizing: border-box;
    }
    
    html {
        background: var(--global-header-bg-color, #000000);
    }

    body {
        background: transparent;
        font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
        min-height: 100vh;
        display: flex;
        align-items: center;
        justify-content: center;
        padding: 20px;
    }
    
    .login-container {
        background: white;
        padding: 50px 40px;
        border-radius: 20px;
        box-shadow: 0 20px 60px rgba(0,0,0,0.3);
        max-width: 400px;
        width: 100%;
    }
    
    .logo {
        text-align: center;
        margin-bottom: 30px;
    }
    
    .logo h1 {
        color: var(--global-header-heading-color, #000000);
        font-size: 32px;
        margin-bottom: 5px;
    }
    
    .logo p {
        color: #666;
        font-size: 14px;
    }
    
    .error-message {
        background: #fee;
        color: #c33;
        padding: 12px 16px;
        border-radius: 8px;
        margin-bottom: 20px;
        border-left: 4px solid #c33;
    }
    
    .success-message {
        background: #efe;
        color: #363;
        padding: 12px 16px;
        border-radius: 8px;
        margin-bottom: 20px;
        border-left: 4px solid #363;
    }
    
    .form-group {
        margin-bottom: 20px;
    }
    
    .form-group label {
        display: block;
        color: #555;
        font-weight: 600;
        margin-bottom: 8px;
        font-size: 14px;
    }
    
    .form-group input {
        width: 100%;
        padding: 12px 16px;
        border: 2px solid #ddd;
        border-radius: 8px;
        font-size: 16px;
        transition: border 0.3s;
    }
    
    .form-group input:focus {
        outline: none;
        border-color: var(--global-button-bg-color, #000000);
    }
    
    .btn-login {
        width: 100%;
        padding: 14px;
        border: none;
        border-radius: 8px;
        font-size: 16px;
        font-weight: 600;
        cursor: pointer;
        transition: transform 0.2s, opacity 0.2s;
        background: var(--global-button-bg-color, #000000) !important;
        color: var(--global-button-text-color, #ffffff) !important;
    }
    
    .btn-login:hover {
        transform: translateY(-2px);
        opacity: 0.9;
    }
    
    .forgot-password {
        text-align: center;
        margin-top: 20px;
    }
    
    .forgot-password a {
        text-decoration: none;
        font-size: 14px;
        color: var(--global-link-color, #667eea);
    }
    
    .forgot-password a:hover {
        text-decoration: underline;
        color: var(--global-link-hover-color, #5568d3);
    }
    </style>
</head>
<body>
    <!-- ... rest bleibt gleich ... -->
<body>
    <div class="login-container">
        <div class="logo">
            <h1><?= htmlspecialchars($siteName) ?></h1>
            <p>Admin-Bereich</p>
        </div>
        
        <?php if ($error): ?>
            <div class="error-message">
                <?= htmlspecialchars($error) ?>
            </div>
        <?php endif; ?>
        
        <?php if ($success): ?>
            <div class="success-message">
                <?= htmlspecialchars($success) ?>
            </div>
        <?php endif; ?>
        
        <form method="POST">
            <?= CSRF::field(); ?>
            
            <div class="form-group">
                <label for="username">Benutzername</label>
                <input 
                    type="text" 
                    id="username" 
                    name="username" 
                    required 
                    autofocus
                    autocomplete="username"
                >
            </div>
            
            <div class="form-group">
                <label for="password">Passwort</label>
                <input 
                    type="password" 
                    id="password" 
                    name="password" 
                    required
                    autocomplete="current-password"
                >
            </div>
            
            <button type="submit" class="btn-login">
                Anmelden
            </button>
        </form>
        
        <div class="forgot-password">
            <a href="/admin/forgot_password.php">🔑 Passwort vergessen?</a>
        </div>
    </div>
</body>
</html>