<?php
/** cms-core/public/webview/admin/media_manage.php - COMPLETE WITH DELETE & EDIT */

require_once __DIR__ . '/../../../backend/authorization/check_admin.php';

$pageBrowserTitle = 'Media Bibliothek';

// Handle DELETE action
if (is_post() && post('action') === 'delete') {
    CSRF::require();
    
    $filePath = post('file_path');
    
    // Security check: Ensure file is within MEDIA_ROOT
    $realPath = realpath($filePath);
    $mediaRoot = realpath(MEDIA_ROOT);
    
    if ($realPath && $mediaRoot && strpos($realPath, $mediaRoot) === 0) {
        if (file_exists($filePath) && is_file($filePath)) {
            if (unlink($filePath)) {
                set_flash('success', 'Datei erfolgreich gelöscht!');
            } else {
                set_flash('error', 'Fehler beim Löschen der Datei.');
            }
        } else {
            set_flash('error', 'Datei nicht gefunden.');
        }
    } else {
        set_flash('error', 'Ungültiger Dateipfad.');
    }
    
    redirect('/admin/media_manage.php?folder=' . get_param('folder', 'pages'));
    exit;
}

// Define media folders
$mediaFolders = [
    'logo' => ['name' => 'Logo', 'icon' => '🏢', 'path' => MEDIA_ROOT . '/logo'],
    'pages' => ['name' => 'Seiten', 'icon' => '📄', 'path' => MEDIA_ROOT . '/pages'],
    'blog' => ['name' => 'Blog', 'icon' => '📝', 'path' => MEDIA_ROOT . '/blog'],
    'favicon' => ['name' => 'Favicon', 'icon' => '⭐', 'path' => MEDIA_ROOT . '/favicon'],
    'images' => ['name' => 'Bilder', 'icon' => '🖼️', 'path' => MEDIA_ROOT . '/images']
];

// Get active folder from URL
$activeFolder = $_GET['folder'] ?? 'pages';
if (!isset($mediaFolders[$activeFolder])) {
    $activeFolder = 'pages';
}

// Get files from active folder
$files = [];
$folderPath = $mediaFolders[$activeFolder]['path'];

if (is_dir($folderPath)) {
    $scanFiles = array_diff(scandir($folderPath), ['.', '..']);
    
    foreach ($scanFiles as $file) {
        $filePath = $folderPath . '/' . $file;
        
        if (is_file($filePath)) {
            $ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));
            $isImage = in_array($ext, ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg']);
            
            // URL for viewing via media.php handler (internal)
            $viewUrl = '/cms-core/public/webview/media.php?file=' . $activeFolder . '/' . $file;
            
            // Clean URL for copying/sharing (external)
            $copyUrl = '/media/' . $activeFolder . '/' . $file;
            
            $files[] = [
                'name' => $file,
                'path' => $filePath,
                'view_url' => $viewUrl,  // For displaying in admin
                'copy_url' => $copyUrl,  // For copying/sharing
                'size' => filesize($filePath),
                'modified' => filemtime($filePath),
                'is_image' => $isImage,
                'ext' => $ext
            ];
        }
    }
    
    // Sort by modified date (newest first)
    usort($files, function($a, $b) {
        return $b['modified'] - $a['modified'];
    });
}

// Calculate total size
$totalSize = array_sum(array_column($files, 'size'));
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= e($pageBrowserTitle) ?></title>
    <?php
    $adminCssPath = APP_ROOT . '/assets/css/admin.css';
    $adminCssVersion = file_exists($adminCssPath) ? filemtime($adminCssPath) : time();
    ?>
    <link rel="stylesheet" href="/assets/css/admin.css?v=<?= $adminCssVersion ?>">
</head>
<body>
    <?php include 'includes/admin_header.php'; ?>
    
    <div class="admin-container">
        <?php include 'includes/admin_sidebar.php'; ?>
        
        <main class="admin-content">
            <div class="page-header">
                <div>
                    <h1>📁 Media Bibliothek</h1>
                    <p class="page-description">Verwalten Sie Ihre Bilder und Dateien</p>
                </div>
            </div>
            
            <!-- Flash Messages -->
            <?php if (has_flash()): ?>
                <?php foreach (get_flash() as $flash): ?>
                <div class="alert alert-<?= $flash['type'] ?>" style="margin-bottom: 20px;">
                    <?= e($flash['message']) ?>
                </div>
                <?php endforeach; ?>
            <?php endif; ?>
            
            <!-- Folder Tabs -->
            <div class="folder-tabs">
                <?php foreach ($mediaFolders as $key => $folder): ?>
                    <a href="?folder=<?= $key ?>" 
                       class="folder-tab <?= $activeFolder === $key ? 'active' : '' ?>">
                        <span class="folder-icon"><?= $folder['icon'] ?></span>
                        <span class="folder-name"><?= $folder['name'] ?></span>
                        <span class="folder-count">
                            <?php
                            if (is_dir($folder['path'])) {
                                $count = count(array_diff(scandir($folder['path']), ['.', '..']));
                                echo $count;
                            } else {
                                echo '0';
                            }
                            ?>
                        </span>
                    </a>
                <?php endforeach; ?>
            </div>
            
            <!-- Stats Bar -->
            <div class="media-stats">
                <div class="stat-item">
                    <span class="stat-label">Dateien:</span>
                    <span class="stat-value"><?= count($files) ?></span>
                </div>
                <div class="stat-item">
                    <span class="stat-label">Größe:</span>
                    <span class="stat-value"><?= number_format($totalSize / 1024 / 1024, 2) ?> MB</span>
                </div>
                <div class="stat-item">
                    <span class="stat-label">Ordner:</span>
                    <span class="stat-value"><?= $mediaFolders[$activeFolder]['name'] ?></span>
                </div>
            </div>
            
            <!-- Files Grid -->
            <?php if (empty($files)): ?>
                <div class="empty-state">
                    <div class="empty-icon">📁</div>
                    <h3>Keine Dateien</h3>
                    <p>Dieser Ordner ist leer</p>
                </div>
            <?php else: ?>
                <div class="media-grid">
                    <?php foreach ($files as $file): ?>
                        <div class="media-item <?= $file['is_image'] ? 'is-image' : 'is-file' ?>">
                            <?php if ($file['is_image']): ?>
                                <div class="media-preview">
                                    <img src="<?= e($file['view_url']) ?>" 
                                         alt="<?= e($file['name']) ?>"
                                         loading="lazy"
                                         onerror="this.parentElement.innerHTML='<div style=\'color:red;padding:20px;text-align:center;font-size:12px;\'>❌ Fehler<br><?= e($file['name']) ?></div>'">
                                </div>
                            <?php else: ?>
                                <div class="media-preview media-preview-file">
                                    <span class="file-ext"><?= strtoupper($file['ext']) ?></span>
                                </div>
                            <?php endif; ?>
                            
                            <div class="media-info">
                                <div class="media-name" title="<?= e($file['name']) ?>">
                                    <?= e($file['name']) ?>
                                </div>
                                <div class="media-meta">
                                    <span><?= number_format($file['size'] / 1024, 1) ?> KB</span>
                                    <span>•</span>
                                    <span><?= date('d.m.Y', $file['modified']) ?></span>
                                </div>
                            </div>
                            
                            <div class="media-actions">
                                <button onclick="viewFile('<?= e($file['view_url']) ?>', '<?= e($file['name']) ?>', <?= $file['is_image'] ? 'true' : 'false' ?>)" 
                                        class="btn-icon" 
                                        title="Ansehen">
                                    👁️
                                </button>
                                <button onclick="copyUrl('<?= e($file['copy_url']) ?>')" 
                                        class="btn-icon" 
                                        title="URL kopieren">
                                    📋
                                </button>
                                <button onclick="deleteFile('<?= e($file['path']) ?>', '<?= e($file['name']) ?>')" 
                                        class="btn-icon btn-danger" 
                                        title="Löschen">
                                    🗑️
                                </button>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </main>
    </div>

<!-- Delete Form (hidden) -->
<form method="POST" id="deleteFileForm" style="display: none;">
    <?= CSRF::field() ?>
    <input type="hidden" name="action" value="delete">
    <input type="hidden" name="file_path" id="delete_file_path">
</form>

<!-- File Viewer Modal -->
<div id="fileViewerModal" class="modal" style="display: none;">
    <div class="modal-backdrop" onclick="closeViewer()"></div>
    <div class="modal-content modal-large">
        <div class="modal-header">
            <h2 id="viewerFileName">Datei Vorschau</h2>
            <button onclick="closeViewer()" class="modal-close">&times;</button>
        </div>
        <div class="modal-body" style="text-align: center; padding: 20px;">
            <div id="fileViewerContent"></div>
        </div>
        <div class="modal-footer">
            <a id="viewerDownloadLink" href="#" download class="btn btn-secondary">📥 Download</a>
            <button onclick="closeViewer()" class="btn btn-primary">Schließen</button>
        </div>
    </div>
</div>

<style>
/* Alert Styles */
.alert {
    padding: 15px 20px;
    border-radius: 8px;
    margin-bottom: 20px;
    display: flex;
    align-items: center;
    gap: 10px;
}
.alert-success {
    background: #d1fae5;
    color: #065f46;
    border: 1px solid #6ee7b7;
}
.alert-error {
    background: #fee2e2;
    color: #991b1b;
    border: 1px solid #fca5a5;
}

/* Folder Tabs */
.folder-tabs {
    display: flex;
    gap: 8px;
    margin-bottom: 24px;
    border-bottom: 2px solid #e5e7eb;
    overflow-x: auto;
    padding-bottom: 8px;
}

.folder-tab {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    border-radius: 8px 8px 0 0;
    text-decoration: none;
    color: #6b7280;
    font-weight: 500;
    transition: all 0.2s;
    white-space: nowrap;
    background: transparent;
    border: 2px solid transparent;
    border-bottom: none;
}

.folder-tab:hover {
    background: #f3f4f6;
    color: #374151;
}

.folder-tab.active {
    background: white;
    color: #667eea;
    border-color: #e5e7eb;
    border-bottom-color: white;
    margin-bottom: -2px;
}

.folder-icon {
    font-size: 20px;
}

.folder-name {
    font-size: 15px;
}

.folder-count {
    background: #e5e7eb;
    color: #6b7280;
    padding: 2px 8px;
    border-radius: 12px;
    font-size: 12px;
    font-weight: 600;
}

.folder-tab.active .folder-count {
    background: #667eea;
    color: white;
}

/* Media Stats */
.media-stats {
    display: flex;
    gap: 24px;
    padding: 16px 20px;
    background: white;
    border-radius: 8px;
    margin-bottom: 24px;
    border: 1px solid #e5e7eb;
}

.stat-item {
    display: flex;
    align-items: center;
    gap: 8px;
}

.stat-label {
    color: #6b7280;
    font-size: 14px;
}

.stat-value {
    color: #111827;
    font-weight: 600;
    font-size: 16px;
}

/* Media Grid */
.media-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
    gap: 20px;
}

.media-item {
    background: white;
    border: 2px solid #e5e7eb;
    border-radius: 12px;
    overflow: hidden;
    transition: all 0.3s;
}

.media-item:hover {
    transform: translateY(-4px);
    box-shadow: 0 8px 24px rgba(0,0,0,0.1);
    border-color: #667eea;
}

.media-preview {
    width: 100%;
    height: 200px;
    background: #f3f4f6;
    display: flex;
    align-items: center;
    justify-content: center;
    overflow: hidden;
}

.media-preview img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.media-preview-file {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
}

.file-ext {
    font-size: 32px;
    font-weight: bold;
    color: white;
}

.media-info {
    padding: 12px;
}

.media-name {
    font-weight: 600;
    font-size: 14px;
    color: #111827;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
    margin-bottom: 4px;
}

.media-meta {
    font-size: 12px;
    color: #9ca3af;
    display: flex;
    gap: 6px;
}

.media-actions {
    display: flex;
    gap: 8px;
    padding: 12px;
    border-top: 1px solid #e5e7eb;
    background: #f9fafb;
}

.btn-icon {
    flex: 1;
    padding: 8px;
    background: white;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    cursor: pointer;
    transition: all 0.2s;
    font-size: 16px;
    text-decoration: none;
    text-align: center;
}

.btn-icon:hover {
    background: #667eea;
    border-color: #667eea;
    color: white;
    transform: scale(1.05);
}

.btn-icon.btn-danger:hover {
    background: #ef4444;
    border-color: #ef4444;
    color: white;
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 80px 20px;
    color: #9ca3af;
}

.empty-icon {
    font-size: 64px;
    margin-bottom: 16px;
    opacity: 0.5;
}

.empty-state h3 {
    font-size: 20px;
    color: #6b7280;
    margin-bottom: 8px;
}

.empty-state p {
    font-size: 14px;
}

/* Modal Styles */
.modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    z-index: 9999;
    display: flex;
    align-items: center;
    justify-content: center;
}

.modal-backdrop {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0,0,0,0.7);
}

.modal-content {
    position: relative;
    background: white;
    border-radius: 12px;
    max-width: 600px;
    width: 90%;
    max-height: 90vh;
    overflow: hidden;
    z-index: 10000;
}

.modal-large {
    max-width: 1000px;
}

.modal-header {
    padding: 20px;
    border-bottom: 1px solid #e5e7eb;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.modal-header h2 {
    margin: 0;
    font-size: 20px;
}

.modal-close {
    background: none;
    border: none;
    font-size: 32px;
    cursor: pointer;
    color: #6b7280;
    padding: 0;
    width: 40px;
    height: 40px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 6px;
    transition: all 0.2s;
}

.modal-close:hover {
    background: #f3f4f6;
    color: #1f2937;
}

.modal-body {
    padding: 20px;
    max-height: calc(90vh - 140px);
    overflow-y: auto;
}

.modal-footer {
    padding: 20px;
    border-top: 1px solid #e5e7eb;
    display: flex;
    gap: 10px;
    justify-content: flex-end;
}

/* Responsive */
@media (max-width: 768px) {
    .media-grid {
        grid-template-columns: repeat(auto-fill, minmax(150px, 1fr));
        gap: 12px;
    }
    
    .folder-tabs {
        flex-wrap: nowrap;
        overflow-x: auto;
    }
    
    .media-stats {
        flex-direction: column;
        gap: 12px;
    }
    
    .modal-content {
        width: 95%;
    }
}
</style>

<script src="/assets/js/main.js"></script>
<script>
// View file in modal
function viewFile(url, fileName, isImage) {
    const modal = document.getElementById('fileViewerModal');
    const content = document.getElementById('fileViewerContent');
    const fileNameEl = document.getElementById('viewerFileName');
    const downloadLink = document.getElementById('viewerDownloadLink');
    
    fileNameEl.textContent = fileName;
    downloadLink.href = url;
    downloadLink.download = fileName;
    
    if (isImage) {
        content.innerHTML = `<img src="${url}" alt="${fileName}" style="max-width: 100%; max-height: 70vh; border-radius: 8px;">`;
    } else {
        content.innerHTML = `
            <div style="padding: 40px; text-align: center;">
                <div style="font-size: 64px; margin-bottom: 20px;">📄</div>
                <h3>${fileName}</h3>
                <p style="color: #6b7280; margin-top: 10px;">Diese Datei kann nicht in der Vorschau angezeigt werden.</p>
                <a href="${url}" target="_blank" class="btn btn-primary" style="margin-top: 20px; display: inline-block;">
                    In neuem Tab öffnen
                </a>
            </div>
        `;
    }
    
    modal.style.display = 'flex';
}

// Close viewer modal
function closeViewer() {
    document.getElementById('fileViewerModal').style.display = 'none';
}

// Copy URL to clipboard
function copyUrl(url) {
    const fullUrl = window.location.origin + url;
    
    navigator.clipboard.writeText(fullUrl).then(() => {
        showToast('✅ URL kopiert!', 'success');
    }).catch(err => {
        alert('URL kopieren fehlgeschlagen');
    });
}

// Delete file with confirmation
function deleteFile(filePath, fileName) {
    if (confirm(`Möchten Sie die Datei "${fileName}" wirklich löschen?\n\nDiese Aktion kann nicht rückgängig gemacht werden!`)) {
        document.getElementById('delete_file_path').value = filePath;
        document.getElementById('deleteFileForm').submit();
    }
}

// Show toast notification
function showToast(message, type = 'success') {
    const toast = document.createElement('div');
    toast.textContent = message;
    toast.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: ${type === 'success' ? '#10b981' : '#ef4444'};
        color: white;
        padding: 12px 24px;
        border-radius: 8px;
        z-index: 99999;
        animation: slideIn 0.3s;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
    `;
    document.body.appendChild(toast);
    
    setTimeout(() => {
        toast.style.animation = 'slideOut 0.3s';
        setTimeout(() => toast.remove(), 300);
    }, 2000);
}

// Add animations
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    @keyframes slideOut {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
`;
document.head.appendChild(style);

// Close modal on ESC key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeViewer();
    }
});
</script>

</body>
</html>