<?php
/** cms-core/public/webview/admin/page_create.php */

require_once __DIR__ . '/../../../backend/authorization/check_admin.php';
require_once __DIR__ . '/../../../backend/core/init.php';
require_once __DIR__ . '/../../../backend/core/helpers.php';

// Ensure required columns exist
function ensurePagesTableColumns() {
    $db = db();
    
    $checkParent = $db->fetchOne("SHOW COLUMNS FROM pages LIKE 'parent_id'");
    if (!$checkParent) {
        $db->query("ALTER TABLE pages ADD COLUMN parent_id INT NULL DEFAULT NULL AFTER menu_order");
    }
    
    $checkPageType = $db->fetchOne("SHOW COLUMNS FROM pages LIKE 'page_type'");
    if (!$checkPageType) {
        $db->query("ALTER TABLE pages ADD COLUMN page_type VARCHAR(50) NOT NULL DEFAULT 'flexible' AFTER is_deletable");
    }
    
    $checkUsesFlexible = $db->fetchOne("SHOW COLUMNS FROM pages LIKE 'uses_flexible_blocks'");
    if (!$checkUsesFlexible) {
        $db->query("ALTER TABLE pages ADD COLUMN uses_flexible_blocks TINYINT(1) NOT NULL DEFAULT 1 AFTER page_type");
    }
}

try {
    ensurePagesTableColumns();
} catch (Exception $e) {
    error_log("Migration error in page_create.php: " . $e->getMessage());
}

$errors = [];

// Popular emoji icons
$iconOptions = [
    '📄', '🏠', '📧', '📞', '🎨', '🐕', '🧠', '👥', '✍️', '📝', 
    '💼', '🎓', '🏆', '⭐', '🌟', '❤️', '💡', '🔧', '⚙️', '🚀',
    '🎯', '📊', '📈', '💰', '🛒', '🎁', '📱', '💻', '🖥️', '⌚',
    '📷', '🎵', '🎬', '🎮', '🏃', '⚽', '🍕', '☕', '🌍', '✨'
];

$topLevelPages = db()->fetchAll("SELECT id, title FROM pages WHERE parent_id IS NULL ORDER BY menu_order ASC");

if (is_post()) {
    CSRF::require();
    
    $title = trim(post('title'));
    $slug = trim(post('slug'));
    $pageIcon = trim(post('page_icon', '📄'));
    $customIcon = trim(post('page_icon_custom', ''));
    $metaDescription = trim(post('meta_description'));
    $pageType = 'flexible';
    $showInMenu = post('show_in_menu') ? 1 : 0;
    $menuLabel = trim(post('menu_label'));
    $parentId = post('parent_id') ? (int)post('parent_id') : null;
    $status = post('status', 'draft');
    
    if (!empty($customIcon)) {
        $pageIcon = $customIcon;
    }
    
    // Validation
    if (empty($title)) {
        $errors['title'] = 'Titel ist erforderlich.';
    }
    
    if (empty($slug)) {
        $errors['slug'] = 'Slug ist erforderlich.';
    } elseif (!preg_match('/^[a-z0-9-]+$/', $slug)) {
        $errors['slug'] = 'Slug darf nur Kleinbuchstaben, Zahlen und Bindestriche enthalten.';
    } else {
        $existing = db()->fetchOne("SELECT id FROM pages WHERE slug = ?", [$slug]);
        if ($existing) {
            $errors['slug'] = 'Dieser Slug wird bereits verwendet.';
        }
    }
    
    if (empty($errors)) {
        $orderQuery = $parentId ? "SELECT MAX(menu_order) as max_order FROM pages WHERE parent_id = ?" : "SELECT MAX(menu_order) as max_order FROM pages WHERE parent_id IS NULL";
        $params = $parentId ? [$parentId] : [];
        $maxOrder = db()->fetchOne($orderQuery, $params);
        $menuOrder = ($maxOrder['max_order'] ?? 0) + 1;
        
        $pageId = db()->insert(
    "INSERT INTO pages (slug, title, page_icon, meta_description, status, show_in_menu, menu_label, menu_order, parent_id, is_deletable, page_type, uses_flexible_blocks) 
     VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 1, ?, 1)",
    [
        $slug,
        $title,
        $pageIcon,
        $metaDescription,
        $status,
        $showInMenu,
        $menuLabel ?: null,
        $menuOrder,
        $parentId,
        $pageType
        // NOTE: uses_flexible_blocks is hardcoded as 1 in the query above
    ]
);
        
        // Create page file template
        $templateContent = <<<'PHP'
<?php
/** public/webview/SLUG.php */

require_once __DIR__ . '/../../backend/core/init.php';
require_once __DIR__ . '/../../backend/core/FlexibleBlocks.php';

$pageSlug = 'SLUG';
$page = db()->fetchOne("SELECT * FROM pages WHERE slug = ?", [$pageSlug]);

if (!$page) {
    http_response_code(404);
    die('Seite nicht gefunden');
}

$blocks = FlexibleBlocks::getPageBlocks($pageSlug);

$pageTitle = $page['title'];
$metaDescription = $page['meta_description'] ?? '';
$bodyClass = 'page-' . $pageSlug;

ob_start();

if (empty($blocks)) {
    ?>
    <section class="page-header">
        <div class="container">
            <h1><?= e($page['title']) ?></h1>
        </div>
    </section>
    
    <section class="content-section">
        <div class="container">
            <p class="empty-page-message" style="text-align: center; color: #6b7280; padding: 40px 0;">
                Diese Seite hat noch keinen Inhalt.
            </p>
            <?php if (is_admin()): ?>
                <div style="text-align: center;">
                    <a href="/admin/page_flexible_edit.php?slug=SLUG" class="btn btn-primary">
                        ✏️ Inhalt bearbeiten
                    </a>
                </div>
            <?php endif; ?>
        </div>
    </section>
    <?php
} else {
    foreach ($blocks as $block) {
        echo FlexibleBlocks::renderBlock($block);
    }
}

$pageContent = ob_get_clean();
include __DIR__ . '/includes/layout.php';
?>
PHP;
        
        $templateContent = str_replace('SLUG', $slug, $templateContent);
        
        $pageFile = __DIR__ . '/../' . $slug . '.php';
        if (file_put_contents($pageFile, $templateContent)) {
            set_flash('success', 'Seite erfolgreich erstellt! Du kannst jetzt Inhalte hinzufügen.');
        } else {
            set_flash('error', 'Seite in DB erstellt, aber Datei konnte nicht geschrieben werden.');
        }
        
        redirect('/cms-core/public/webview/admin/page_flexible_edit.php?slug=' . $slug);
    }
}

$flashMessages = get_flash();
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Neue Seite erstellen</title>
<?php
$adminCssPath = APP_ROOT . '/assets/css/admin.css';
$adminCssVersion = file_exists($adminCssPath) ? filemtime($adminCssPath) : time();
?>
<link rel="stylesheet" href="/assets/css/admin.css?v=<?= $adminCssVersion ?>">
 <style>
    /* Mobile-optimized layout matching blog */
    @media (max-width: 768px) {
        .page-grid {
            grid-template-columns: 1fr !important;
        }
        
        .icon-grid {
            grid-template-columns: repeat(auto-fill, minmax(45px, 1fr)) !important;
        }
        
        .form-input, .form-textarea {
            font-size: 16px !important; /* Prevents zoom on iOS */
        }
    }
    
    .page-grid {
        display: grid;
        grid-template-columns: 2fr 1fr;
        gap: 20px;
    }
    
    .icon-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(40px, 1fr));
        gap: 8px;
        margin-bottom: 10px;
    }
    
    .icon-option {
        cursor: pointer;
        text-align: center;
        padding: 8px;
        border: 2px solid #e5e7eb;
        border-radius: 8px;
        transition: all 0.2s;
    }
    
    .icon-option:hover {
        border-color: #667eea;
        background: #f0f4ff;
    }
    
    .icon-option input[type="radio"] {
        display: none;
    }
    
    .icon-option input[type="radio"]:checked + span {
        display: block;
    }
    
    .icon-option.selected {
        border-color: #667eea;
        background: #f0f4ff;
    }
    </style>
</head>
<body>
    <?php include 'includes/admin_header.php'; ?>
    
    <div class="admin-container">
        <?php include 'includes/admin_sidebar.php'; ?>
        
        <main class="admin-content">
            <?php foreach ($flashMessages as $flash): ?>
                <div class="alert alert-<?= e($flash['type']) ?>">
                    <?= e($flash['message']) ?>
                </div>
            <?php endforeach; ?>
            
            <div class="page-header">
                <h1>➕ Neue Seite erstellen</h1>
                <p>Erstelle eine neue Seite für deine Website</p>
            </div>
            
            <?php if (!empty($errors)): ?>
                <div class="alert alert-error" style="margin-bottom: 20px;">
                    <strong>Fehler beim Erstellen:</strong>
                    <ul style="margin: 10px 0 0 20px;">
                        <?php foreach ($errors as $error): ?>
                            <li><?= e($error) ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>
            
            <form method="POST" id="page-form">
                <?= CSRF::field() ?>
                
                <div class="page-grid">
                    <!-- Main Content -->
                    <div>
                        <div class="content-card">
                            <div class="form-group">
                                <label class="form-label" for="title">Seitentitel *</label>
                                <input type="text" 
                                       id="title" 
                                       name="title" 
                                       class="form-input"
                                       value="<?= e(post('title', '')) ?>"
                                       required
                                       placeholder="z.B. Über uns">
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label" for="slug">
    Slug *
    <span class="tooltip-wrapper">
        <span class="tooltip-trigger">?</span>
        <span class="tooltip-content">
            Der Slug ist die URL-freundliche Version des Titels. 
            Nur Kleinbuchstaben, Zahlen und Bindestriche erlaubt.
            Beispiel: "ueber-uns" → yoursite.com/ueber-uns
        </span>
    </span>
</label>
                                <input type="text" 
                                       id="slug" 
                                       name="slug" 
                                       class="form-input"
                                       value="<?= e(post('slug', '')) ?>"
                                       required
                                       pattern="[a-z0-9-]+"
                                       placeholder="ueber-uns">
                                <span style="font-size: 12px; color: #6b7280;">Nur Kleinbuchstaben, Zahlen und Bindestriche</span>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label" for="meta_description">
    Meta-Beschreibung
    <span class="tooltip-wrapper">
        <span class="tooltip-trigger">?</span>
        <span class="tooltip-content">
            Erscheint in Google-Suchergebnissen unter dem Titel.
            Optimal: 150-160 Zeichen. Beschreibe kurz den Inhalt der Seite.
        </span>
    </span>
</label>
                                <textarea id="meta_description" 
                                          name="meta_description" 
                                          class="form-textarea" 
                                          rows="3" 
                                          placeholder="Kurze Beschreibung für Suchmaschinen (150-160 Zeichen)"><?= e(post('meta_description', '')) ?></textarea>
                                <span style="font-size: 12px; color: #6b7280;">Wird in Suchmaschinen-Ergebnissen angezeigt</span>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">Seiten-Icon</label>
                                <div class="icon-grid">
                                    <?php foreach ($iconOptions as $icon): ?>
                                        <label class="icon-option <?= post('page_icon', '📄') === $icon ? 'selected' : '' ?>">
                                            <input type="radio" name="page_icon" value="<?= e($icon) ?>" <?= post('page_icon', '📄') === $icon ? 'checked' : '' ?>>
                                            <span style="font-size: 24px;"><?= $icon ?></span>
                                        </label>
                                    <?php endforeach; ?>
                                </div>
                                <div style="margin-top: 10px;">
                                    <label class="form-label" style="font-size: 13px;">Oder eigenes Icon/Emoji:</label>
                                    <input type="text" name="page_icon_custom" class="form-input" placeholder="🎨" maxlength="2" style="width: 80px; text-align: center; font-size: 20px;">
                                </div>
                            </div>
                        </div>
                        
                        <div class="content-card" style="margin-top: 20px;">
                            <h3 style="margin-bottom: 20px;">Menü-Einstellungen</h3>
                            
                            <div class="form-group">
                                <label class="form-label">
                                    <input type="checkbox" name="show_in_menu" value="1" <?= post('show_in_menu', true) ? 'checked' : '' ?> style="margin-right: 8px;">
                                    Im Menü anzeigen
                                </label>
                                <span style="font-size: 12px; color: #6b7280; display: block; margin-top: 4px;">Wenn aktiviert, erscheint die Seite in der Navigation</span>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label" for="menu_label">Menü-Label (optional)</label>
                                <input type="text" 
                                       id="menu_label" 
                                       name="menu_label" 
                                       class="form-input"
                                       value="<?= e(post('menu_label', '')) ?>"
                                       placeholder="Leer = Seitentitel verwenden">
                                <span style="font-size: 12px; color: #6b7280;">Alternative Bezeichnung für die Navigation</span>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label" for="parent_id">Unterseite von (optional)</label>
                                <select id="parent_id" name="parent_id" class="form-input">
                                    <option value="">Keine - Top-Level Seite</option>
                                    <?php foreach ($topLevelPages as $topPage): ?>
                                        <option value="<?= $topPage['id'] ?>" <?= post('parent_id') == $topPage['id'] ? 'selected' : '' ?>>
                                            <?= e($topPage['title']) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <span style="font-size: 12px; color: #6b7280;">Wähle eine übergeordnete Seite für Dropdown-Menüs</span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Sidebar -->
                    <div>
                        <!-- Status & Publish -->
                        <div class="content-card">
                            <h3 style="margin: 0 0 15px 0;">📊 Status</h3>
                            
                            <div class="form-group">
                                <label class="form-label" for="status">Status</label>
                                <select id="status" name="status" class="form-input">
                                    <option value="draft" <?= post('status', 'draft') === 'draft' ? 'selected' : '' ?>>
                                        🔸 Entwurf
                                    </option>
                                    <option value="published" <?= post('status') === 'published' ? 'selected' : '' ?>>
                                        ✅ Veröffentlicht
                                    </option>
                                </select>
                            </div>
                            
                            <button type="submit" class="btn btn-primary" style="width: 100%; margin-top: 15px;">
                                ➕ Seite erstellen
                            </button>
                            
                            <a href="page_manage.php" class="btn btn-secondary" style="width: 100%; margin-top: 10px; text-align: center; display: block;">
                                Abbrechen
                            </a>
                        </div>
                    </div>
                </div>
            </form>
        </main>
    </div>
    
    <script>
    // Auto-generate slug from title
    document.getElementById('title').addEventListener('input', function(e) {
        const slug = e.target.value
            .toLowerCase()
            .replace(/ä/g, 'ae')
            .replace(/ö/g, 'oe')
            .replace(/ü/g, 'ue')
            .replace(/ß/g, 'ss')
            .replace(/[^a-z0-9-]/g, '-')
            .replace(/-+/g, '-')
            .replace(/^-|-$/g, '');
        
        if (!document.getElementById('slug').value || document.getElementById('slug').dataset.autoGenerated !== 'false') {
            document.getElementById('slug').value = slug;
            document.getElementById('slug').dataset.autoGenerated = 'true';
        }
    });
    
    document.getElementById('slug').addEventListener('input', function(e) {
        e.target.dataset.autoGenerated = 'false';
    });
    
    // Icon selection
    document.querySelectorAll('.icon-option').forEach(option => {
        option.addEventListener('click', function() {
            document.querySelectorAll('.icon-option').forEach(opt => opt.classList.remove('selected'));
            this.classList.add('selected');
            this.querySelector('input[type="radio"]').checked = true;
        });
    });
    </script>
    <!-- CRITICAL: Add main.js for mobile menu functionality -->
    <script src="/assets/js/main.js"></script>
</body>
</html>