<?php
/** cms-core/public/webview/admin/page_flexible_edit.php */

require_once __DIR__ . '/../../../backend/authorization/check_admin.php';
require_once __DIR__ . '/../../../backend/core/init.php';
require_once __DIR__ . '/../../../backend/core/helpers.php';
require_once __DIR__ . '/../../../backend/core/FlexibleBlocks.php';

// RIGHT AFTER: require_once __DIR__ . '/../../../backend/core/FlexibleBlocks.php';

// Start output buffer to capture everything
ob_start();
// end of debug 

$pageSlug = isset($_GET['slug']) ? trim($_GET['slug']) : '';

if (!$pageSlug) {
    set_flash('error', 'Keine Seite ausgewählt.');
    redirect('/admin/page_manage');
    exit;
}

// FETCH THE PAGE FROM DATABASE
$currentPage = db()->fetchOne("SELECT * FROM pages WHERE slug = ?", [$pageSlug]);

if (!$currentPage) {
    set_flash('error', 'Seite nicht gefunden: ' . htmlspecialchars($pageSlug));
    redirect('/admin/page_manage');
    exit;
}

// PROTECT THE DATA: Store in separate variables BEFORE includes can overwrite
$pageTitle = $currentPage['title'] ?? '';
$pageMetaDesc = $currentPage['meta_description'] ?? '';
$pageStatus = $currentPage['status'] ?? 'published';
$pageSlugSafe = $currentPage['slug'] ?? '';

$blocks = FlexibleBlocks::getPageBlocks($pageSlug);
$availableBlockTypes = FlexibleBlocks::getAvailableBlockTypes();

// Handle AJAX requests
if (is_post() && get_param('ajax')) {
    // CRITICAL: Clean any output buffer before JSON response
    while (ob_get_level() > 0) {
        ob_end_clean();
    }
    
    header('Content-Type: application/json');
    
    try {
        $action = post('action');
        
        switch ($action) {
            case 'add_block':
                $blockType = post('block_type');
                if (!array_key_exists($blockType, $availableBlockTypes)) {
                    throw new Exception('Invalid block type');
                }
                $blockId = FlexibleBlocks::addBlock($pageSlug, $blockType);
                echo json_encode(['success' => true, 'block_id' => $blockId]);
                exit;
                
            case 'delete_block':
                $blockId = (int)post('block_id');
                FlexibleBlocks::deleteBlock($blockId);
                echo json_encode(['success' => true]);
                exit;
                
            case 'reorder_blocks':
                $blockIds = json_decode(post('block_ids'), true);
                if (!is_array($blockIds)) {
                    throw new Exception('Invalid block IDs');
                }
                $indexedBlockIds = array_combine(range(1, count($blockIds)), $blockIds);
                FlexibleBlocks::reorderBlocks($pageSlug, $indexedBlockIds);
                echo json_encode(['success' => true]);
                exit;
                
            default:
                throw new Exception('Unknown action');
        }
    } catch (Exception $e) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        exit;
    }
}

// Handle Form Submit (Save all blocks)
$errors = [];
if (is_post() && !get_param('ajax')) {
    try {
        CSRF::require();
    } catch (Exception $e) {
        set_flash('error', 'Ungültiger CSRF-Token oder Serverfehler.');
        error_log("CSRF validation failed: " . $e->getMessage());
        redirect('/admin/page_flexible_edit?slug=' . urlencode($pageSlug));
        exit;
    }
    
    $blockData = post('block_data', []);
    $hasError = false;
    
    // Handle checkboxes properly  
    foreach ($blocks as $block) {
        $blockId = $block['id'];
        $blockType = $availableBlockTypes[$block['block_type_template']] ?? null;
        
        if ($blockType) {
            foreach ($blockType['fields'] as $fieldKey => $field) {
                // Handle top-level checkboxes
                if ($field['type'] === 'checkbox') {
                    if (!isset($blockData[$blockId][$fieldKey])) {
                        $blockData[$blockId][$fieldKey] = false;
                    } else {
                        $blockData[$blockId][$fieldKey] = (bool)$blockData[$blockId][$fieldKey];
                    }
                }
                
                // Handle checkboxes in repeaters
                if ($field['type'] === 'repeater' && isset($field['fields']) && isset($blockData[$blockId][$fieldKey]) && is_array($blockData[$blockId][$fieldKey])) {
                    $hasCheckboxes = false;
                    foreach ($field['fields'] as $subField) {
                        if ($subField['type'] === 'checkbox') {
                            $hasCheckboxes = true;
                            break;
                        }
                    }
                    
                    if ($hasCheckboxes) {
                        foreach ($field['fields'] as $subFieldKey => $subField) {
                            if ($subField['type'] === 'checkbox') {
                                foreach ($blockData[$blockId][$fieldKey] as $index => $itemData) {
                                    if (!is_array($itemData)) continue;
                                    if (!isset($itemData[$subFieldKey])) {
                                        $blockData[$blockId][$fieldKey][$index][$subFieldKey] = false;
                                    } else {
                                        $blockData[$blockId][$fieldKey][$index][$subFieldKey] = (bool)$itemData[$subFieldKey];
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    
    // Process file uploads
    if (isset($_FILES['block_images'])) {
        foreach ($_FILES['block_images']['tmp_name'] as $blockId => $files) {
            if (!is_array($files)) continue;
            foreach ($files as $fieldPath => $tmpName) {
                if ($tmpName && is_uploaded_file($tmpName)) {
                    $fileInfo = [
                        'name' => $_FILES['block_images']['name'][$blockId][$fieldPath] ?? '',
                        'type' => $_FILES['block_images']['type'][$blockId][$fieldPath] ?? '',
                        'tmp_name' => $tmpName,
                        'error' => $_FILES['block_images']['error'][$blockId][$fieldPath] ?? 0,
                        'size' => $_FILES['block_images']['size'][$blockId][$fieldPath] ?? 0
                    ];
                    
                    $_FILES['temp_upload'] = $fileInfo;
                    $uploadResult = handle_file_upload('temp_upload', APP_ROOT . '/media/pages');
                    
                    if ($uploadResult['success']) {
                        $keys = explode('.', $fieldPath);
                        $temp = &$blockData[$blockId];
                        foreach ($keys as $i => $key) {
                            if ($i === count($keys) - 1) {
                                $temp[$key] = $uploadResult['filename'];
                            } else {
                                if (!isset($temp[$key]) || !is_array($temp[$key])) $temp[$key] = [];
                                $temp = &$temp[$key];
                            }
                        }
                    } else {
                        $errors["block_{$blockId}_{$fieldPath}"] = $uploadResult['error'];
                        $hasError = true;
                    }
                    
                    unset($_FILES['temp_upload']);
                }
            }
        }
    }

    // Save blocks
    if (!$hasError) {
        $validBlockIds = array_map(function($block) { return $block['id']; }, $blocks);
        $cleanedBlockData = [];
        
        foreach ($blockData as $blockId => $blockFields) {
            if (!in_array($blockId, $validBlockIds)) continue;
            
            $cleanedBlockData[$blockId] = $blockFields;
            
            // Process repeater fields
            foreach ($blockFields as $fieldKey => $fieldValue) {
                if (!is_array($fieldValue) || empty($fieldValue)) continue;
                
                $firstItem = reset($fieldValue);
                if (!is_array($firstItem)) continue;
                
                // Filter empty items
                $cleanedItems = [];
                foreach ($fieldValue as $item) {
                    if (!is_array($item)) continue;
                    
                    $hasData = false;
                    foreach ($item as $value) {
                        if ($value !== '' && $value !== null && $value !== false && $value !== []) {
                            $hasData = true;
                            break;
                        }
                    }
                    
                    if ($hasData) {
                        $cleanedItems[] = $item;
                    }
                }
                
                $cleanedBlockData[$blockId][$fieldKey] = $cleanedItems;
            }
        }
        
        // Save cleaned data
        foreach ($cleanedBlockData as $blockId => $data) {
            FlexibleBlocks::updateBlock($blockId, $data);
        }
        
        // Update page metadata (FIXED: removed updated_at column)
        $newTitle = trim(post('title', $currentPage['title']));
        $newMetaDesc = trim(post('meta_description', $currentPage['meta_description'] ?? ''));
        $newStatus = post('status', $currentPage['status'] ?? 'published');
        
        db()->execute(
            "UPDATE pages SET title = ?, meta_description = ?, status = ? WHERE slug = ?",
            [$newTitle, $newMetaDesc, $newStatus, $pageSlug]
        );
        
        set_flash('success', 'Seite erfolgreich aktualisiert.');
        redirect('/admin/page_flexible_edit?slug=' . urlencode($pageSlug));
        exit;
    }
}

$pageBrowserTitle = 'Seite bearbeiten: ' . ($pageTitle);
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= e($pageBrowserTitle) ?></title>
<?php
$adminCssPath = APP_ROOT . '/assets/css/admin.css';
$adminCssVersion = file_exists($adminCssPath) ? filemtime($adminCssPath) : time();
?>
<link rel="stylesheet" href="/assets/css/admin.css?v=<?= $adminCssVersion ?>">
</head>
<body>
    <?php include 'includes/admin_header.php'; ?>
    
    <div class="admin-container">
        <?php include 'includes/admin_sidebar.php'; ?>
        
        <main class="admin-content">
            <div class="page-header">
    <div>
        <h1>📄 <?= e($pageTitle) ?> bearbeiten</h1>
        <p class="page-description">Flexible Blöcke hinzufügen, bearbeiten und sortieren</p>
    </div>
    <div style="display: flex; gap: 10px;">
        <a href="/admin/page_visual_edit.php?slug=<?= urlencode($pageSlug) ?>" class="btn btn-sm btn-primary" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border: none;">
            🎨 Visual Editor
        </a>
        <a href="<?= $pageSlugSafe === 'home' ? '/' : '/' . e($pageSlugSafe) ?>" class="btn btn-sm btn-secondary" target="_blank">Vorschau</a>
    </div>
</div>
            
            <?php 
            $flashMessages = get_flash(); 
            foreach ($flashMessages as $flash): 
            ?>
                <div class="alert alert-<?= e($flash['type']) ?>">
                    <?= e($flash['message']) ?>
                </div>
            <?php endforeach; ?>
            
            <?php if (!empty($errors)): ?>
                <div class="alert alert-error">
                    <?php foreach ($errors as $error): ?>
                        <p><?= e($error) ?></p>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
            
            <form method="POST" action="/admin/page_flexible_edit?slug=<?= urlencode($pageSlug) ?>" enctype="multipart/form-data">
                <?= CSRF::field() ?>
                
                <!-- Page Meta -->
<div class="content-card">
    <h3 class="card-title">Seiteneinstellungen</h3>
    
    <div class="form-group">
    <label class="form-label" for="title">Seitentitel *</label>
    <input type="text" id="title" name="title" class="form-input" 
       value="<?= e($pageTitle) ?>" required>
</div>

<div class="form-group">
    <label class="form-label" for="meta_description">Meta-Beschreibung</label>
    <textarea id="meta_description" name="meta_description" class="form-textarea" 
          rows="2" maxlength="160"><?= e($pageMetaDesc) ?></textarea>
    <span class="form-help">SEO (max. 160 Zeichen)</span>
</div>

<div class="form-group">
    <label class="form-label" for="status">Status</label>
    <select id="status" name="status" class="form-select">
    <option value="draft" <?= $pageStatus === 'draft' ? 'selected' : '' ?>>Entwurf</option>
    <option value="published" <?= $pageStatus === 'published' ? 'selected' : '' ?>>Veröffentlicht</option>
</select>
</div>
</div>
                
              <!-- Blocks Section -->
                <div class="content-card">
                    <div class="card-header">
                        <h3 class="card-title">Content Blöcke</h3>
                        <button type="button" class="btn btn-primary" onclick="showAddBlockModal()">
                            ➕ Block hinzufügen
                        </button>
                    </div>
                    
                    <div id="blocks-container" class="blocks-container">
                        <?php if (empty($blocks)): ?>
                            <div class="empty-content">
                                <div class="empty-icon">📦</div>
                                <p class="empty-text">Noch keine Blöcke vorhanden. Füge deinen ersten Block hinzu!</p>
                                <button type="button" class="btn btn-primary" onclick="showAddBlockModal()">
                                    ➕ Block hinzufügen
                                </button>
                            </div>
                        <?php else: ?>
                            <?php foreach ($blocks as $block): ?>
                                <?php include __DIR__ . '/includes/block_editor.php'; ?>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Submit -->
                <div class="form-actions">
                    <a href="/admin/page_manage" class="btn btn-secondary">Abbrechen</a>
                    <button type="submit" class="btn btn-primary">💾 Änderungen speichern</button>
                </div>
            </form>
        </main>
    </div>
    
    <!-- Add Block Modal -->
    <div id="add-block-modal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Block hinzufügen</h3>
                <button type="button" class="modal-close" onclick="closeAddBlockModal()">×</button>
            </div>
            <div class="modal-body">
                <div class="block-types-grid">
                    <?php foreach ($availableBlockTypes as $typeKey => $type): ?>
                        <div class="block-type-card" onclick="addBlock('<?= e($typeKey) ?>')">
                            <div class="block-type-icon"><?= e($type['icon']) ?></div>
                            <div class="block-type-name"><?= e($type['name']) ?></div>
                            <div class="block-type-description"><?= e($type['description']) ?></div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Media Library Modal -->
    <div id="media-library-modal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>📁 Mediathek</h3>
                <button type="button" class="modal-close" onclick="closeMediaLibrary()">×</button>
            </div>
            <div class="modal-body">
                <div id="media-library-grid" class="media-library-grid">
                    <div class="empty-media">Lade Bilder...</div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="/cms-core/public/webview/assets/js/main.js"></script>
<script>
let currentImageFieldId = null;
let currentImageBlockId = null;
let currentImageFieldKey = null;

function showAddBlockModal() {
    document.getElementById('add-block-modal').style.display = 'flex';
}

function closeAddBlockModal() {
    document.getElementById('add-block-modal').style.display = 'none';
}

async function addBlock(blockType) {
    const formData = new FormData();
    formData.append('action', 'add_block');
    formData.append('block_type', blockType);
    
    const url = '/admin/page_flexible_edit?slug=<?= urlencode($pageSlug) ?>&ajax=1';
    
    try {
        const response = await fetch(url, {
            method: 'POST',
            body: formData
        });
        
        if (!response.ok) {
            const errorText = await response.text();
            throw new Error(`HTTP ${response.status}: ${response.statusText} - ${errorText.substring(0, 200)}`);
        }
        
        const result = await response.json();
        
        if (result.success) {
            location.reload();
        } else {
            alert('Fehler beim Hinzufügen: ' + (result.error || 'Unbekannter Fehler'));
        }
    } catch (error) {
        console.error('AJAX Error:', error);
        alert('Fehler beim Hinzufügen: ' + error.message);
    }
}

async function deleteBlock(blockId) {
    if (!confirm('Block wirklich löschen?')) return;
    
    const formData = new FormData();
    formData.append('action', 'delete_block');
    formData.append('block_id', blockId);
    
    const url = '/admin/page_flexible_edit?slug=<?= urlencode($pageSlug) ?>&ajax=1';
    
    try {
        const response = await fetch(url, {
            method: 'POST',
            body: formData
        });
        
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        const result = await response.json();
        
        if (result.success) {
            location.reload();
        } else {
            alert('Fehler beim Löschen: ' + (result.error || 'Unbekannter Fehler'));
        }
    } catch (error) {
        console.error('AJAX Error:', error);
        alert('Fehler beim Löschen: ' + error.message);
    }
}

function toggleBlock(blockId) {
    const content = document.getElementById('block-content-' + blockId);
    const icon = document.getElementById('toggle-icon-' + blockId);
    
    if (content && icon) {
        // Toggle between block and none
        const isCurrentlyVisible = content.style.display !== 'none';
        content.style.display = isCurrentlyVisible ? 'none' : 'block';
        icon.textContent = isCurrentlyVisible ? '▶' : '▼';
    }
}

function openMediaLibrary(fieldId, blockId, fieldKey) {
    currentImageFieldId = fieldId;
    currentImageBlockId = blockId;
    currentImageFieldKey = fieldKey;
    
    document.getElementById('media-library-modal').style.display = 'flex';
    
    fetch('/admin/media_selector')
        .then(response => response.json())
        .then(data => {
            const grid = document.getElementById('media-library-grid');
            
            if (data.images && data.images.length > 0) {
                grid.innerHTML = data.images.map(img => `
                    <div class="media-item" onclick="selectMediaImage('${img.filename}')">
                        <img src="/cms-core/public/webview/media.php?file=pages/${img.filename}" alt="${img.filename}">
                        <div class="media-item-info">${img.filename}</div>
                    </div>
                `).join('');
            } else {
                grid.innerHTML = '<div class="empty-media">Keine Bilder in der Mediathek</div>';
            }
        })
        .catch(error => {
            console.error('Error loading media:', error);
            document.getElementById('media-library-grid').innerHTML = 
                '<div class="empty-media error">Fehler beim Laden der Bilder</div>';
        });
}

function closeMediaLibrary() {
    document.getElementById('media-library-modal').style.display = 'none';
}

function selectMediaImage(filename) {
    document.getElementById(`hidden-${currentImageFieldId}`).value = filename;
    
    const preview = document.getElementById(`preview-${currentImageFieldId}`);
    const previewContainer = document.getElementById(`preview-container-${currentImageFieldId}`);
    
    if (preview) {
        preview.src = `/media/pages/${filename}`;
        if (previewContainer) {
            previewContainer.style.display = 'block';
        }
    }
    
    closeMediaLibrary();
}

function previewNewImage(input, fieldId) {
    if (input.files && input.files[0]) {
        const reader = new FileReader();
        reader.onload = function(e) {
            const preview = document.getElementById(`preview-${fieldId}`);
            const previewContainer = document.getElementById(`preview-container-${fieldId}`);
            
            if (preview) {
                preview.src = e.target.result;
                if (previewContainer) {
                    previewContainer.style.display = 'block';
                }
            }
        };
        reader.readAsDataURL(input.files[0]);
    }
}

let draggedElement = null;

function handleDragStart(e) {
    draggedElement = e.target.closest('.block-editor');
    if (draggedElement) {
        draggedElement.classList.add('dragging');
    }
    e.dataTransfer.effectAllowed = 'move';
}

function handleDragEnd(e) {
    if (draggedElement) {
        draggedElement.classList.remove('dragging');
    }
    draggedElement = null;
}

function handleDragOver(e) {
    if (e.preventDefault) {
        e.preventDefault();
    }
    e.dataTransfer.dropEffect = 'move';
    return false;
}

function handleDrop(e) {
    if (e.stopPropagation) {
        e.stopPropagation();
    }
    
    if (draggedElement && draggedElement !== e.currentTarget.closest('.block-editor')) {
        const container = document.getElementById('blocks-container');
        const all = Array.from(container.querySelectorAll('.block-editor'));
        const draggedIndex = all.indexOf(draggedElement);
        const targetIndex = all.indexOf(e.currentTarget.closest('.block-editor'));
        
        if (draggedIndex !== -1 && targetIndex !== -1) {
            if (draggedIndex < targetIndex) {
                e.currentTarget.closest('.block-editor').after(draggedElement);
            } else {
                e.currentTarget.closest('.block-editor').before(draggedElement);
            }
            
            saveBlockOrder();
        }
    }
    
    return false;
}

async function saveBlockOrder() {
    const blocks = document.querySelectorAll('.block-editor');
    const blockIds = Array.from(blocks).map(block => block.dataset.blockId).filter(id => id);
    
    const formData = new FormData();
    formData.append('action', 'reorder_blocks');
    formData.append('block_ids', JSON.stringify(blockIds));
    
    const url = '/admin/page_flexible_edit?slug=<?= urlencode($pageSlug) ?>&ajax=1';
    
    try {
        const response = await fetch(url, {
            method: 'POST',
            body: formData
        });
        
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        await response.json();
    } catch (error) {
        console.error('AJAX Error:', error);
    }
}

function addRepeaterItem(blockId, fieldName) {
    const container = document.getElementById(`repeater-${blockId}-${fieldName}`);
    if (!container) return;
    
    let template = container.querySelector('.repeater-template');
    if (!template) {
        const firstItem = container.querySelector('.repeater-item');
        if (!firstItem) return;
        template = firstItem.cloneNode(true);
    }
    
    const existingItems = container.querySelectorAll('.repeater-item:not(.repeater-template)');
    const nextIndex = existingItems.length;
    
    const newItem = template.cloneNode(true);
    newItem.classList.remove('repeater-template');
    newItem.style.display = 'block';
    
    const namePattern = new RegExp(`\\[${fieldName}\\]\\[\\d+\\]`, 'g');
    const idPattern = new RegExp(`_${fieldName}_\\d+_`, 'g');
    
    newItem.querySelectorAll('input, textarea, select').forEach(el => {
        el.disabled = false;
        el.removeAttribute('data-template-field');
        
        if (el.name) {
            el.name = el.name.replace(namePattern, `[${fieldName}][${nextIndex}]`);
        }
        
        if (el.id) {
            el.id = el.id.replace(idPattern, `_${fieldName}_${nextIndex}_`);
        }
        
        if (el.type === 'checkbox') {
            el.checked = false;
        } else if (el.type === 'file') {
            el.value = '';
        } else if (el.type !== 'hidden') {
            el.value = '';
        }
    });
    
    newItem.querySelectorAll('label[for]').forEach(label => {
        if (label.getAttribute('for')) {
            const newFor = label.getAttribute('for').replace(idPattern, `_${fieldName}_${nextIndex}_`);
            label.setAttribute('for', newFor);
        }
    });
    
    const header = newItem.querySelector('.repeater-item-header strong');
    if (header) {
        const labelText = header.textContent.replace(/#\d+/, '').trim();
        header.textContent = `${labelText} #${nextIndex + 1}`;
    }
    
    const removeBtn = newItem.querySelector('.block-action-btn.danger');
    if (removeBtn) {
        removeBtn.onclick = function() { removeRepeaterItem(this); };
    }
    
    if (template.parentNode === container) {
        container.insertBefore(newItem, template);
    } else {
        container.appendChild(newItem);
    }
}

function removeRepeaterItem(btn) {
    const item = btn.closest('.repeater-item');
    if (item && !item.classList.contains('repeater-template')) {
        item.remove();
    }
}

// Toggle between Emoji and Image input for icon_or_image fields
function toggleIconType(radio, fieldId) {
    const emojiDiv = document.getElementById('emoji-input-' + fieldId);
    const imageDiv = document.getElementById('image-input-' + fieldId);
    const emojiInput = document.getElementById(fieldId + '-emoji');
    const imageHiddenInput = document.getElementById('hidden-' + fieldId + '-image');
    
    if (radio.value === 'emoji') {
        // Show emoji, hide image
        if (emojiDiv) emojiDiv.style.display = 'block';
        if (imageDiv) imageDiv.style.display = 'none';
        
        // Clear image values when switching to emoji
        if (imageHiddenInput) {
            imageHiddenInput.value = '';
            imageHiddenInput.disabled = true;
        }
        
        // Make sure emoji input is enabled and visible
        if (emojiInput) {
            emojiInput.disabled = false;
        }
    } else {
        // Show image, hide emoji
        if (emojiDiv) emojiDiv.style.display = 'none';
        if (imageDiv) imageDiv.style.display = 'block';
        
        // Clear emoji values when switching to image
        if (emojiInput) {
            emojiInput.value = '';
            emojiInput.disabled = true;
        }
        
        // Make sure image input is enabled
        if (imageHiddenInput) {
            imageHiddenInput.disabled = false;
        }
    }
}

// Remove icon image
function removeIconImage(fieldId, blockId, fieldKey, index) {
    if (!confirm('Bild wirklich entfernen?')) {
        return;
    }
    
    // Clear the hidden input value
    const hiddenInput = document.getElementById('hidden-' + fieldId + '-image');
    if (hiddenInput) {
        hiddenInput.value = '';
    }
    
    // Hide preview
    const previewContainer = document.getElementById('preview-container-' + fieldId + '-image');
    if (previewContainer) {
        previewContainer.style.display = 'none';
    }
    
    // Switch back to emoji
    const emojiRadio = document.querySelector(`input[name="block_data[${blockId}][${fieldKey}][${index}][icon_type]"][value="emoji"]`);
    if (emojiRadio) {
        emojiRadio.checked = true;
        toggleIconType(emojiRadio, fieldId);
    }
}

document.addEventListener('DOMContentLoaded', function() {
    // Setup drag and drop
    const blocks = document.querySelectorAll('.block-editor');
    blocks.forEach(block => {
        block.addEventListener('dragstart', handleDragStart);
        block.addEventListener('dragend', handleDragEnd);
        block.addEventListener('dragover', handleDragOver);
        block.addEventListener('drop', handleDrop);
    });
    
    // Disable template fields
    document.querySelectorAll('.repeater-template').forEach(template => {
        template.querySelectorAll('input, textarea, select').forEach(input => {
            input.disabled = true;
            input.setAttribute('data-template-field', 'true');
        });
    });
    
    // COLLAPSE ALL BLOCKS ON PAGE LOAD
    blocks.forEach(block => {
        const blockId = block.dataset.blockId;
        if (blockId) {
            const content = document.getElementById('block-content-' + blockId);
            const icon = document.getElementById('toggle-icon-' + blockId);
            
            if (content && icon) {
                content.style.display = 'none';
                icon.textContent = '▶';
            }
        }
    });
});
</script>
</body>
</html>