<?php
/** cms-core/public/webview/admin/page_manage.php */

require_once __DIR__ . '/../../../backend/authorization/check_admin.php';
require_once __DIR__ . '/../../../backend/core/PageManager.php';
require_once __DIR__ . '/../../../backend/core/MenuManager.php';
require_once __DIR__ . '/../../../backend/core/LegalPagesManager.php';
require_once __DIR__ . '/../../../backend/security/csrf.php';

// Pagination & Search Settings
$perPage = 20;
$page = max(1, (int)get_param('page', 1));
$search = trim(get_param('search', ''));
$statusFilter = get_param('status', 'all'); // all, published, draft

// Handle Delete
if (is_post() && ($_POST['action'] ?? '') === 'delete') {
    try {
        CSRF::require();
        $pageId = (int)post('page_id');
        $result = PageManager::deletePage($pageId);
        if ($result['success']) {
            set_flash('success', 'Seite erfolgreich gelöscht.');
        } else {
            set_flash('error', $result['error']);
        }
    } catch (Exception $e) {
        set_flash('error', 'Fehler beim Löschen: ' . $e->getMessage());
        error_log("Delete Error: " . $e->getMessage());
    }
    redirect('page_manage.php' . buildQueryString());
}

// Handle Quick Update
if (is_post() && ($_POST['action'] ?? '') === 'quick_update') {
    try {
        CSRF::require();
        $pageId = (int)post('page_id');
        $field = post('field');
        $value = post('value');
        
        $pageData = db()->fetchOne("SELECT slug FROM pages WHERE id = ?", [$pageId]);
        if (!$pageData) {
            echo json_encode(['success' => false, 'error' => 'Seite nicht gefunden']);
            exit;
        }
        
        if ($field === 'status') {
            db()->execute("UPDATE pages SET status = ? WHERE id = ?", [$value, $pageId]);
        } elseif ($field === 'show_in_menu') {
            MenuManager::toggleMenuVisibility($pageId, (int)$value);
        } elseif ($field === 'parent_id') {
            MenuManager::setPageParent($pageId, $value ? (int)$value : null);
        } elseif ($field === 'show_in_footer') {
            LegalPagesManager::updateFooterVisibility($pageId, (int)$value);
        } else {
            PageManager::updatePageSettings($pageData['slug'], [$field => $value]);
        }
        
        echo json_encode(['success' => true, 'field' => $field, 'value' => $value]);
    } catch (Exception $e) {
        error_log("Quick Update Error: " . $e->getMessage());
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// Handle Reorder
if (is_post() && ($_POST['action'] ?? '') === 'reorder') {
    try {
        CSRF::require();
        $order = json_decode(post('order'), true);
        $parentId = post('parent_id');
        MenuManager::reorderMenu($order, $parentId);
        echo json_encode(['success' => true]);
    } catch (Exception $e) {
        error_log("Reorder Error: " . $e->getMessage());
        echo json_encode(['success' => false, 'error' => 'Fehler beim Neusortieren']);
    }
    exit;
}

// Build query for pagination with search/filter
function buildSearchQuery() {
    global $search, $statusFilter;
    
    $where = "1=1";
    $params = [];
    
    if ($search) {
        $where .= " AND (title LIKE ? OR slug LIKE ? OR meta_description LIKE ?)";
        $searchTerm = "%{$search}%";
        $params = [$searchTerm, $searchTerm, $searchTerm];
    }
    
    if ($statusFilter === 'published') {
        $where .= " AND status = 'published'";
    } elseif ($statusFilter === 'draft') {
        $where .= " AND status = 'draft'";
    }
    
    return ['where' => $where, 'params' => $params];
}

function buildQueryString() {
    global $search, $statusFilter, $page;
    $params = [];
    if ($search) $params[] = 'search=' . urlencode($search);
    if ($statusFilter && $statusFilter !== 'all') $params[] = 'status=' . urlencode($statusFilter);
    if ($page > 1) $params[] = 'page=' . $page;
    return $params ? '?' . implode('&', $params) : '';
}

try {
    $query = buildSearchQuery();
    
    // Get total count for pagination
    $countSql = "SELECT COUNT(*) as count FROM pages WHERE {$query['where']}";
    $totalCount = db()->fetchOne($countSql, $query['params'])['count'];
    $totalPages = max(1, ceil($totalCount / $perPage));
    $offset = ($page - 1) * $perPage;
    
    // Fetch paginated pages
    $sql = "SELECT *, COALESCE(page_icon, '📄') AS page_icon 
            FROM pages 
            WHERE {$query['where']}
            ORDER BY 
                CASE WHEN is_system_page = 1 THEN 0 ELSE 1 END,
                menu_order ASC, 
                created_at DESC
            LIMIT {$perPage} OFFSET {$offset}";
    
    $pages = db()->fetchAll($sql, $query['params']);
    
    // Get all pages for statistics (not filtered)
    $allPages = PageManager::getAllPages();
    $menuPages = MenuManager::getMenuPagesWithHierarchy();
    
    // Calculate stats
    $totalPagesCount = count($allPages);
    $publishedCount = count(array_filter($allPages, fn($p) => $p['status'] === 'published'));
    $draftCount = count(array_filter($allPages, fn($p) => $p['status'] === 'draft'));
    
} catch (Exception $e) {
    error_log("Error fetching pages: " . $e->getMessage());
    set_flash('error', 'Fehler beim Laden der Seiten: ' . $e->getMessage());
    $pages = [];
    $allPages = [];
    $menuPages = [];
    $totalCount = 0;
    $totalPages = 1;
    $totalPagesCount = 0;
    $publishedCount = 0;
    $draftCount = 0;
}

$hasLegalPages = !empty(array_filter($allPages, fn($p) => isset($p['is_legal_page']) && $p['is_legal_page'] == 1));
$flashMessages = get_flash();
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Seiten verwalten</title>
<?php
$adminCssPath = APP_ROOT . '/assets/css/admin.css';
$adminCssVersion = file_exists($adminCssPath) ? filemtime($adminCssPath) : time();
?>
<link rel="stylesheet" href="/assets/css/admin.css?v=<?= $adminCssVersion ?>">
<style>
.search-filter-bar {
    display: flex;
    gap: 15px;
    margin-bottom: 25px;
    flex-wrap: wrap;
    align-items: center;
}

.search-box {
    position: relative;
    flex: 1;
    min-width: 300px;
}

.search-box input {
    width: 100%;
    padding: 12px 45px 12px 15px;
    border: 2px solid #e5e7eb;
    border-radius: 8px;
    font-size: 14px;
    transition: all 0.2s;
}

.search-box input:focus {
    border-color: #667eea;
    outline: none;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
}

.search-box button {
    position: absolute;
    right: 5px;
    top: 50%;
    transform: translateY(-50%);
    background: none;
    border: none;
    cursor: pointer;
    padding: 8px 12px;
    font-size: 18px;
    color: #667eea;
    transition: color 0.2s;
}

.search-box button:hover {
    color: #5568d3;
}

.filter-select {
    padding: 12px 15px;
    border: 2px solid #e5e7eb;
    border-radius: 8px;
    font-size: 14px;
    background: white;
    cursor: pointer;
    transition: all 0.2s;
}

.filter-select:hover {
    border-color: #667eea;
}

.pagination {
    display: flex;
    justify-content: center;
    align-items: center;
    gap: 8px;
    margin-top: 30px;
    padding: 20px 0;
}

.pagination a, .pagination span {
    padding: 10px 15px;
    background: white;
    border: 2px solid #e5e7eb;
    border-radius: 8px;
    text-decoration: none;
    color: #374151;
    transition: all 0.2s;
    min-width: 45px;
    text-align: center;
}

.pagination a:hover {
    border-color: #667eea;
    color: #667eea;
    transform: translateY(-1px);
}

.pagination .active {
    background: #667eea;
    color: white;
    border-color: #667eea;
    font-weight: 600;
}

.pagination .disabled {
    opacity: 0.5;
    cursor: not-allowed;
    pointer-events: none;
}

.results-info {
    text-align: center;
    color: #6b7280;
    font-size: 14px;
    margin-bottom: 20px;
}

.no-results {
    text-align: center;
    padding: 60px 20px;
}

.no-results-icon {
    font-size: 64px;
    margin-bottom: 20px;
}

.no-results h3 {
    color: #6b7280;
    margin-bottom: 10px;
}

.no-results p {
    color: #9ca3af;
    margin-bottom: 30px;
}
</style>
</head>
<body>
    <?php include 'includes/admin_header.php'; ?>
    <?php include 'includes/maintenance_banner.php'; ?>
    
    <div class="admin-container">
        <?php include 'includes/admin_sidebar.php'; ?>
        
        <main class="admin-content">
            <?= CSRF::field() ?>
            
            <?php foreach ($flashMessages as $flash): ?>
                <div class="alert alert-<?= e($flash['type']) ?>">
                    <?= e($flash['message']) ?>
                </div>
            <?php endforeach; ?>
            
            <div class="page-header">
                <h1>📄 Seiten verwalten</h1>
                <p>Alle Seiten bearbeiten, verwalten und organisieren</p>
            </div>
            
            <div class="content-card">
                <div class="card-header">
                    <h2 class="card-title">Alle Seiten (<?= $totalCount ?> Ergebnis<?= $totalCount !== 1 ? 'se' : '' ?>)</h2>
                    <a href="page_create.php" class="btn btn-primary">➕ Neue Seite</a>
                </div>
                
                <!-- Search & Filter Bar -->
                <div class="search-filter-bar">
                    <form method="get" class="search-box">
                        <input type="hidden" name="status" value="<?= e($statusFilter) ?>">
                        <input type="text" 
                               name="search" 
                               placeholder="🔍 Seiten durchsuchen (Titel, Slug, Beschreibung)..." 
                               value="<?= e($search) ?>"
                               autocomplete="off">
                        <button type="submit">🔍</button>
                    </form>
                    
                    <select class="filter-select" 
                            onchange="window.location.href='?status=' + this.value + '<?= $search ? '&search=' . urlencode($search) : '' ?>'">
                        <option value="all" <?= $statusFilter === 'all' ? 'selected' : '' ?>>Alle Status</option>
                        <option value="published" <?= $statusFilter === 'published' ? 'selected' : '' ?>>✓ Veröffentlicht</option>
                        <option value="draft" <?= $statusFilter === 'draft' ? 'selected' : '' ?>>○ Entwürfe</option>
                    </select>
                    
                    <?php if ($search || $statusFilter !== 'all'): ?>
                        <a href="page_manage.php" class="btn btn-secondary">Filter zurücksetzen</a>
                    <?php endif; ?>
                </div>
                
                <!-- Results Info -->
                <?php if ($search || $statusFilter !== 'all'): ?>
                    <div class="results-info">
                        Zeige <?= $totalCount ?> von <?= $totalPagesCount ?> Seiten
                        <?php if ($search): ?>
                            für "<strong><?= e($search) ?></strong>"
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
                
                <?php if (empty($pages)): ?>
                    <div class="no-results">
                        <div class="no-results-icon">
                            <?= $search ? '🔍' : '📄' ?>
                        </div>
                        <h3><?= $search ? 'Keine Seiten gefunden' : 'Noch keine Seiten' ?></h3>
                        <p>
                            <?= $search 
                                ? 'Versuchen Sie einen anderen Suchbegriff oder passen Sie die Filter an.' 
                                : 'Erstellen Sie Ihre erste Seite!' 
                            ?>
                        </p>
                        <?php if (!$search): ?>
                            <a href="page_create.php" class="btn btn-primary">Erste Seite erstellen</a>
                        <?php endif; ?>
                    </div>
                <?php else: ?>
                    <div class="table-container">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th class="icon-column">Icon</th>
                                    <th class="title-column">Titel</th>
                                    <th>Meta</th>
                                    <th class="status-column">Status</th>
                                    <th>Menü</th>
                                    <?php if ($hasLegalPages): ?>
                                        <th>Footer</th>
                                    <?php endif; ?>
                                    <th>Aktionen</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($pages as $page): ?>
                                    <?php 
                                    $isLegal = isset($page['is_legal_page']) && $page['is_legal_page'] == 1;
                                    $isSystem = isset($page['is_system_page']) && $page['is_system_page'] == 1;
                                    $isIndex = $page['slug'] === 'index';
                                    $isDeletable = !$isLegal && !$isSystem && !$isIndex && $page['is_deletable'] != 0;
                                    ?>
                                    <tr data-page-id="<?= $page['id'] ?>">
                                        <td class="icon-column" style="text-align: center; font-size: 24px;">
                                            <?= e($page['page_icon']) ?>
                                        </td>
                                        <td class="title-column">
                                            <div style="display: flex; flex-direction: column; gap: 4px;">
                                                <div style="display: flex; align-items: center; gap: 8px; flex-wrap: wrap;">
                                                    <strong><?= e($page['title']) ?></strong>
                                                    <?php if ($isIndex): ?>
                                                        <span class="badge badge-system">🏠 Startseite</span>
                                                    <?php endif; ?>
                                                    <?php if ($isLegal): ?>
                                                        <span class="badge badge-legal">⚖️ Rechtlich</span>
                                                    <?php endif; ?>
                                                    <?php if ($isSystem && !$isLegal && !$isIndex): ?>
                                                        <span class="badge badge-system">🔒 System</span>
                                                    <?php endif; ?>
                                                </div>
                                                <span style="font-size: 12px; color: #9ca3af;">
                                                    <?= $page['slug'] === 'index' ? '/' : '/' . e($page['slug']) ?>
                                                </span>
                                            </div>
                                        </td>
                                        <td><span style="font-size: 12px; color: #6b7280;"><?= truncate(e($page['meta_description'] ?? ''), 50) ?></span></td>
                                        <td class="status-cell" data-page-id="<?= $page['id'] ?>" data-current-status="<?= e($page['status']) ?>" title="Klicken zum Ändern">
                                            <span class="badge <?= $page['status'] === 'published' ? 'badge-success' : 'badge-warning' ?>">
                                                <?= $page['status'] === 'published' ? '✓ Veröffentlicht' : '○ Entwurf' ?>
                                            </span>
                                        </td>
                                        <td class="menu-toggle" style="text-align: center;">
                                            <label class="switch">
                                                <input type="checkbox" data-page-id="<?= $page['id'] ?>" <?= $page['show_in_menu'] ? 'checked' : '' ?>>
                                                <span class="slider"></span>
                                            </label>
                                        </td>
                                        <?php if ($hasLegalPages): ?>
                                            <td style="text-align: center;">
                                                <?php if ($isLegal): ?>
                                                    <label class="switch footer-toggle">
                                                        <input type="checkbox" 
                                                               data-page-id="<?= $page['id'] ?>" 
                                                               data-field="show_in_footer"
                                                               <?= (isset($page['show_in_footer']) && $page['show_in_footer']) ? 'checked' : '' ?>>
                                                        <span class="slider"></span>
                                                    </label>
                                                <?php else: ?>
                                                    <span style="color: #d1d5db;">—</span>
                                                <?php endif; ?>
                                            </td>
                                        <?php endif; ?>
                                        <td class="actions-column">
                                            <div class="actions">
                                                <a href="page_flexible_edit.php?slug=<?= e($page['slug']) ?>" class="btn btn-sm btn-primary" title="Bearbeiten">✏️</a>
                                                <a href="<?= $page['slug'] === 'index' ? '/' : '/' . e($page['slug']) ?>" class="btn btn-sm btn-secondary" target="_blank" title="Anzeigen">👁️</a>
                                                <?php if ($isDeletable): ?>
                                                    <form method="POST" style="display: inline;" onsubmit="return confirm('Wirklich löschen?');">
                                                        <?= CSRF::field() ?>
                                                        <input type="hidden" name="action" value="delete">
                                                        <input type="hidden" name="page_id" value="<?= $page['id'] ?>">
                                                        <button type="submit" class="btn btn-sm btn-danger" title="Löschen">🗑️</button>
                                                    </form>
                                                <?php else: ?>
                                                    <button class="btn btn-sm btn-secondary" style="opacity: 0.5; cursor: not-allowed;" disabled title="Geschützte Seite">🔒</button>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Pagination -->
                    <?php if ($totalPages > 1): ?>
                        <div class="pagination">
                            <?php if ($page > 1): ?>
                                <a href="?page=1<?= $search ? '&search=' . urlencode($search) : '' ?><?= $statusFilter !== 'all' ? '&status=' . $statusFilter : '' ?>" title="Erste Seite">
                                    ⏮️
                                </a>
                                <a href="?page=<?= $page - 1 ?><?= $search ? '&search=' . urlencode($search) : '' ?><?= $statusFilter !== 'all' ? '&status=' . $statusFilter : '' ?>">
                                    ← Zurück
                                </a>
                            <?php else: ?>
                                <span class="disabled">⏮️</span>
                                <span class="disabled">← Zurück</span>
                            <?php endif; ?>
                            
                            <?php
                            $range = 2;
                            $start = max(1, $page - $range);
                            $end = min($totalPages, $page + $range);
                            
                            if ($start > 1): ?>
                                <a href="?page=1<?= $search ? '&search=' . urlencode($search) : '' ?><?= $statusFilter !== 'all' ? '&status=' . $statusFilter : '' ?>">1</a>
                                <?php if ($start > 2): ?>
                                    <span>...</span>
                                <?php endif; ?>
                            <?php endif; ?>
                            
                            <?php for ($i = $start; $i <= $end; $i++): ?>
                                <a href="?page=<?= $i ?><?= $search ? '&search=' . urlencode($search) : '' ?><?= $statusFilter !== 'all' ? '&status=' . $statusFilter : '' ?>" 
                                   class="<?= $i === $page ? 'active' : '' ?>">
                                    <?= $i ?>
                                </a>
                            <?php endfor; ?>
                            
                            <?php if ($end < $totalPages): ?>
                                <?php if ($end < $totalPages - 1): ?>
                                    <span>...</span>
                                <?php endif; ?>
                                <a href="?page=<?= $totalPages ?><?= $search ? '&search=' . urlencode($search) : '' ?><?= $statusFilter !== 'all' ? '&status=' . $statusFilter : '' ?>"><?= $totalPages ?></a>
                            <?php endif; ?>
                            
                            <?php if ($page < $totalPages): ?>
                                <a href="?page=<?= $page + 1 ?><?= $search ? '&search=' . urlencode($search) : '' ?><?= $statusFilter !== 'all' ? '&status=' . $statusFilter : '' ?>">
                                    Weiter →
                                </a>
                                <a href="?page=<?= $totalPages ?><?= $search ? '&search=' . urlencode($search) : '' ?><?= $statusFilter !== 'all' ? '&status=' . $statusFilter : '' ?>" title="Letzte Seite">
                                    ⏭️
                                </a>
                            <?php else: ?>
                                <span class="disabled">Weiter →</span>
                                <span class="disabled">⏭️</span>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                    
                    <?php if ($hasLegalPages): ?>
                        <div class="alert alert-info" style="margin-top: 20px;">
                            ℹ️ <strong>Rechtliche Pflichtseiten</strong> können nicht gelöscht werden und sind im Footer verfügbar.
                        </div>
                    <?php endif; ?>
                    <div class="alert alert-info" style="margin-top: 10px;">
                        ℹ️ <strong>Startseite (index)</strong> ist eine geschützte System-Seite und kann nicht gelöscht werden.
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- Menu Management -->
            <div class="content-card" style="margin-top: 30px;">
                <div class="card-header">
                    <h2 class="card-title">🔗 Menü-Verwaltung</h2>
                </div>
                
                <?php if (empty($menuPages)): ?>
                    <div class="empty-content">
                        <div style="font-size: 64px; margin-bottom: 20px;">📋</div>
                        <h3 style="color: #6b7280; margin-bottom: 10px;">Keine Seiten im Menü</h3>
                        <p style="color: #9ca3af; margin-bottom: 30px;">Aktiviere Menü-Sichtbarkeit oben.</p>
                    </div>
                <?php else: ?>
                    <div class="table-container">
                        <table class="data-table" id="menu-table">
                            <thead>
                                <tr>
                                    <th style="width: 40px;"></th>
                                    <th class="icon-column" style="width: 50px;"></th>
                                    <th class="title-column">Titel</th>
                                    <th style="width: 200px;">Menü-Label</th>
                                    <th class="actions-column">Aktionen</th>
                                </tr>
                            </thead>
                            <tbody id="menu-items">
                                <?php foreach ($menuPages as $page): ?>
                                    <tr class="menu-row" data-page-id="<?= $page['id'] ?>" draggable="true">
                                        <td style="cursor: move; color: #9ca3af; text-align: center;" class="drag-handle">
                                            <span style="font-size: 18px;">⋮⋮</span>
                                        </td>
                                        <td class="icon-column" style="text-align: center; font-size: 24px;">
                                            <?= e($page['page_icon']) ?>
                                        </td>
                                        <td class="title-column">
                                            <strong><?= e($page['title']) ?></strong><br>
                                            <span style="font-size: 12px; color: #9ca3af;"><?= $page['slug'] === 'index' ? '/' : '/' . e($page['slug']) ?></span>
                                        </td>
                                        <td>
                                            <?php if ($page['menu_label']): ?>
                                                <span style="background: #e0e7ff; color: #4338ca; padding: 4px 10px; border-radius: 6px; font-size: 13px;"><?= e($page['menu_label']) ?></span>
                                            <?php else: ?>
                                                <span style="color: #9ca3af; font-size: 13px;">(nutzt Seitentitel)</span>
                                            <?php endif; ?>
                                            <?php if (isset($page['children']) && count($page['children']) > 0): ?>
                                                <br><span style="font-size: 11px; background: #d1fae5; color: #065f46; padding: 2px 6px; border-radius: 3px;">📁 <?= count($page['children']) ?> Unterseite(n)</span>
                                            <?php endif; ?>
                                        </td>
                                        <td class="actions-column">
                                            <div class="actions">
                                                <button type="button" class="btn btn-sm btn-danger" onclick="removeFromMenu(<?= $page['id'] ?>)">❌</button>
                                                <?php if (isset($page['children']) && !empty($page['children'])): ?>
                                                    <button type="button" class="btn btn-sm btn-info" onclick="toggleSubmenu(<?= $page['id'] ?>)">▼</button>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php if (isset($page['children']) && !empty($page['children'])): ?>
                                        <?php foreach ($page['children'] as $child): ?>
                                            <tr class="submenu-row" data-page-id="<?= $child['id'] ?>" data-parent-id="<?= $page['id'] ?>" style="display: none;">
                                                <td></td>
                                                <td class="icon-column" style="font-size: 20px; text-align: center;">
                                                    <?= e($child['page_icon']) ?>
                                                </td>
                                                <td class="title-column">
                                                    <strong style="color: #6b7280;">↳ <?= e($child['title']) ?></strong><br>
                                                    <span style="font-size: 11px; color: #9ca3af;">/<?= e($child['slug']) ?></span>
                                                </td>
                                                <td style="color: #9ca3af; font-size: 12px;">(Unterseite)</td>
                                                <td class="actions-column">
                                                    <div class="actions">
                                                        <a href="page_flexible_edit.php?slug=<?= e($child['slug']) ?>" class="btn btn-sm btn-primary">✏️</a>
                                                        <button type="button" class="btn btn-sm btn-warning" onclick="makeTopLevel(<?= $child['id'] ?>)">⬆️</button>
                                                        <button type="button" class="btn btn-sm btn-danger" onclick="removeFromSubmenu(<?= $child['id'] ?>)">❌</button>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </main>
    </div>
    
    <script src="/assets/js/main.js"></script>
    <script>
    function showSaveIndicator(success, message) {
        console.log((success ? '✓' : '✗') + ' ' + message);
    }
    
    // Status toggle
    document.addEventListener('click', function(e) {
        const statusCell = e.target.closest('.status-cell');
        if (!statusCell) return;
        
        const pageId = statusCell.dataset.pageId;
        const currentStatus = statusCell.dataset.currentStatus;
        const newStatus = currentStatus === 'published' ? 'draft' : 'published';
        
        const csrfToken = document.querySelector('input[name="csrf_token"]');
        if (!csrfToken) {
            showSaveIndicator(false, 'CSRF Token fehlt');
            return;
        }
        
        const formData = new FormData();
        formData.append('action', 'quick_update');
        formData.append('page_id', pageId);
        formData.append('field', 'status');
        formData.append('value', newStatus);
        formData.append('csrf_token', csrfToken.value);
        
        fetch(window.location.href, {
            method: 'POST',
            body: formData
        })
        .then(response => {
            if (!response.ok) throw new Error('HTTP ' + response.status);
            return response.json();
        })
        .then(data => {
            if (data.success) {
                const badge = statusCell.querySelector('.badge');
                badge.className = 'badge ' + (newStatus === 'published' ? 'badge-success' : 'badge-warning');
                badge.textContent = newStatus === 'published' ? '✓ Veröffentlicht' : '○ Entwurf';
                statusCell.dataset.currentStatus = newStatus;
                showSaveIndicator(true, 'Status: ' + (newStatus === 'published' ? 'Veröffentlicht' : 'Entwurf'));
            } else {
                showSaveIndicator(false, data.error || 'Fehler');
            }
        })
        .catch(error => showSaveIndicator(false, 'Netzwerkfehler: ' + error.message));
    });
    
    // Menu and Footer toggle
    document.addEventListener('change', function(e) {
        if (e.target.type === 'checkbox' && (e.target.closest('.menu-toggle') || e.target.closest('.footer-toggle'))) {
            const pageId = e.target.dataset.pageId;
            const field = e.target.dataset.field || 'show_in_menu';
            const checked = e.target.checked;
            
            const formData = new FormData();
            formData.append('action', 'quick_update');
            formData.append('page_id', pageId);
            formData.append('field', field);
            formData.append('value', checked ? '1' : '0');
            const csrfToken = document.querySelector('input[name="csrf_token"]');
            if (csrfToken) formData.append('csrf_token', csrfToken.value);
            
            fetch(window.location.href, {
                method: 'POST',
                body: formData
            })
            .then(response => {
                if (!response.ok) throw new Error('HTTP ' + response.status);
                return response.json();
            })
            .then(data => {
                if (data.success) {
                    showSaveIndicator(true, field === 'show_in_menu' ? (checked ? 'Im Menü' : 'Aus Menü') : (checked ? 'Im Footer' : 'Aus Footer'));
                    if (field === 'show_in_menu') {
                        setTimeout(() => location.reload(), 500);
                    }
                } else {
                    showSaveIndicator(false, data.error || 'Fehler');
                }
            })
            .catch(error => showSaveIndicator(false, 'Netzwerkfehler: ' + error.message));
        }
    });
    
    async function removeFromMenu(pageId) {
        if (!confirm('Aus Menü entfernen?')) return;
        const formData = new FormData();
        formData.append('action', 'quick_update');
        formData.append('page_id', pageId);
        formData.append('field', 'show_in_menu');
        formData.append('value', '0');
        const csrfToken = document.querySelector('input[name="csrf_token"]');
        if (csrfToken) formData.append('csrf_token', csrfToken.value);
        
        try {
            const response = await fetch(window.location.href, { method: 'POST', body: formData });
            if (response.ok) location.reload();
        } catch (error) {
            showSaveIndicator(false, 'Netzwerkfehler: ' + error.message);
        }
    }
    
    async function makeTopLevel(pageId) {
        if (!confirm('Zur Top-Level machen?')) return;
        const formData = new FormData();
        formData.append('action', 'quick_update');
        formData.append('page_id', pageId);
        formData.append('field', 'parent_id');
        formData.append('value', '');
        const csrfToken = document.querySelector('input[name="csrf_token"]');
        if (csrfToken) formData.append('csrf_token', csrfToken.value);
        
        try {
            const response = await fetch(window.location.href, { method: 'POST', body: formData });
            if (response.ok) location.reload();
        } catch (error) {
            showSaveIndicator(false, 'Netzwerkfehler: ' + error.message);
        }
    }
    
    async function removeFromSubmenu(pageId) {
        if (!confirm('Entfernen?')) return;
        const formData = new FormData();
        formData.append('action', 'quick_update');
        formData.append('page_id', pageId);
        formData.append('field', 'show_in_menu');
        formData.append('value', '0');
        const csrfToken = document.querySelector('input[name="csrf_token"]');
        if (csrfToken) formData.append('csrf_token', csrfToken.value);
        
        try {
            const response = await fetch(window.location.href, { method: 'POST', body: formData });
            if (response.ok) location.reload();
        } catch (error) {
            showSaveIndicator(false, 'Netzwerkfehler: ' + error.message);
        }
    }
    
    function toggleSubmenu(parentId) {
        const subRows = document.querySelectorAll('tr[data-parent-id="' + parentId + '"]');
        const button = event.target;
        const isVisible = subRows[0]?.style.display !== 'none';
        subRows.forEach(row => row.style.display = isVisible ? 'none' : 'table-row');
        button.textContent = isVisible ? '▶' : '▼';
    }
    
    // Drag and drop for menu reordering
    let draggedElement = null;
    
    document.addEventListener('DOMContentLoaded', function() {
        const menuRows = document.querySelectorAll('.menu-row');
        menuRows.forEach(row => {
            row.addEventListener('dragstart', e => {
                draggedElement = e.currentTarget;
                e.currentTarget.classList.add('dragging');
                e.dataTransfer.effectAllowed = 'move';
            });
            row.addEventListener('dragend', e => {
                e.currentTarget.classList.remove('dragging');
                draggedElement = null;
            });
            row.addEventListener('dragover', e => {
                if (e.preventDefault) e.preventDefault();
                e.dataTransfer.dropEffect = 'move';
            });
            row.addEventListener('drop', async e => {
                if (e.stopPropagation) e.stopPropagation();
                if (draggedElement && draggedElement !== e.currentTarget) {
                    const container = document.getElementById('menu-items');
                    const all = Array.from(container.querySelectorAll('.menu-row'));
                    const draggedIndex = all.indexOf(draggedElement);
                    const targetIndex = all.indexOf(e.currentTarget);
                    
                    if (draggedIndex < targetIndex) {
                        e.currentTarget.after(draggedElement);
                    } else {
                        e.currentTarget.before(draggedElement);
                    }
                    
                    const order = Array.from(container.querySelectorAll('.menu-row')).map(row => row.dataset.pageId);
                    const formData = new FormData();
                    formData.append('action', 'reorder');
                    formData.append('order', JSON.stringify(order));
                    const csrfToken = document.querySelector('input[name="csrf_token"]');
                    if (csrfToken) formData.append('csrf_token', csrfToken.value);
                    
                    try {
                        await fetch(window.location.href, { method: 'POST', body: formData });
                        showSaveIndicator(true, 'Reihenfolge gespeichert');
                    } catch (error) {
                        showSaveIndicator(false, 'Netzwerkfehler: ' + error.message);
                    }
                }
            });
        });
    });
    </script>
</body>
</html>