<?php
/**
 * cms-core/public/webview/admin/page_visual_edit.php
 * Visual Page Builder - Mobile-First with Perfect Block Rendering
 */

require_once __DIR__ . '/../../../backend/authorization/check_admin.php';
require_once __DIR__ . '/../../../backend/core/FlexibleBlocks.php';
require_once __DIR__ . '/../../../backend/core/RichTextField.php';

$pageSlug = get_param('slug', '');

if (!$pageSlug) {
    set_flash('error', 'Keine Seite ausgewählt.');
    redirect('/admin/page_manage.php');
    exit;
}

$page = db()->fetchOne("SELECT * FROM pages WHERE slug = ?", [$pageSlug]);

if (!$page) {
    set_flash('error', 'Seite nicht gefunden.');
    redirect('/admin/page_manage.php');
    exit;
}

$pageTitle = $page['title'];
$blocks = FlexibleBlocks::getPageBlocks($pageSlug);
$availableBlockTypes = FlexibleBlocks::getAvailableBlockTypes();

header("Content-Security-Policy: script-src 'self' 'unsafe-inline' https://cdn.jsdelivr.net https://cdn.quilljs.com;");

// AJAX HANDLERS
if (is_post() && get_param('ajax')) {
    while (ob_get_level() > 0) ob_end_clean();
    header('Content-Type: application/json');
    
    try {
        $action = post('action');
        
        switch ($action) {
            case 'update_block':
                $blockId = (int)post('block_id');
                $blockData = json_decode(post('block_data'), true);
                if (!$blockData) throw new Exception('Invalid block data');
                FlexibleBlocks::updateBlock($blockId, $blockData);
                
                $block = db()->fetchOne("SELECT * FROM page_blocks WHERE id = ?", [$blockId]);
                if (!$block) throw new Exception('Block not found');
                $html = FlexibleBlocks::renderBlock($block);
                
                echo json_encode(['success' => true, 'html' => $html, 'block_id' => $blockId]);
                exit;
                
            case 'add_block':
                $blockType = post('block_type');
                if (!isset($availableBlockTypes[$blockType])) throw new Exception('Invalid block type');
                $blockId = FlexibleBlocks::addBlock($pageSlug, $blockType);
                
                $block = db()->fetchOne("SELECT * FROM page_blocks WHERE id = ?", [$blockId]);
                if (!$block) throw new Exception('Failed to create block');
                $html = FlexibleBlocks::renderBlock($block);
                
                echo json_encode(['success' => true, 'block_id' => $blockId, 'html' => $html]);
                exit;
                
            case 'delete_block':
                $blockId = (int)post('block_id');
                FlexibleBlocks::deleteBlock($blockId);
                echo json_encode(['success' => true]);
                exit;
                
            case 'reorder_blocks':
                $blockIds = json_decode(post('block_ids'), true);
                if (!is_array($blockIds)) throw new Exception('Invalid block IDs');
                $indexed = array_combine(range(0, count($blockIds) - 1), $blockIds);
                FlexibleBlocks::reorderBlocks($pageSlug, $indexed);
                echo json_encode(['success' => true]);
                exit;
                
            case 'get_block_form':
                $blockId = (int)post('block_id');
                $block = db()->fetchOne("SELECT * FROM page_blocks WHERE id = ?", [$blockId]);
                if (!$block) throw new Exception('Block not found');
                $data = json_decode($block['block_data'] ?? '{}', true);
                $blockTypeKey = $block['block_type_template'] ?? '';
                $blockType = $availableBlockTypes[$blockTypeKey] ?? null;
                
                ob_start();
                ?>
                <div class="block-edit-header">
                    <h3><?= e($blockType['name'] ?? 'Block') ?></h3>
                    <button type="button" onclick="closeBlockEditor()" class="btn-close">×</button>
                </div>
                
                <form id="block-edit-form" data-block-id="<?= $blockId ?>">
                    <?php 
                    function renderFieldEditor($key, $field, $value, $blockId, $parentPath = []) {
                        $fieldName = $key;
                        if (!empty($parentPath)) {
                            $fieldName = implode('[', $parentPath) . '][' . $key . ']';
                        }
                        $fieldId = 'field_' . $blockId . '_' . implode('_', array_merge($parentPath, [$key]));
                        
                        // RichText fields
                        if (in_array($field['type'], ['richtext', 'title', 'subtitle'])) {
                            $richValue = $value;
                            if (is_string($richValue)) {
                                $decoded = @json_decode($richValue, true);
                                $richValue = (is_array($decoded) && isset($decoded['text'])) ? $decoded : ['text' => $richValue];
                            } elseif (!is_array($richValue)) {
                                $richValue = ['text' => ''];
                            }
                            echo RichTextField::render($fieldId, $fieldName, $richValue, $field['label'], $field['required'] ?? false);
                        }
                        // Text/Textarea
                        elseif (in_array($field['type'], ['text', 'textarea'])) {
                            ?>
                            <div class="form-group">
                                <label class="form-label"><?= e($field['label'] ?? $key) ?></label>
                                <?php if ($field['type'] === 'textarea'): ?>
                                    <textarea name="<?= e($fieldName) ?>" class="form-input" rows="4"><?= e($value ?? '') ?></textarea>
                                <?php else: ?>
                                    <input type="text" name="<?= e($fieldName) ?>" class="form-input" value="<?= e($value ?? '') ?>">
                                <?php endif; ?>
                            </div>
                            <?php
                        }
                        // URL
                        elseif ($field['type'] === 'url') {
                            ?>
                            <div class="form-group">
                                <label class="form-label"><?= e($field['label'] ?? $key) ?></label>
                                <input type="url" name="<?= e($fieldName) ?>" class="form-input" value="<?= e($value ?? '') ?>">
                            </div>
                            <?php
                        }
                        // Checkbox
                        elseif ($field['type'] === 'checkbox') {
                            ?>
                            <div class="form-group">
                                <label style="display: flex; align-items: center; gap: 12px; cursor: pointer;">
                                    <input type="checkbox" name="<?= e($fieldName) ?>" value="1" <?= $value ? 'checked' : '' ?> class="form-checkbox">
                                    <span><?= e($field['label'] ?? $key) ?></span>
                                </label>
                            </div>
                            <?php
                        }
                        // Select
                        elseif ($field['type'] === 'select') {
                            ?>
                            <div class="form-group">
                                <label class="form-label"><?= e($field['label'] ?? $key) ?></label>
                                <select name="<?= e($fieldName) ?>" class="form-select">
                                    <?php foreach ($field['options'] ?? [] as $optKey => $optLabel): ?>
                                        <option value="<?= e($optKey) ?>" <?= ($value === $optKey) ? 'selected' : '' ?>>
                                            <?= e($optLabel) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <?php
                        }
                        // Image
                        elseif ($field['type'] === 'image') {
                            ?>
                            <div class="form-group">
                                <label class="form-label"><?= e($field['label'] ?? $key) ?></label>
                                <input type="text" name="<?= e($fieldName) ?>" class="form-input" value="<?= e($value ?? '') ?>" placeholder="/media/pages/...">
                                <p style="font-size: 12px; color: #6b7280; margin-top: 4px;">Enter image path (e.g., /media/pages/image.jpg)</p>
                            </div>
                            <?php
                        }
                        // Repeater
                        elseif ($field['type'] === 'repeater') {
                            $items = is_array($value) ? $value : [];
                            ?>
                            <div class="form-group">
                                <label class="form-label"><?= e($field['label'] ?? $key) ?></label>
                                <div class="repeater-container" data-repeater-key="<?= e($key) ?>" data-block-id="<?= $blockId ?>">
                                    <?php foreach ($items as $idx => $item): ?>
                                        <div class="repeater-item" data-index="<?= $idx ?>">
                                            <div class="repeater-header">
                                                <span style="font-weight: 600;">Item <?= $idx + 1 ?></span>
                                                <button type="button" class="btn-remove" onclick="removeRepeaterItem(this)">×</button>
                                            </div>
                                            <div class="repeater-fields">
                                                <?php foreach ($field['fields'] ?? [] as $subKey => $subField): ?>
                                                    <?php renderFieldEditor($subKey, $subField, $item[$subKey] ?? null, $blockId, array_merge($parentPath, [$key, $idx])); ?>
                                                <?php endforeach; ?>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                    
                                    <!-- Template for new items -->
                                    <template class="repeater-template">
                                        <div class="repeater-item" data-index="__INDEX__">
                                            <div class="repeater-header">
                                                <span style="font-weight: 600;">Item __NUM__</span>
                                                <button type="button" class="btn-remove" onclick="removeRepeaterItem(this)">×</button>
                                            </div>
                                            <div class="repeater-fields">
                                                <?php foreach ($field['fields'] ?? [] as $subKey => $subField): ?>
                                                    <?php 
                                                    $templateFieldName = $key . '[__INDEX__][' . $subKey . ']';
                                                    ?>
                                                    
                                                    <?php if (in_array($subField['type'], ['richtext', 'title', 'subtitle'])): ?>
                                                        <div class="form-group">
                                                            <label class="form-label"><?= e($subField['label']) ?></label>
                                                            <input type="text" name="<?= e($templateFieldName) ?>" class="form-input" placeholder="Enter text...">
                                                            <p style="font-size: 11px; color: #9ca3af; margin-top: 4px;">⚠️ Full RichText in classic editor</p>
                                                        </div>
                                                    
                                                    <?php elseif ($subField['type'] === 'text'): ?>
                                                        <div class="form-group">
                                                            <label class="form-label"><?= e($subField['label']) ?></label>
                                                            <input type="text" name="<?= e($templateFieldName) ?>" class="form-input">
                                                        </div>
                                                    
                                                    <?php elseif ($subField['type'] === 'textarea'): ?>
                                                        <div class="form-group">
                                                            <label class="form-label"><?= e($subField['label']) ?></label>
                                                            <textarea name="<?= e($templateFieldName) ?>" class="form-input" rows="3"></textarea>
                                                        </div>
                                                    
                                                    <?php elseif ($subField['type'] === 'url'): ?>
                                                        <div class="form-group">
                                                            <label class="form-label"><?= e($subField['label']) ?></label>
                                                            <input type="url" name="<?= e($templateFieldName) ?>" class="form-input">
                                                        </div>
                                                    
                                                    <?php elseif ($subField['type'] === 'image'): ?>
                                                        <div class="form-group">
                                                            <label class="form-label"><?= e($subField['label']) ?></label>
                                                            <input type="text" name="<?= e($templateFieldName) ?>" class="form-input" placeholder="/media/pages/...">
                                                        </div>
                                                    <?php endif; ?>
                                                <?php endforeach; ?>
                                            </div>
                                        </div>
                                    </template>
                                    
                                    <button type="button" class="btn-add" onclick="addRepeaterItem(this)">
                                        + Add <?= e($field['add_button'] ?? 'Item') ?>
                                    </button>
                                </div>
                            </div>
                            <?php
                        }
                    }
                    
                    foreach ($blockType['fields'] ?? [] as $key => $field) {
                        renderFieldEditor($key, $field, $data[$key] ?? null, $blockId);
                    }
                    ?>
                    
                    <div class="form-actions">
                        <button type="submit" class="btn-save-block">💾 Save Changes</button>
                    </div>
                </form>
                <?php
                $formHtml = ob_get_clean();
                
                echo json_encode(['success' => true, 'formHtml' => $formHtml]);
                exit;
                
            default:
                throw new Exception('Unknown action');
        }
    } catch (Exception $e) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        exit;
    }
}

?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Visual Editor - <?= e($pageTitle) ?></title>
    
    <!-- CRITICAL: Load ALL frontend CSS in correct order for proper block rendering -->
    <link rel="stylesheet" href="/cms-core/public/webview/global-styles.php?v=<?= time() ?>">
    <link rel="stylesheet" href="/cms-core/public/webview/assets/css/flexible-blocks.css?v=<?= time() ?>">
    <link rel="stylesheet" href="/assets/css/style.css?v=<?= time() ?>">
    
    <style>
    /* ==================== RESET & VARIABLES ==================== */
    * {
        margin: 0;
        padding: 0;
        box-sizing: border-box;
        -webkit-tap-highlight-color: transparent;
    }
    
    :root {
        --primary: #667eea;
        --primary-dark: #5a67d8;
        --header-height: 60px;
        --sidebar-width: 420px;
    }
    
    body {
        font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
        overflow: hidden;
        background: #f5f7fa;
    }
    
    /* ==================== HEADER ==================== */
    .editor-header {
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        height: var(--header-height);
        background: linear-gradient(135deg, var(--primary), var(--primary-dark));
        color: white;
        display: flex;
        align-items: center;
        justify-content: space-between;
        padding: 0 16px;
        z-index: 1000;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    }
    
    .editor-title {
        font-size: 16px;
        font-weight: 600;
        flex: 1;
        padding: 0 12px;
        white-space: nowrap;
        overflow: hidden;
        text-overflow: ellipsis;
    }
    
    .header-btn {
        width: 44px;
        height: 44px;
        border-radius: 12px;
        border: none;
        background: rgba(255,255,255,0.2);
        color: white;
        font-size: 20px;
        cursor: pointer;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: all 0.2s;
        flex-shrink: 0;
    }
    
    .header-btn:active {
        transform: scale(0.95);
        background: rgba(255,255,255,0.3);
    }
    
    .header-btn.back {
        background: white;
        color: var(--primary);
        font-weight: 600;
    }
    
    /* ==================== SIDEBAR (Mobile Bottom Sheet) ==================== */
    .sidebar-overlay {
        position: fixed;
        inset: 0;
        background: rgba(0,0,0,0.4);
        opacity: 0;
        pointer-events: none;
        transition: opacity 0.3s;
        z-index: 998;
    }
    
    .sidebar-overlay.active {
        opacity: 1;
        pointer-events: auto;
    }
    
    .sidebar {
        position: fixed;
        bottom: 0;
        left: 0;
        right: 0;
        background: white;
        border-radius: 20px 20px 0 0;
        transform: translateY(100%);
        transition: transform 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        z-index: 999;
        max-height: 90vh;
        display: flex;
        flex-direction: column;
        box-shadow: 0 -4px 20px rgba(0,0,0,0.15);
    }
    
    .sidebar.open {
        transform: translateY(0);
    }
    
    .sidebar-handle {
        width: 40px;
        height: 4px;
        background: #d1d5db;
        border-radius: 2px;
        margin: 12px auto 8px;
        cursor: grab;
    }
    
    .sidebar-tabs {
        display: flex;
        border-bottom: 1px solid #e5e7eb;
        background: #f9fafb;
        padding: 0 16px;
    }
    
    .sidebar-tab {
        flex: 1;
        padding: 16px 8px;
        text-align: center;
        border: none;
        background: none;
        font-size: 14px;
        font-weight: 600;
        color: #6b7280;
        cursor: pointer;
        position: relative;
        transition: color 0.2s;
    }
    
    .sidebar-tab.active {
        color: var(--primary);
    }
    
    .sidebar-tab.active::after {
        content: '';
        position: absolute;
        bottom: -1px;
        left: 0;
        right: 0;
        height: 3px;
        background: var(--primary);
    }
    
    .sidebar-content {
        flex: 1;
        overflow-y: auto;
        padding: 20px 16px;
        -webkit-overflow-scrolling: touch;
    }
    
    .tab-content {
        display: none;
    }
    
    .tab-content.active {
        display: block;
    }
    
    /* ==================== BLOCK TYPES GRID ==================== */
    .block-types-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(140px, 1fr));
        gap: 12px;
    }
    
    .block-type-card {
        padding: 20px 16px;
        border: 2px solid #e5e7eb;
        border-radius: 12px;
        text-align: center;
        background: white;
        cursor: pointer;
        transition: all 0.2s;
    }
    
    .block-type-card:active {
        transform: scale(0.95);
        border-color: var(--primary);
        background: #f0f4ff;
    }
    
    .block-type-icon {
        font-size: 36px;
        margin-bottom: 8px;
    }
    
    .block-type-name {
        font-size: 13px;
        font-weight: 600;
        color: #1f2937;
    }
    
    /* ==================== BLOCK EDITOR ==================== */
    .block-edit-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding-bottom: 16px;
        border-bottom: 2px solid #e5e7eb;
        margin-bottom: 20px;
    }
    
    .block-edit-header h3 {
        font-size: 18px;
        font-weight: 700;
        color: #1f2937;
    }
    
    .btn-close {
        width: 32px;
        height: 32px;
        border-radius: 8px;
        border: none;
        background: #fee;
        color: #e11d48;
        font-size: 24px;
        cursor: pointer;
        line-height: 1;
    }
    
    /* ==================== FORM ELEMENTS ==================== */
    .form-group {
        margin-bottom: 20px;
    }
    
    .form-label {
        display: block;
        font-weight: 600;
        margin-bottom: 8px;
        font-size: 14px;
        color: #374151;
    }
    
    .form-input,
    .form-textarea,
    .form-select {
        width: 100%;
        padding: 12px;
        border: 2px solid #e5e7eb;
        border-radius: 10px;
        font-size: 15px;
        transition: border-color 0.2s;
    }
    
    .form-input:focus,
    .form-textarea:focus,
    .form-select:focus {
        outline: none;
        border-color: var(--primary);
    }
    
    .form-textarea {
        min-height: 100px;
        resize: vertical;
        font-family: inherit;
    }
    
    .form-checkbox {
        width: 20px;
        height: 20px;
        cursor: pointer;
    }
    
    /* ==================== REPEATER ==================== */
    .repeater-container {
        border: 2px solid #e5e7eb;
        border-radius: 12px;
        padding: 16px;
        background: #f9fafb;
    }
    
    .repeater-item {
        background: white;
        border: 1px solid #e5e7eb;
        border-radius: 10px;
        padding: 16px;
        margin-bottom: 12px;
    }
    
    .repeater-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding-bottom: 12px;
        margin-bottom: 12px;
        border-bottom: 1px solid #e5e7eb;
    }
    
    .btn-remove {
        width: 32px;
        height: 32px;
        border-radius: 8px;
        border: none;
        background: #fee;
        color: #e11d48;
        font-size: 20px;
        font-weight: 600;
        cursor: pointer;
        line-height: 1;
    }
    
    .btn-add {
        width: 100%;
        padding: 12px;
        border-radius: 10px;
        border: 2px dashed #d1d5db;
        background: white;
        color: #6b7280;
        font-size: 14px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.2s;
    }
    
    .btn-add:active {
        background: #f9fafb;
        border-color: var(--primary);
        color: var(--primary);
    }
    
    .form-actions {
        margin-top: 24px;
        padding-top: 20px;
        border-top: 2px solid #e5e7eb;
    }
    
    .btn-save-block {
        width: 100%;
        padding: 16px;
        border-radius: 12px;
        border: none;
        background: var(--primary);
        color: white;
        font-size: 16px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.2s;
    }
    
    .btn-save-block:active {
        transform: scale(0.98);
        background: var(--primary-dark);
    }
    
    /* ==================== PREVIEW AREA ==================== */
    .preview-area {
        position: fixed;
        top: var(--header-height);
        left: 0;
        right: 0;
        bottom: 0;
        overflow-y: auto;
        -webkit-overflow-scrolling: touch;
        background: #e5e7eb;
        padding: 20px 12px;
    }
    
    .preview-toolbar {
        position: sticky;
        top: 0;
        background: white;
        padding: 10px 12px;
        border-radius: 12px;
        margin-bottom: 20px;
        display: flex;
        justify-content: center;
        gap: 8px;
        z-index: 10;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
    }
    
    .preview-mode-btn {
        padding: 10px 20px;
        border: 2px solid #e5e7eb;
        background: white;
        border-radius: 10px;
        font-size: 12px;
        font-weight: 600;
        color: #6b7280;
        cursor: pointer;
        transition: all 0.2s;
    }
    
    .preview-mode-btn.active {
        border-color: var(--primary);
        background: var(--primary);
        color: white;
    }
    
    .preview-frame {
        background: white;
        margin: 0 auto;
        min-height: 600px;
        transition: all 0.3s;
        box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        /* CRITICAL: Force iframe-like behavior for media queries */
        container-type: inline-size;
        container-name: preview;
    }
    
    /* Device Frames */
    .preview-frame.mobile {
        width: 390px;
        max-width: 100%;
        border-radius: 30px;
        overflow: hidden;
    }
    
    .preview-frame.tablet {
        width: 390px; /* Same as mobile on small screens */
        max-width: 100%;
        border-radius: 30px;
        overflow: hidden;
        transform: scale(0.85); /* Scale down to show tablet proportions */
        transform-origin: top center;
    }
    
    .preview-frame.desktop {
        width: 390px; /* Same as mobile on small screens */
        max-width: 100%;
        border-radius: 30px;
        overflow: hidden;
        transform: scale(0.7); /* Scale down more to show desktop proportions */
        transform-origin: top center;
    }
    
    /* ==================== BLOCK DISPLAY WITH PROPER RENDERING ==================== */
    .visual-block {
        position: relative;
        min-height: 60px;
        transition: all 0.2s;
    }
    
    /* CRITICAL: Don't interfere with block content styling */
    .visual-block > *:not(.block-overlay):not(.block-actions) {
        pointer-events: none; /* Prevent clicks on content, only on block wrapper */
    }
    
    .visual-block.selected {
        outline: 3px solid var(--primary);
        outline-offset: 2px;
    }
    
    .block-overlay {
        position: absolute;
        top: 12px;
        right: 12px;
        background: var(--primary);
        color: white;
        padding: 8px 16px;
        border-radius: 10px;
        font-size: 11px;
        font-weight: 600;
        opacity: 0;
        pointer-events: none;
        z-index: 100;
        transition: opacity 0.2s;
    }
    
    .visual-block.selected .block-overlay,
    .visual-block:hover .block-overlay {
        opacity: 1;
    }
    
    .block-actions {
        position: absolute;
        top: 12px;
        left: 12px;
        display: flex;
        gap: 8px;
        opacity: 0;
        z-index: 101;
        transition: opacity 0.2s;
        pointer-events: none;
    }
    
    .visual-block.selected .block-actions,
    .visual-block:hover .block-actions {
        opacity: 1;
        pointer-events: auto;
    }
    
    .block-action-btn {
        width: 44px;
        height: 44px;
        border-radius: 12px;
        border: none;
        background: white;
        box-shadow: 0 2px 8px rgba(0,0,0,0.15);
        font-size: 18px;
        cursor: pointer;
        transition: all 0.2s;
        pointer-events: auto;
    }
    
    .block-action-btn:active {
        transform: scale(0.9);
    }
    
    .block-action-btn.danger {
        background: #fee;
        color: #e11d48;
    }
    
    /* ==================== LOADING ==================== */
    .loading-overlay {
        position: fixed;
        inset: 0;
        background: rgba(0,0,0,0.6);
        display: none;
        align-items: center;
        justify-content: center;
        z-index: 9999;
    }
    
    .loading-overlay.active {
        display: flex;
    }
    
    .loading-spinner {
        width: 60px;
        height: 60px;
        border: 5px solid rgba(255,255,255,0.2);
        border-top-color: white;
        border-radius: 50%;
        animation: spin 0.8s linear infinite;
    }
    
    @keyframes spin {
        to { transform: rotate(360deg); }
    }
    
    /* ==================== EMPTY STATE ==================== */
    .empty-state {
        text-align: center;
        padding: 80px 20px;
        color: #9ca3af;
    }
    
    .empty-state-icon {
        font-size: 64px;
        margin-bottom: 16px;
    }
    
    .empty-state h3 {
        font-size: 20px;
        font-weight: 600;
        margin-bottom: 8px;
        color: #6b7280;
    }
    
    .empty-state p {
        font-size: 14px;
    }
    
    /* ==================== DESKTOP VIEW ==================== */
    @media (min-width: 1024px) {
        .header-btn.menu {
            display: none;
        }
        
        .sidebar {
            position: fixed;
            left: 0;
            top: var(--header-height);
            bottom: 0;
            width: var(--sidebar-width);
            max-height: none;
            border-radius: 0;
            transform: none;
        }
        
        .sidebar-handle {
            display: none;
        }
        
        .preview-area {
            left: var(--sidebar-width);
        }
        
        .sidebar-overlay {
            display: none;
        }
        
        .preview-toolbar {
            position: static;
        }
        
        /* Desktop: Show actual device sizes */
        .preview-frame.mobile {
            width: 390px;
            border-radius: 30px;
            overflow: hidden;
        }
        
        .preview-frame.tablet {
            width: 768px;
            border-radius: 16px;
            overflow: hidden;
            transform: none; /* Remove scale on desktop */
        }
        
        .preview-frame.desktop {
            width: 100%;
            max-width: 1200px;
            border-radius: 0;
            transform: none; /* Remove scale on desktop */
        }
    }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="editor-header">
        <button class="header-btn menu" onclick="toggleSidebar()">☰</button>
        <div class="editor-title">🎨 <?= e($pageTitle) ?></div>
        <button class="header-btn back" onclick="location.href='/admin/page_manage.php'">←</button>
    </div>
    
    <!-- Sidebar Overlay -->
    <div class="sidebar-overlay" onclick="closeSidebar()"></div>
    
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-handle"></div>
        
        <div class="sidebar-tabs">
            <button class="sidebar-tab active" onclick="switchTab('blocks')">📦 Blocks</button>
            <button class="sidebar-tab" onclick="switchTab('settings')">⚙️ Edit</button>
        </div>
        
        <div class="sidebar-content">
            <!-- Blocks Tab -->
            <div id="tab-blocks" class="tab-content active">
                <h3 style="margin-bottom: 16px; font-size: 16px; font-weight: 700;">Add Block</h3>
                <div class="block-types-grid">
                    <?php foreach ($availableBlockTypes as $key => $type): ?>
                        <div class="block-type-card" onclick="addBlock('<?= e($key) ?>')">
                            <div class="block-type-icon"><?= $type['icon'] ?? '📦' ?></div>
                            <div class="block-type-name"><?= e($type['name']) ?></div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
            
            <!-- Settings Tab -->
            <div id="tab-settings" class="tab-content">
                <div id="block-settings">
                    <p style="color: #9ca3af; text-align: center; padding: 60px 20px; font-size: 14px;">
                        Tap a block to edit
                    </p>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Preview Area -->
    <div class="preview-area">
        <div class="preview-toolbar">
            <button class="preview-mode-btn active" onclick="setPreviewMode('mobile')">📱 Mobile</button>
            <button class="preview-mode-btn" onclick="setPreviewMode('tablet')">📱 Tablet</button>
            <button class="preview-mode-btn" onclick="setPreviewMode('desktop')">🖥️ Desktop</button>
        </div>
        
        <div class="preview-frame mobile" id="preview-blocks">
            <?php if (!empty($blocks)): ?>
                <?php foreach ($blocks as $block): ?>
                    <div class="visual-block" 
                         data-block-id="<?= $block['id'] ?>"
                         data-block-type="<?= e($block['block_type_template']) ?>"
                         onclick="selectBlock(<?= $block['id'] ?>)">
                        <div class="block-overlay">
                            <?= $availableBlockTypes[$block['block_type_template']]['icon'] ?? '📦' ?>
                            <?= e($availableBlockTypes[$block['block_type_template']]['name'] ?? 'Block') ?>
                        </div>
                        <div class="block-actions">
                            <button class="block-action-btn" onclick="event.stopPropagation(); editBlock(<?= $block['id'] ?>)" title="Edit">✏️</button>
                            <button class="block-action-btn danger" onclick="event.stopPropagation(); deleteBlock(<?= $block['id'] ?>)" title="Delete">🗑️</button>
                        </div>
                        <?php 
                        $rendered = FlexibleBlocks::renderBlock($block);
                        echo $rendered ?: '<div style="padding: 40px; text-align: center; color: #ef4444; border: 2px dashed #ef4444;">⚠️ Render Error</div>';
                        ?>
                    </div>
                <?php endforeach; ?>
            <?php else: ?>
                <div class="empty-state">
                    <div class="empty-state-icon">📦</div>
                    <h3>No blocks yet</h3>
                    <p>Tap the ☰ menu to add your first block</p>
                </div>
            <?php endif; ?>
        </div>
    </div>
    
    <!-- Loading Overlay -->
    <div class="loading-overlay" id="loading">
        <div class="loading-spinner"></div>
    </div>
    
    <script>
    const pageSlug = '<?= e($pageSlug) ?>';
    let currentBlockId = null;
    
    // ==================== SIDEBAR ====================
    function toggleSidebar() {
        const sidebar = document.querySelector('.sidebar');
        const overlay = document.querySelector('.sidebar-overlay');
        sidebar.classList.toggle('open');
        overlay.classList.toggle('active');
    }
    
    function closeSidebar() {
        const sidebar = document.querySelector('.sidebar');
        const overlay = document.querySelector('.sidebar-overlay');
        sidebar.classList.remove('open');
        overlay.classList.remove('active');
    }
    
    function switchTab(tab) {
        document.querySelectorAll('.sidebar-tab').forEach(t => t.classList.remove('active'));
        document.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));
        
        event.target.classList.add('active');
        document.getElementById(`tab-${tab}`).classList.add('active');
    }
    
    // ==================== PREVIEW MODE ====================
    function setPreviewMode(mode) {
        const frame = document.querySelector('.preview-frame');
        const buttons = document.querySelectorAll('.preview-mode-btn');
        
        buttons.forEach(btn => btn.classList.remove('active'));
        event.target.classList.add('active');
        
        frame.className = `preview-frame ${mode}`;
        
        // CRITICAL: Force media queries to re-evaluate by toggling a class on body
        // This helps blocks re-render with correct mobile/tablet/desktop styles
        document.body.className = `preview-mode-${mode}`;
        
        // Trigger a resize event to force media query re-evaluation
        window.dispatchEvent(new Event('resize'));
    }
    
    // ==================== BLOCK SELECTION ====================
    function selectBlock(blockId) {
        document.querySelectorAll('.visual-block').forEach(b => b.classList.remove('selected'));
        const block = document.querySelector(`[data-block-id="${blockId}"]`);
        if (block) {
            block.classList.add('selected');
            // Scroll into view with smooth behavior
            block.scrollIntoView({ behavior: 'smooth', block: 'center' });
        }
        currentBlockId = blockId;
    }
    
    // ==================== ADD BLOCK ====================
    async function addBlock(blockType) {
        showLoading();
        closeSidebar();
        
        try {
            const formData = new FormData();
            formData.append('action', 'add_block');
            formData.append('block_type', blockType);
            
            const response = await fetch(`${window.location.href}&ajax=1`, {
                method: 'POST',
                body: formData
            });
            
            const result = await response.json();
            if (!result.success) throw new Error(result.error);
            
            // Insert new block into DOM with proper rendering
            const previewBlocks = document.getElementById('preview-blocks');
            
            // Remove empty state if exists
            const emptyState = previewBlocks.querySelector('.empty-state');
            if (emptyState) emptyState.remove();
            
            // Create new block element
            const blockDiv = document.createElement('div');
            blockDiv.className = 'visual-block';
            blockDiv.dataset.blockId = result.block_id;
            blockDiv.dataset.blockType = blockType;
            blockDiv.onclick = () => selectBlock(result.block_id);
            
            // Get block type info
            const blockTypes = <?= json_encode($availableBlockTypes) ?>;
            const blockTypeInfo = blockTypes[blockType] || {};
            
            blockDiv.innerHTML = `
                <div class="block-overlay">
                    ${blockTypeInfo.icon || '📦'}
                    ${blockTypeInfo.name || 'Block'}
                </div>
                <div class="block-actions">
                    <button class="block-action-btn" onclick="event.stopPropagation(); editBlock(${result.block_id})" title="Edit">✏️</button>
                    <button class="block-action-btn danger" onclick="event.stopPropagation(); deleteBlock(${result.block_id})" title="Delete">🗑️</button>
                </div>
                ${result.html}
            `;
            
            previewBlocks.appendChild(blockDiv);
            
            // Auto-select and scroll to new block
            setTimeout(() => {
                selectBlock(result.block_id);
                editBlock(result.block_id);
            }, 100);
            
        } catch (error) {
            alert('Error: ' + error.message);
        } finally {
            hideLoading();
        }
    }
    
    // ==================== EDIT BLOCK ====================
    async function editBlock(blockId) {
        showLoading();
        selectBlock(blockId);
        
        try {
            const formData = new FormData();
            formData.append('action', 'get_block_form');
            formData.append('block_id', blockId);
            
            const response = await fetch(`${window.location.href}&ajax=1`, {
                method: 'POST',
                body: formData
            });
            
            const result = await response.json();
            if (!result.success) throw new Error(result.error);
            
            document.getElementById('block-settings').innerHTML = result.formHtml;
            
            // Attach form submit handler
            const form = document.getElementById('block-edit-form');
            form.onsubmit = async (e) => {
                e.preventDefault();
                await saveBlockData(blockId);
            };
            
            // Switch to settings tab
            document.querySelector('.sidebar-tab[onclick*="settings"]').click();
            
            // Open sidebar on mobile
            if (window.innerWidth < 1024) {
                document.querySelector('.sidebar').classList.add('open');
                document.querySelector('.sidebar-overlay').classList.add('active');
            }
        } catch (error) {
            alert('Error: ' + error.message);
        } finally {
            hideLoading();
        }
    }
    
    function closeBlockEditor() {
        document.getElementById('block-settings').innerHTML = '<p style="color: #9ca3af; text-align: center; padding: 60px 20px; font-size: 14px;">Tap a block to edit</p>';
        document.querySelectorAll('.visual-block').forEach(b => b.classList.remove('selected'));
        closeSidebar();
    }
    
    // ==================== SAVE BLOCK ====================
    async function saveBlockData(blockId) {
        showLoading();
        
        try {
            const form = document.getElementById('block-edit-form');
            const formData = new FormData(form);
            
            // Convert FormData to nested object
            const data = {};
            for (const [key, value] of formData.entries()) {
                setNestedValue(data, key, value);
            }
            
            const saveData = new FormData();
            saveData.append('action', 'update_block');
            saveData.append('block_id', blockId);
            saveData.append('block_data', JSON.stringify(data));
            
            const response = await fetch(`${window.location.href}&ajax=1`, {
                method: 'POST',
                body: saveData
            });
            
            const result = await response.json();
            if (!result.success) throw new Error(result.error);
            
            // Update block in DOM without page reload
            const blockEl = document.querySelector(`[data-block-id="${blockId}"]`);
            if (blockEl) {
                // Keep the overlay and actions
                const overlay = blockEl.querySelector('.block-overlay');
                const actions = blockEl.querySelector('.block-actions');
                
                // Create temp container
                const temp = document.createElement('div');
                temp.innerHTML = result.html;
                
                // Clear block content but keep structure
                while (blockEl.firstChild) {
                    if (blockEl.firstChild === overlay || blockEl.firstChild === actions) {
                        break;
                    }
                    blockEl.removeChild(blockEl.firstChild);
                }
                
                // Insert new content before overlay
                blockEl.insertBefore(temp.firstChild, overlay);
                
                // Flash to indicate update
                blockEl.style.transition = 'background 0.3s';
                blockEl.style.background = '#f0f9ff';
                setTimeout(() => {
                    blockEl.style.background = '';
                }, 300);
            }
            
            // Show success message
            const saveBtn = form.querySelector('.btn-save-block');
            const originalText = saveBtn.textContent;
            saveBtn.textContent = '✅ Saved!';
            saveBtn.style.background = '#10b981';
            
            setTimeout(() => {
                saveBtn.textContent = originalText;
                saveBtn.style.background = '';
            }, 2000);
            
        } catch (error) {
            alert('Error: ' + error.message);
        } finally {
            hideLoading();
        }
    }
    
    function setNestedValue(obj, path, value) {
        const keys = path.replace(/\]/g, '').split('[');
        let current = obj;
        
        for (let i = 0; i < keys.length - 1; i++) {
            const key = keys[i];
            if (!current[key]) {
                current[key] = isNaN(keys[i + 1]) ? {} : [];
            }
            current = current[key];
        }
        
        current[keys[keys.length - 1]] = value;
    }
    
    // ==================== DELETE BLOCK ====================
    async function deleteBlock(blockId) {
        if (!confirm('Delete this block? This cannot be undone.')) return;
        
        showLoading();
        
        try {
            const formData = new FormData();
            formData.append('action', 'delete_block');
            formData.append('block_id', blockId);
            
            const response = await fetch(`${window.location.href}&ajax=1`, {
                method: 'POST',
                body: formData
            });
            
            const result = await response.json();
            if (!result.success) throw new Error(result.error);
            
            // Remove from DOM with animation
            const blockEl = document.querySelector(`[data-block-id="${blockId}"]`);
            if (blockEl) {
                blockEl.style.transition = 'opacity 0.3s, transform 0.3s';
                blockEl.style.opacity = '0';
                blockEl.style.transform = 'scale(0.9)';
                
                setTimeout(() => {
                    blockEl.remove();
                    
                    // Show empty state if no blocks left
                    const previewBlocks = document.getElementById('preview-blocks');
                    if (!previewBlocks.querySelector('.visual-block')) {
                        previewBlocks.innerHTML = `
                            <div class="empty-state">
                                <div class="empty-state-icon">📦</div>
                                <h3>No blocks yet</h3>
                                <p>Tap the ☰ menu to add your first block</p>
                            </div>
                        `;
                    }
                }, 300);
            }
            
            closeBlockEditor();
        } catch (error) {
            alert('Error: ' + error.message);
        } finally {
            hideLoading();
        }
    }
    
    // ==================== REPEATER ====================
    function addRepeaterItem(button) {
        const container = button.closest('.repeater-container');
        const template = container.querySelector('.repeater-template');
        const items = container.querySelectorAll('.repeater-item');
        const newIndex = items.length;
        
        const clone = template.content.cloneNode(true);
        const itemEl = clone.querySelector('.repeater-item');
        
        // Replace __INDEX__ and __NUM__ placeholders
        itemEl.innerHTML = itemEl.innerHTML
            .replace(/__INDEX__/g, newIndex)
            .replace(/__NUM__/g, newIndex + 1)
            .replace(/IDX/g, newIndex);
        
        itemEl.dataset.index = newIndex;
        
        container.insertBefore(itemEl, button);
        
        // Scroll to new item
        itemEl.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
    }
    
    function removeRepeaterItem(button) {
        if (confirm('Remove this item?')) {
            const item = button.closest('.repeater-item');
            item.style.transition = 'opacity 0.2s';
            item.style.opacity = '0';
            setTimeout(() => item.remove(), 200);
        }
    }
    
    // ==================== UI HELPERS ====================
    function showLoading() {
        document.getElementById('loading').classList.add('active');
    }
    
    function hideLoading() {
        document.getElementById('loading').classList.remove('active');
    }
    
    // ==================== KEYBOARD SHORTCUTS ====================
    document.addEventListener('keydown', (e) => {
        // ESC to close sidebar/editor
        if (e.key === 'Escape') {
            closeSidebar();
            closeBlockEditor();
        }
    });
    </script>
</body>
</html>